#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/common/env.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/wikimap/mapspro/services/mrc/libs/toloka_common/include/json.h>

#include <boost/range/adaptor/filtered.hpp>

namespace maps::mrc::toloka::common::test {
namespace {

const std::string INPUT_DATA = R"(
[ {
  "assignmentId" : "assignment_1",
  "inputValues" : {
    "source" : "photo_1"
  },
  "outputValues" : {
    "polygons" : "[[[10,10],[20,20]],[[30,30],[50,50]]]"
  }
}, {
  "assignmentId" : "assignment_1",
  "inputValues" : {
    "source" : "photo_2"
  },
  "outputValues" : {
    "polygons" : "[[[100,200],[150,260]],[[300,300],[400,400]]]"
  }
}, {
  "assignmentId" : "assignment_2",
  "inputValues" : {
    "source" : "photo_1"
  },
  "outputValues" : {
    "polygons" : "[[[10.5,11.5],[20.5,21]],[[29,29],[52,52]]]"
  }
}, {
  "assignmentId" : "assignment_2",
  "inputValues" : {
    "source" : "photo_2"
  },
  "outputValues" : {
    "polygons" : "[[[105,205],[145,255]],[[350,350],[450,450]]]"
  }
}, {
  "assignmentId" : "assignment_3",
  "inputValues" : {
    "source" : "photo_1"
  },
  "outputValues" : null
}, {
  "assignmentId" : "assignment_3",
  "inputValues" : {
    "source" : "photo_2"
  },
  "outputValues" : {
    "polygons" : "[[[95,195],[145,261]]]"
  }
} ]
)";

const std::string AGGREGATED_PHOTO_2
    = R"([{"inputValues":{"image":"photo_2","bboxes":[[[100,200],[145,260]]]}}])";

const std::string ASSIGNMENT_3_STATUS
    = R"([{"assignmentId":"assignment_3","status":{"value":"REJECT_SUBMITTED","comment":"Вы разметили не всё либо нарушили правила разметки, описанные в инструкции"}}])";

const std::string ASSIGNMENT_1 = "assignment_1";
const std::string ASSIGNMENT_2 = "assignment_2";
const std::string ASSIGNMENT_3 = "assignment_3";
const std::string PHOTO_1 = "photo_1";
const std::string PHOTO_2 = "photo_2";

bool isEqual(const Rect& lhs, const Rect& rhs)
{
    constexpr double EPS = 1e-6;
    return std::fabs(lhs.minX() - rhs.minX()) < EPS
        && std::fabs(lhs.minY() - rhs.minY()) < EPS
        && std::fabs(lhs.maxX() - rhs.maxX()) < EPS
        && std::fabs(lhs.maxY() - rhs.maxY()) < EPS;
}

template <class RectRange>
bool isPermutation(const RectRange& lhs, const std::vector<Rect>& rhs)
{
    return std::is_permutation(std::begin(lhs), std::end(lhs),
        std::begin(rhs), std::end(rhs), isEqual);
}

bool isEqual(const PrecisionAndRecall& lhs, const PrecisionAndRecall& rhs)
{
    return std::tie(lhs.selected, lhs.relevant, lhs.truePositive)
        == std::tie(rhs.selected, rhs.relevant, rhs.truePositive);
}

auto photo2 = [](const auto& aggregatedPhoto) {
    const auto & [ photo, rects ] = aggregatedPhoto;
    return photo == PHOTO_2;
};

auto assignment3 = [](const auto& aggregatedAssignment) {
    const auto & [ assignment, precisionAndRecall ] = aggregatedAssignment;
    return assignment == ASSIGNMENT_3;
};

} // anonymous namespace

    TEST(suit, test)
    {
        std::istringstream inputStream(INPUT_DATA);
        auto answers = parseTolokaAnswers(inputStream);
        EXPECT_EQ(answers.size(), 6u);

        auto aggregatedPhotos = aggregatePhotos(answers);
        EXPECT_TRUE(isPermutation(aggregatedPhotos.at(PHOTO_1),
            {rect(10.5, 11.5, 20.5, 21), rect(30, 30, 52, 52)}));
        EXPECT_TRUE(isPermutation(
            aggregatedPhotos.at(PHOTO_2), {rect(100, 200, 145, 260)}));

        auto aggregatedAssignments
            = aggregateAssignments(answers, aggregatedPhotos);
        EXPECT_TRUE(isEqual(aggregatedAssignments.at(ASSIGNMENT_1),
            PrecisionAndRecall{/*selected*/ 3u,
                /*relevant*/ 2u,
                /*truePositive*/ 2u}));
        EXPECT_TRUE(isEqual(aggregatedAssignments.at(ASSIGNMENT_2),
            PrecisionAndRecall{/*selected*/ 3u,
                /*relevant*/ 2u,
                /*truePositive*/ 2u}));
        EXPECT_TRUE(isEqual(aggregatedAssignments.at(ASSIGNMENT_3),
            PrecisionAndRecall{/*selected*/ 1u,
                /*relevant*/ 2u,
                /*truePositive*/ 1u}));

        auto aggregatePhoto2
            = aggregatedPhotos | boost::adaptors::filtered(photo2);
        std::ostringstream aggregateData;
        save(PhotoToRectsMap{std::begin(aggregatePhoto2),
                 std::end(aggregatePhoto2)},
            aggregateData);
        EXPECT_EQ(aggregateData.str(), AGGREGATED_PHOTO_2);

        auto aggregatedAssignment3
            = aggregatedAssignments | boost::adaptors::filtered(assignment3);
        std::ostringstream statusData;
        save(AssignmentToPrecisionAndRecallMap{std::begin(
                                                   aggregatedAssignment3),
                 std::end(aggregatedAssignment3)},
            statusData);
        EXPECT_EQ(statusData.str(), ASSIGNMENT_3_STATUS);
    }

} // maps::mrc::toloka::common::test
