#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/detection.h>

#include <maps/libs/enum_io/include/enum_io.h>

#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/std.h>
#include <maps/libs/json/include/value.h>

namespace maps::mrc::toloka {

namespace {

void serializeBBoxes(
    json::ArrayBuilder b,
    const std::vector<geolib3::BoundingBox>& bboxes)
{
    for (size_t i = 0; i < bboxes.size(); i++) {
        b << [&](json::ObjectBuilder b) {
            b[FIELD_POLYGON_TYPE] = POLYGON_TYPE_RECTANGLE;
            b[FIELD_DATA] = [&](json::ObjectBuilder b) {
                b[FIELD_P1] = [&](json::ObjectBuilder b) {
                    b[FIELD_X] = bboxes[i].minX();
                    b[FIELD_Y] = bboxes[i].minY();
                };

                b[FIELD_P2] = [&](json::ObjectBuilder b) {
                    b[FIELD_X] = bboxes[i].maxX();
                    b[FIELD_Y] = bboxes[i].maxY();
                };
            };
        };
    }
}

std::vector<geolib3::BoundingBox> deserializeBBoxes(const json::Value& jsonBboxes)
{
    std::vector<geolib3::BoundingBox> bboxes;

    REQUIRE(jsonBboxes.isArray(), "Invalid detection output format");
    for (const auto& jsonBbox : jsonBboxes) {
        REQUIRE(jsonBbox[FIELD_POLYGON_TYPE].as<std::string>() == POLYGON_TYPE_RECTANGLE,
                "Output polygon is not rectangle");
        const auto& data = jsonBbox[FIELD_DATA];
        bboxes.emplace_back(
            geolib3::Point2(
                data[FIELD_P1][FIELD_X].as<double>(),
                data[FIELD_P1][FIELD_Y].as<double>()
            ),
            geolib3::Point2(
                data[FIELD_P2][FIELD_X].as<double>(),
                data[FIELD_P2][FIELD_Y].as<double>()
            )
        );
    }

    return bboxes;
}

} // namespace

static constexpr enum_io::Representations<DetectionResult> DETECTION_RESULT_REPRESENTATION{
    {DetectionResult::IsNotEmpty, "is_not_empty"},
    {DetectionResult::IsEmpty, "is_empty"},
    {DetectionResult::IsNotLoaded, "is_not_loaded"}
};
DEFINE_ENUM_IO(DetectionResult, DETECTION_RESULT_REPRESENTATION);


void DetectionInput::json(json::ObjectBuilder builder) const
{
    builder[FIELD_IMAGE] = imageUrl();
    if (bboxes_.has_value()) {
        builder[FIELD_GOLDEN_POLYGONS] = [&](json::ArrayBuilder b) {
            serializeBBoxes(b, bboxes_.value());
        };
    }
}

template <>
DetectionInput parseJson(const json::Value& jsonInput)
{
    auto imageUrl = jsonInput[FIELD_IMAGE].as<std::string>();

    std::optional<std::vector<geolib3::BoundingBox>> bboxes;
    if (jsonInput.hasField(FIELD_GOLDEN_POLYGONS)) {
        bboxes = deserializeBBoxes(jsonInput[FIELD_GOLDEN_POLYGONS]);
    }
    return DetectionInput{imageUrl, bboxes};
}

void DetectionOutput::json(json::ObjectBuilder builder) const
{
    // isCorrect определен только для knownSolutions у контрольных
    // заданий и у ответов толокеров на контрольные задания
    if (isCorrect_.has_value()) {
        builder[FIELD_IS_CORRECT] = isCorrect_.value();
    }

    // если result_ = DetectionResult::IsNotEmpty и
    // bboxes_ - пустой вектор, то возникает "неопределенность".
    // Эта неопределенность используется для задания значений в
    // knownSolutions у контрольных заданий. Таким образом
    // в json для контрольных заданий поля 'result' и 'polygons' не задаются
    if (result_ != DetectionResult::IsNotEmpty || !bboxes_.empty()) {
        builder[FIELD_POLYGONS] = [&](json::ArrayBuilder b) {
            serializeBBoxes(b, bboxes_);
        };
        builder[FIELD_RESULT] = toString(result_);
    }
}

template <>
DetectionOutput parseJson(const json::Value& jsonOutput)
{
    std::optional<bool> isCorrect;
    // is_correct опеределен только у knownSolutions у контрольных заданий
    // и у ответов толокеров на контрольные задания
    if (jsonOutput.hasField(FIELD_IS_CORRECT)) {
        isCorrect = jsonOutput[FIELD_IS_CORRECT].as<bool>();
    }

    DetectionResult result = DetectionResult::IsNotEmpty;
    std::vector<geolib3::BoundingBox> bboxes = {};
    // поля 'result' и 'polygons' должны быть определены или неопределены одновременно.
    // оба поля могут быть неопределены только у knownSolutions контрольных заданий.
    if (jsonOutput.hasField(FIELD_RESULT) || jsonOutput.hasField(FIELD_POLYGONS)) {
        fromString(jsonOutput[FIELD_RESULT].as<std::string>(), result);
        bboxes = deserializeBBoxes(jsonOutput[FIELD_POLYGONS]);
    }

    return DetectionOutput(result, bboxes, isCorrect);
}

/**
 * DetectionInput implementation
 */

DetectionInput::DetectionInput(
    std::string imageUrl,
    std::optional<std::vector<geolib3::BoundingBox>> bboxes)
    : imageUrl_(std::move(imageUrl))
    , bboxes_(std::move(bboxes))
{
}

const std::string& DetectionInput::imageUrl() const { return imageUrl_; }

DetectionInput& DetectionInput::setImageUrl(std::string url)
{
    imageUrl_ = url;
    return *this;
}

const std::optional<std::vector<geolib3::BoundingBox>>& DetectionInput::bboxes() const
{
    return bboxes_;
}

/**
 * DetectionOutput implementation
 */

DetectionOutput::DetectionOutput(
    DetectionResult result,
    std::vector<geolib3::BoundingBox> bboxes,
    std::optional<bool> isCorrect)
    : result_(result)
    , bboxes_(std::move(bboxes))
    , isCorrect_(isCorrect)
{
}

DetectionResult DetectionOutput::result() const
{
    return result_;
}

const std::vector<geolib3::BoundingBox>& DetectionOutput::bboxes() const
{
    return bboxes_;
}

std::optional<bool> DetectionOutput::isCorrect() const
{
    return isCorrect_;
}

std::ostream& operator<<(std::ostream& out, const DetectionOutput& taskOutput)
{
    json::Builder builder(out);
    builder << [&](json::ObjectBuilder b) {
        taskOutput.json(b);
    };
    return out;
}

} // namespace maps::mrc::toloka
