#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/task.h>

#include <maps/libs/enum_io/include/enum_io_fwd.h>

#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>

#include <string>
#include <vector>

namespace maps::mrc::toloka {

/**
 * Входные значения для задачи разметки объектов прямоугольниками
 *
 * @param image - URL изображения, на котором будут размечаться объекты
 * @param golden_polygons - набор прямоугольников на изображении. Если этот набор задан,
 *                          то задание может быть использовано в качество контрольного,
 *                          если в knownSolutions задано значение "is_correct" = true.
 *                          Координаты прямоугольников задаются в интервале [0.0, 1.0],
 *                          где точка [0.0, 0.0] - верхний левый угол изображения,
 *                          [1.0, 1.0] - нижний правый угол изображения.
 */
class DetectionInput {
public:
    DetectionInput(
        std::string imageUrl,
        std::optional<std::vector<geolib3::BoundingBox>> bboxes = std::nullopt);

    // Ссылка на изображение для разметки
    const std::string& imageUrl() const;

    DetectionInput& setImageUrl(std::string url);

    // Набор прямоугольников, которые используются для контрольных заданий.
    // Он задается в входных значениях, так как Толока не может корректно
    // сравнивать прямоугольники в контрольных заданиях, из-за чего правильный
    // ответ на задание задается во входных значениях, а проверка ответа выполняется
    // на стороне Толоки в javascript'е.
    const std::optional<std::vector<geolib3::BoundingBox>>& bboxes() const;

    void json(json::ObjectBuilder) const;

private:
    std::string imageUrl_;
    std::optional<std::vector<geolib3::BoundingBox>> bboxes_;
};

using DetectionInputs = std::vector<DetectionInput>;

enum class DetectionResult {
    IsNotEmpty,
    IsEmpty,
    IsNotLoaded
};

DECLARE_ENUM_IO(DetectionResult)

/**
 * Выходные значения для задачи разметки объектов прямоугольниками
 *
 * @param result - одно из трех возможных результатов выполнения задания:
 *                   - is_not_empty - на изображении есть объекты
 *                   - is_empty - на изображении нет объетов
 *                   - is_not_loaded - картинка не смогла корректно загрузиться
 * @param polygons - набор прямоугольников на изображении.
 *                   Координаты прямоугольников задаются в интервале [0.0, 1.0],
 *                   где точка [0.0, 0.0] - верхний левый угол изображения,
 *                   [1.0, 1.0] - нижний правый угол изображения.
 * @param is_correct - необязательное поле, которое используется только в контрольных
 *                     заданиях. в контрольных заданиях это поле всегда имеет значение
 *                     true, а в ответах толокеров true/false. Это значение задается через
 *                     javascript на стороне Толоки при сравнении прямоугольников из
 *                     входных прямоугольников 'golden_polygons' и выходных 'polygons'.
 *                     При этом в контрольных заданиях значения result и polygons не задаются,
 *                     так как в этом случае сравнение контрольных заданий будет некорректным.
 */
class DetectionOutput {
public:
    DetectionOutput(
        DetectionResult result,
        std::vector<geolib3::BoundingBox> bboxes,
        std::optional<bool> isCorrect = std::nullopt);

    DetectionResult result() const;

    const std::vector<geolib3::BoundingBox>& bboxes() const;

    // Определено только для knownSolutions у контрольных заданий (всегда true)
    // или у ответов толокеров на контрольные заданий:
    //   - true - толокер правильно ответил на контрольное задание
    //   - false - толокерв неверно ответил на контрольное задание
    std::optional<bool> isCorrect() const;

    void json(json::ObjectBuilder) const;

private:
    DetectionResult result_;
    std::vector<geolib3::BoundingBox> bboxes_;
    std::optional<bool> isCorrect_;
};

using DetectionOutputs = std::vector<DetectionOutput>;

std::ostream& operator<<(std::ostream& out, const DetectionOutput& taskOutput);

} // namespace maps::mrc::toloka
