#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/task.h>

#include <maps/libs/enum_io/include/enum_io.h>

#include <string>
#include <vector>
#include <optional>

namespace maps::mrc::toloka {

/**
 * Input values for 'House number recognition' type of task
 *
 * @param imageUrl URL of the image to recognize house number on.
 * @param bbox pixel bounding box of the house number on the image
 *             with the coordinates relative to the upper left corner.
 */

class HouseNumberRecognitionInput {
public:
    HouseNumberRecognitionInput(std::string imageUrl, common::ImageBox bbox);

    const std::string& imageUrl() const;
    const common::ImageBox& bbox() const;

    void json(json::ObjectBuilder) const;

private:
    std::string imageUrl_;
    common::ImageBox bbox_;
};

using HouseNumberRecognitionInputs = std::vector<HouseNumberRecognitionInput>;

enum class HouseNumberRecognitionState {
    Ok,
    NotRecognized,
    NotHouseNumber,
    NotLoaded
};

DECLARE_ENUM_IO(HouseNumberRecognitionState);

/**
 * Output values for 'House number recognition' type of task
 *
 * @param state  recognition state
 * @param number recognized house number
 */
class HouseNumberRecognitionOutput {
public:
    HouseNumberRecognitionOutput(
        HouseNumberRecognitionState state,
        std::string houseNumber = "");

    HouseNumberRecognitionState state() const;
    const std::string& houseNumber() const;

    void json(json::ObjectBuilder) const;

    auto introspect() const {
        return std::tie(state_, houseNumber_);
    }

private:
    HouseNumberRecognitionState state_;
    std::string houseNumber_;
};

bool operator==(const HouseNumberRecognitionOutput& lhs, const HouseNumberRecognitionOutput& rhs);

bool operator<(const HouseNumberRecognitionOutput& lhs, const HouseNumberRecognitionOutput& rhs);

std::ostream& operator<<(std::ostream& out, const HouseNumberRecognitionOutput& taskOutput);

/**
 * Task traits for recognition of house number on image
 */
struct HouseNumberRecognitionTaskTraits {
    using InputType = HouseNumberRecognitionInput;
    using OutputType = HouseNumberRecognitionOutput;
    static constexpr auto DB_TASK_TYPE = DbTaskType::HouseNumberRecognition;
};

using HouseNumberRecognitionTask = Task<HouseNumberRecognitionTaskTraits>;
using HouseNumberRecognitionTasks = std::vector<HouseNumberRecognitionTask>;

} // namespace maps::mrc::toloka
