#pragma once

#include <boost/optional.hpp>

#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/conversion.h>
#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/known_solution.h>

#include <memory>
#include <vector>

namespace maps {
namespace mrc {
namespace toloka {

/**
 * This is a template class representing Toloka Manager Task.
 * It should be provided with structure containing
 * input and output data types specific for a particular Task.
 */

template <typename TaskTraits>
class Task {
public:
    using InputType = InputType<TaskTraits>;
    using OutputType = OutputType<TaskTraits>;

    static constexpr auto DB_TASK_TYPE = TaskTraits::DB_TASK_TYPE;
    static constexpr int DEFAULT_OVERLAP = 3;

    Task(TaskId id, TaskStatus status, TimePoint createdAt);
    Task(TaskId id,
         TaskStatus status,
         TimePoint createdAt,
         InputType inputValues);
    Task(const db::toloka::Task& dbTask);
    Task(Task&&) noexcept = default;
    Task& operator=(Task&&) noexcept = default;
    ~Task() {}

    Task& setKnownSolutions(KnownSolutions<Task> solutions);
    Task& setMessageOnUnknownSolution(std::string message);

    TaskId id() const;
    TaskStatus status() const;
    TimePoint createdAt() const;
    const std::optional<TimePoint>& postedAt() const;
    const std::optional<TimePoint>& solvedAt() const;

    const InputType& inputValues() const;
    const std::optional<OutputType>& outputValues() const;

    const KnownSolutions<Task>& knownSolutions() const;
    const std::string& messageOnUnknownSolution() const;

private:
    TaskId id_;
    TaskStatus status_;
    TimePoint createdAt_;
    std::optional<TimePoint> postedAt_;
    std::optional<TimePoint> solvedAt_;

    InputType inputValues_;
    std::optional<OutputType> outputValues_;

    KnownSolutions<Task> knownSolutions_;
    std::string messageOnUnknownSolution_;
};

template <typename TaskTraits>
Task<TaskTraits>::Task(TaskId id, TaskStatus status, TimePoint createdAt)
    : id_(id), status_(status), createdAt_(createdAt)
{
}

template <typename TaskTraits>
Task<TaskTraits>::Task(TaskId id,
                       TaskStatus status,
                       TimePoint createdAt,
                       InputType inputValues)
    : id_(id)
    , status_(status)
    , createdAt_(createdAt)
    , inputValues_(std::move(inputValues))
{
}

template <typename TaskTraits>
Task<TaskTraits>::Task(const db::toloka::Task& dbTask)
    : id_(dbTask.id())
    , status_(dbTask.status())
    , createdAt_(dbTask.createdAt())
    , inputValues_(parseJson<InputType>(dbTask.inputValues()))
{
    postedAt_ = dbTask.postedAt();
    solvedAt_ = dbTask.solvedAt();
    if (dbTask.outputValues()) {
        outputValues_ = parseJson<OutputType>(*dbTask.outputValues());
    }
    if (dbTask.knownSolutions()) {
        knownSolutions_ = parseKnownSolutionsJson<Task<TaskTraits>>(
            *dbTask.knownSolutions());
    }
    if (dbTask.messageOnUnknownSolution()) {
        messageOnUnknownSolution_ = *dbTask.messageOnUnknownSolution();
    }
}

template <typename T>
Task<T>& Task<T>::setKnownSolutions(KnownSolutions<Task> solutions)
{
    knownSolutions_ = std::move(solutions);
    return *this;
}

template <typename T>
Task<T>& Task<T>::setMessageOnUnknownSolution(std::string message)
{
    messageOnUnknownSolution_ = std::move(message);
    return *this;
}

template <typename T>
TaskId Task<T>::id() const
{
    return id_;
}

template <typename T>
TaskStatus Task<T>::status() const
{
    return status_;
}

template <typename T>
TimePoint Task<T>::createdAt() const
{
    return createdAt_;
}

template <typename T>
const std::optional<TimePoint>& Task<T>::postedAt() const
{
    return postedAt_;
}

template <typename T>
const std::optional<TimePoint>& Task<T>::solvedAt() const
{
    return solvedAt_;
}

template <typename T>
const InputType<T>& Task<T>::inputValues() const
{
    return inputValues_;
}

template <typename T>
const std::optional<OutputType<T>>& Task<T>::outputValues() const
{
    return outputValues_;
}

template <typename T>
const KnownSolutions<Task<T>>& Task<T>::knownSolutions() const
{
    return knownSolutions_;
}

template <typename T>
const std::string& Task<T>::messageOnUnknownSolution() const
{
    return messageOnUnknownSolution_;
}

using TaskIds = std::vector<TaskId>;
template <typename T>
using Tasks = std::vector<Task<T>>;

} // namespace toloka
} // namespace mrc
} // namespace maps
