#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/task.h>

#include <string>
#include <vector>

namespace maps {
namespace mrc {
namespace toloka {

/**
 * Input values for 'Taxonomy' type of task
 *
 * @param imageUrl URL of the image to classify.
 *                 It does not need to be externally reachable.
 * @param bbox pixel bounding box of the area of interest on the image
 *             with the coordinates relative to the upper left corner.
 */
class TaxonomyInput {
public:
    TaxonomyInput(std::string imageUrl, const common::ImageBox& bbox);

    const std::string& imageUrl() const;
    const common::ImageBox& bbox() const;

    TaxonomyInput& setImageUrl(std::string url);

    void json(json::ObjectBuilder) const;

private:
    std::string imageUrl_;
    common::ImageBox bbox_;
};

using TaxonomyInputs = std::vector<TaxonomyInput>;

/**
 * Output values for 'Taxonomy' type of task
 *
 * Examples of @param result:
 * 1.18.1_Russian_road_marking
 * ..
 * 1.18.6_Russian_road_marking
 * 1.19.1_Russian_road_marking
 * 1.19.2_Russian_road_marking
 * unknown
 */
class TaxonomyOutput {
public:
    TaxonomyOutput(std::string result);

    const std::string& result() const;

    void json(json::ObjectBuilder) const;

    auto introspect() const {
        return result_;
    }

private:
    std::string result_;
};

bool operator==(const TaxonomyOutput& lhs, const TaxonomyOutput& rhs);

bool operator<(const TaxonomyOutput& lhs, const TaxonomyOutput& rhs);

std::ostream& operator<<(std::ostream& out, const TaxonomyOutput& taskOutput);

/**
 * Task traits for classification of a traffic sign on image
 */
struct TaxonomyTaskTraits {
    using InputType = TaxonomyInput;
    using OutputType = TaxonomyOutput;
    static constexpr auto DB_TASK_TYPE = DbTaskType::Taxonomy;
};

using TaxonomyTask = Task<TaxonomyTaskTraits>;
using TaxonomyTasks = std::vector<TaxonomyTask>;

} // namespace toloka
} // namespace mrc
} // namespace maps
