#pragma once

#include "toloka/platform.h"
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/image_box.h>
#include <maps/wikimap/mapspro/services/mrc/libs/toloka_manager/include/toloka_manager.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>

namespace maps::mrc::toloka::tests {

const std::string DUMMY_IMAGE_URL = "https://yandex.ru/image.jpg";
const common::ImageBox BOX(10, 15, 20, 25);

struct Fixture : testing::Test,
                 unittest::WithUnittestConfig<unittest::DatabaseFixture> {
};

template <typename Task>
void testCreateGetFreeTask(
    pgpool3::Pool& pool,
    const typename Task::InputType& input)
{
    auto txn = pool.masterWriteableTransaction();
    const auto platform = db::toloka::Platform::Toloka;

    auto task = createTask<Task>(*txn, platform, input);
    mrc::db::TId id = task.id();

    auto ids = getTaskIds(*txn, platform, Task::DB_TASK_TYPE);
    EXPECT_EQ(ids.size(), 1u);
    EXPECT_EQ(ids[0], id);

    auto readTask = getTask<Task>(*txn, id);
    EXPECT_EQ(readTask.id(), id);
    EXPECT_TRUE(readTask.status() == TaskStatus::New);
    EXPECT_TRUE(!readTask.outputValues());
    EXPECT_TRUE(!readTask.postedAt());
    EXPECT_TRUE(!readTask.solvedAt());

    freeTask(*txn, id);
    EXPECT_TRUE(getTask<Task>(*txn, id).status()
                == TaskStatus::Free);

    auto freeIds = getTasks<Task>(*txn, platform, TaskStatus::Free);
    EXPECT_EQ(ids.size(), 1u);
    EXPECT_EQ(ids[0], id);

    EXPECT_TRUE(getTasks<Task>(*txn, platform, TaskStatus::InProgress).empty());

    EXPECT_TRUE(getTaskStatus(*txn, id) == TaskStatus::Free);

    auto statusMap = getTaskStatuses(*txn, {id});
    EXPECT_TRUE(statusMap.at(id) == TaskStatus::Free);
}

template <typename Task>
void testSerializeKnownSolutionsDetectionTask()
{
    KnownSolutions<Task> solutions{
        KnownSolution<Task>{
            OutputType<Task>{
                DetectionResult(),
                std::vector<geolib3::BoundingBox>(),
                true
            },
            0.5
        }
    };
    std::string expectedJson = R"(
    [
      {
        "output_values": {
          "is_correct": true
        },
        "correctness_weight": 0.5
      }
    ])";
    EXPECT_EQ(minifyJson(toJson(solutions)), minifyJson(expectedJson));
}

template <typename Task>
void testCreateKnownSolutionsTask(
    pgpool3::Pool& pool,
    const typename Task::InputType& input,
    const KnownSolutions<Task>& knownSolutions)
{
    auto txn = pool.masterWriteableTransaction();

    std::string messageOnUnknownSolution = "Hint message";

    auto task = createTask<Task>(
        *txn, db::toloka::Platform::Toloka,
        input, knownSolutions, messageOnUnknownSolution);
    auto id = task.id();

    auto readTask = getTask<Task>(*txn, id);
    EXPECT_EQ(readTask.id(), id);
    EXPECT_EQ(toJson(readTask.knownSolutions()), toJson(knownSolutions));
    EXPECT_EQ(readTask.messageOnUnknownSolution(), messageOnUnknownSolution);

    freeTask(*txn, id);
    EXPECT_TRUE(getTask<Task>(*txn, id).status() == TaskStatus::Free);
}

template <typename Task>
void testCreateKnownSolutionsDetectionTask(pgpool3::Pool& pool) {
    DetectionInput input(
        DUMMY_IMAGE_URL,
        std::vector<geolib3::BoundingBox>({
            geolib3::BoundingBox(
                geolib3::Point2(10, 15),
                geolib3::Point2(20, 25)
            )
        })
    );
    KnownSolutions<Task> knownSolutions{
        KnownSolution<Task>{
            OutputType<Task>{
                DetectionResult(),
                std::vector<geolib3::BoundingBox>(),
                true
            },
            0.5
        }
    };

    testCreateKnownSolutionsTask(pool, input, knownSolutions);
}

} // namespace maps::mrc::toloka::tests
