#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/common/env.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/wikimap/mapspro/services/mrc/libs/graph_matcher_adapter/include/compact_graph_matcher_adapter.h>
#include <maps/wikimap/mapspro/services/mrc/libs/track_classifier/include/track_classifier.h>

namespace maps::mrc::track_classifier {

namespace {

struct TestData {
    geolib3::Point2 mercatorPos;
    chrono::TimePoint timestamp;
};

const std::string TEST_GRAPH_PATH = BinaryPath("maps/data/test/graph3");

std::vector<TestData> SHORT_TRACK = {
    {{4.18425e+06, 7.47035e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:05.425000000+00:00")},
    {{4.18428e+06, 7.47032e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:43.043000000+00:00")},
    {{4.18427e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:44.236000000+00:00")},
    {{4.18428e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:46.183000000+00:00")},
    {{4.18428e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:48.142000000+00:00")},
};

std::vector<TestData> PEDESTRIAN = {
    {{4.18425e+06, 7.47035e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:05.425000000+00:00")},
    {{4.18428e+06, 7.47032e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:43.043000000+00:00")},
    {{4.18427e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:44.236000000+00:00")},
    {{4.18428e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:46.183000000+00:00")},
    {{4.18428e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:48.142000000+00:00")},
    {{4.18429e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:50.541000000+00:00")},
    {{4.18430e+06, 7.47028e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:52.000000000+00:00")},
    {{4.18431e+06, 7.47028e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:53.449000000+00:00")},
    {{4.18432e+06, 7.47028e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:55.230000000+00:00")},
    {{4.18433e+06, 7.47028e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:56.903000000+00:00")},
    {{4.18433e+06, 7.47027e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:58.388000000+00:00")},
    {{4.18434e+06, 7.47027e+06}, chrono::parseSqlDateTime("2020-02-27 08:00:59.828000000+00:00")},
    {{4.18435e+06, 7.47027e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:01.801000000+00:00")},
    {{4.18436e+06, 7.47027e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:03.748000000+00:00")},
    {{4.18436e+06, 7.47028e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:06.189000000+00:00")},
    {{4.18437e+06, 7.47028e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:07.641000000+00:00")},
    {{4.18438e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:09.606000000+00:00")},
    {{4.18438e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:11.643000000+00:00")},
    {{4.18439e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:13.223000000+00:00")},
    {{4.18439e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:14.739000000+00:00")},
    {{4.18440e+06, 7.47030e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:17.086000000+00:00")},
    {{4.18440e+06, 7.47030e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:19.184000000+00:00")},
    {{4.18440e+06, 7.47030e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:20.982000000+00:00")},
    {{4.18441e+06, 7.47029e+06}, chrono::parseSqlDateTime("2020-02-27 08:01:22.822000000+00:00")}
};

std::vector<TestData> VEHICLE = {
    {{1.03276e+07, 7.52635e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:17.031000000+00:00")},
    {{1.03276e+07, 7.52633e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:19.027000000+00:00")},
    {{1.03276e+07, 7.52631e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:20.034000000+00:00")},
    {{1.03276e+07, 7.52628e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:22.021000000+00:00")},
    {{1.03276e+07, 7.52625e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:24.026000000+00:00")},
    {{1.03276e+07, 7.52623e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:25.038000000+00:00")},
    {{1.03276e+07, 7.52620e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:27.030000000+00:00")},
    {{1.03275e+07, 7.52617e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:29.031000000+00:00")},
    {{1.03275e+07, 7.52615e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:30.035000000+00:00")},
    {{1.03275e+07, 7.52612e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:32.042000000+00:00")},
    {{1.03275e+07, 7.52608e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:34.027000000+00:00")},
    {{1.03275e+07, 7.52607e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:35.030000000+00:00")},
    {{1.03275e+07, 7.52606e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:36.044000000+00:00")},
    {{1.03275e+07, 7.52604e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:38.026000000+00:00")},
    {{1.03275e+07, 7.52603e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:40.049000000+00:00")},
    {{1.03275e+07, 7.52603e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:42.035000000+00:00")},
    {{1.03275e+07, 7.52603e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:44.045000000+00:00")},
    {{1.03275e+07, 7.52602e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:49.057000000+00:00")},
    {{1.03275e+07, 7.52602e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:51.055000000+00:00")},
    {{1.03275e+07, 7.52602e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:52.062000000+00:00")},
    {{1.03275e+07, 7.52601e+06}, chrono::parseSqlDateTime("2019-10-24 05:16:54.065000000+00:00")},
    {{1.03275e+07, 7.52602e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:00.057000000+00:00")},
    {{1.03275e+07, 7.52602e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:02.724000000+00:00")},
    {{1.03275e+07, 7.52602e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:03.847000000+00:00")},
    {{1.03275e+07, 7.52604e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:05.055000000+00:00")},
    {{1.03275e+07, 7.52606e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:07.062000000+00:00")},
    {{1.03275e+07, 7.52607e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:08.062000000+00:00")},
    {{1.03275e+07, 7.52609e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:09.076000000+00:00")},
    {{1.03275e+07, 7.52612e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:11.062000000+00:00")},
    {{1.03275e+07, 7.52614e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:12.062000000+00:00")},
    {{1.03275e+07, 7.52616e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:13.069000000+00:00")},
    {{1.03276e+07, 7.52619e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:15.055000000+00:00")},
    {{1.03276e+07, 7.52621e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:16.060000000+00:00")},
    {{1.03276e+07, 7.52623e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:17.080000000+00:00")},
    {{1.03276e+07, 7.52627e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:19.076000000+00:00")},
    {{1.03276e+07, 7.52632e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:21.052000000+00:00")},
    {{1.03276e+07, 7.52633e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:22.058000000+00:00")},
    {{1.03276e+07, 7.52635e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:23.069000000+00:00")},
    {{1.03276e+07, 7.52636e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:24.069000000+00:00")},
    {{1.03276e+07, 7.52639e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:26.052000000+00:00")},
    {{1.03276e+07, 7.52640e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:27.087000000+00:00")},
    {{1.03276e+07, 7.52640e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:29.066000000+00:00")},
    {{1.03276e+07, 7.52640e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:30.073000000+00:00")},
    {{1.03276e+07, 7.52640e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:31.085000000+00:00")},
    {{1.03277e+07, 7.52639e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:33.063000000+00:00")},
    {{1.03277e+07, 7.52639e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:34.063000000+00:00")},
    {{1.03277e+07, 7.52638e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:35.073000000+00:00")},
    {{1.03278e+07, 7.52636e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:37.081000000+00:00")},
    {{1.03278e+07, 7.52636e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:38.093000000+00:00")},
    {{1.03278e+07, 7.52635e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:40.078000000+00:00")},
    {{1.03278e+07, 7.52634e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:41.090000000+00:00")},
    {{1.03278e+07, 7.52633e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:42.098000000+00:00")},
    {{1.03278e+07, 7.52631e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:44.087000000+00:00")},
    {{1.03278e+07, 7.52628e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:46.081000000+00:00")},
    {{1.03277e+07, 7.52623e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:48.081000000+00:00")},
    {{1.03277e+07, 7.52619e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:50.073000000+00:00")},
    {{1.03277e+07, 7.52617e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:51.074000000+00:00")},
    {{1.03277e+07, 7.52615e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:52.090000000+00:00")},
    {{1.03277e+07, 7.52612e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:54.088000000+00:00")},
    {{1.03277e+07, 7.52607e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:56.100000000+00:00")},
    {{1.03277e+07, 7.52603e+06}, chrono::parseSqlDateTime("2019-10-24 05:17:58.099000000+00:00")},
    {{1.03276e+07, 7.52599e+06}, chrono::parseSqlDateTime("2019-10-24 05:18:00.089000000+00:00")},
    {{1.03276e+07, 7.52595e+06}, chrono::parseSqlDateTime("2019-10-24 05:18:02.090000000+00:00")},
    {{1.03276e+07, 7.52593e+06}, chrono::parseSqlDateTime("2019-10-24 05:18:04.078000000+00:00")},
    {{1.03276e+07, 7.52591e+06}, chrono::parseSqlDateTime("2019-10-24 05:18:06.102000000+00:00")},
    {{1.03276e+07, 7.52590e+06}, chrono::parseSqlDateTime("2019-10-24 05:18:07.110000000+00:00")}
};

db::TrackPoints fromData(const std::vector<TestData>& dataVector) {
    db::TrackPoints result(dataVector.size());
    for (size_t i = 0; i < dataVector.size(); i++) {
        const TestData& data = dataVector[i];
        db::TrackPoint& tpt = result[i];
        tpt.setMercatorPos(data.mercatorPos);
        tpt.setTimestamp(data.timestamp);
    }
    return result;
}

} // namespace

namespace tests {

TEST(test_track_classifier, test_undefined_tracks)
{
    const adapters::CompactGraphMatcherAdapter matcher(TEST_GRAPH_PATH);
    std::vector<TrackInterval> tracks = classify(fromData(SHORT_TRACK), {{db::GraphType::Road, &matcher}, {db::GraphType::Pedestrian, &matcher}});
    EXPECT_EQ(tracks.size(), 1u);
    EXPECT_EQ(tracks[0].type, TrackType::Undefined);
} // test_undefined_track

TEST(test_track_classifier, test_pedestrian_track)
{
    const adapters::CompactGraphMatcherAdapter matcher(TEST_GRAPH_PATH);
    std::vector<TrackInterval> tracks = classify(fromData(PEDESTRIAN), {{db::GraphType::Road, &matcher}, {db::GraphType::Pedestrian, &matcher}});
    EXPECT_EQ(tracks.size(), 1u);
    EXPECT_EQ(tracks[0].type, TrackType::Pedestrian);
} // test_pedestrian_track

TEST(test_track_classifier, test_vehicle_track)
{
    const adapters::CompactGraphMatcherAdapter matcher(TEST_GRAPH_PATH);
    std::vector<TrackInterval> tracks = classify(fromData(VEHICLE), {{db::GraphType::Road, &matcher}, {db::GraphType::Pedestrian, &matcher}});
    EXPECT_EQ(tracks.size(), 1u);
    EXPECT_EQ(tracks[0].type,  TrackType::Vehicle);
} // test_vehicle_track

} // namespace tests
} // maps::mrc::track_classifier
