#pragma once

#include <string>

namespace maps {
namespace mrc {
namespace traffic_signs {

enum class TrafficSignCategory {
    Unknown,

    // PriorityRoad
    YellowRhomb,

    // Prohibitory signs
    RedCircle,

    // Mandatory signs
    BlueCircle,

    // Warning signs
    RedTriangleWithWhiteBackground,

    // Give way
    RedFlippedTriagnleWithWhiteBackgound,

    BlackCircleWithWhiteBackground,

    // Special regulations signs
    BlueRectangle,

    Plate,
};

enum class TrafficSign {
    Unknown,

    // 2.1 https://upload.wikimedia.org/wikipedia/commons/thumb/c/c4/2.1_Russian_road_sign.svg/50px-2.1_Russian_road_sign.svg.png
    PriorityPriorityRoad,

    // 2.2 https://upload.wikimedia.org/wikipedia/commons/thumb/3/3e/2.2_Russian_road_sign.svg/50px-2.2_Russian_road_sign.svg.png
    PriorityEofPriorityRoad,

    // 2.4 https://upload.wikimedia.org/wikipedia/commons/thumb/d/d1/2.4_Russian_road_sign.svg/50px-2.4_Russian_road_sign.svg.png
    PriorityGiveWay,

    // 2.5 https://upload.wikimedia.org/wikipedia/commons/thumb/6/6a/2.5_Russian_road_sign.svg/50px-2.5_Russian_road_sign.svg.png
    PriorityStop,

    // https://upload.wikimedia.org/wikipedia/commons/thumb/6/6e/Israeli_Stop_Sign.png/50px-Israeli_Stop_Sign.png
    PriorityStopGesture,

    // 2.6
    PriorityOncomingVehicles,

    // 2.7
    PriorityOverOncomingVehicles,

    // 3.1 https://upload.wikimedia.org/wikipedia/commons/thumb/3/32/3.1_Russian_road_sign.svg/50px-3.1_Russian_road_sign.svg.png
    ProhibitoryNoEntry,

    // 3.2 https://upload.wikimedia.org/wikipedia/commons/thumb/f/fa/3.2_Russian_road_sign.svg/50px-3.2_Russian_road_sign.svg.png
    ProhibitoryNoVehicles,

    // 3.4 https://upload.wikimedia.org/wikipedia/commons/thumb/e/eb/3.4_Russian_road_sign.svg/50px-3.4_Russian_road_sign.svg.png
    ProhibitoryNoHeavyGoodsVehicles,

    // 3.18.1 https://upload.wikimedia.org/wikipedia/commons/thumb/c/c8/3.18.1_Russian_road_sign.svg/50px-3.18.1_Russian_road_sign.svg.png
    ProhibitoryNoRightTurn,

    // 3.18.2 https://upload.wikimedia.org/wikipedia/commons/thumb/5/53/3.18.2_Russian_road_sign.svg/50px-3.18.2_Russian_road_sign.svg.png
    ProhibitoryNoLeftTurn,

    // 3.19 https://upload.wikimedia.org/wikipedia/commons/thumb/e/e6/3.19_Russian_road_sign.svg/50px-3.19_Russian_road_sign.svg.png
    ProhibitoryNoUturn,

    // 3.11 https://upload.wikimedia.org/wikipedia/commons/thumb/d/d3/3.11_Russian_road_sign.svg/50px-3.11_Russian_road_sign.svg.png
    ProhibitoryMaxWeight,

    // 3.12 https://upload.wikimedia.org/wikipedia/commons/thumb/8/85/3.12_Russian_road_sign.svg/50px-3.12_Russian_road_sign.svg.png
    ProhibitoryMaxWeightPerAxle,

    // 3.13 https://upload.wikimedia.org/wikipedia/commons/thumb/a/a1/3.13_Russian_road_sign.svg/50px-3.13_Russian_road_sign.svg.png
    ProhibitoryMaxHeight,

    // 3.14 https://upload.wikimedia.org/wikipedia/commons/thumb/7/79/3.14_Russian_road_sign.svg/50px-3.14_Russian_road_sign.svg.png
    ProhibitoryMaxWidth,

    // 3.24 https://upload.wikimedia.org/wikipedia/commons/thumb/d/d9/3.24_Russian_road_sign.svg/50px-3.24_Russian_road_sign.svg.png
    ProhibitoryMaxSpeed,
    ProhibitoryMaxSpeed5,
    ProhibitoryMaxSpeed10,
    ProhibitoryMaxSpeed20,
    ProhibitoryMaxSpeed30,
    ProhibitoryMaxSpeed40,
    ProhibitoryMaxSpeed50,
    ProhibitoryMaxSpeed60,
    ProhibitoryMaxSpeed70,
    ProhibitoryMaxSpeed80,
    ProhibitoryMaxSpeed90,
    ProhibitoryMaxSpeed100,
    ProhibitoryMaxSpeed110,
    ProhibitoryMaxSpeed120,
    ProhibitoryMaxSpeed130,
    ProhibitoryMaxSpeed15,
    ProhibitoryMaxSpeed25,
    ProhibitoryMaxSpeed35,
    ProhibitoryMaxSpeed45,
    ProhibitoryMaxSpeed55,

    // 3.25 https://upload.wikimedia.org/wikipedia/commons/thumb/9/91/3.25_Russian_road_sign.svg/50px-3.25_Russian_road_sign.svg.png
    ProhibitoryEofMaxSpeed,
    ProhibitoryEofMaxSpeed5,
    ProhibitoryEofMaxSpeed10,
    ProhibitoryEofMaxSpeed15,
    ProhibitoryEofMaxSpeed20,
    ProhibitoryEofMaxSpeed25,
    ProhibitoryEofMaxSpeed30,
    ProhibitoryEofMaxSpeed35,
    ProhibitoryEofMaxSpeed40,
    ProhibitoryEofMaxSpeed45,
    ProhibitoryEofMaxSpeed50,
    ProhibitoryEofMaxSpeed55,
    ProhibitoryEofMaxSpeed60,
    ProhibitoryEofMaxSpeed70,
    ProhibitoryEofMaxSpeed80,
    ProhibitoryEofMaxSpeed90,
    ProhibitoryEofMaxSpeed100,
    ProhibitoryEofMaxSpeed110,
    ProhibitoryEofMaxSpeed120,
    ProhibitoryEofMaxSpeed130,

    // 3.28 https://upload.wikimedia.org/wikipedia/commons/thumb/8/81/3.28_Russian_road_sign.svg/50px-3.28_Russian_road_sign.svg.png
    ProhibitoryNoParking,

    // 3.27 https://upload.wikimedia.org/wikipedia/commons/thumb/9/98/3.27_Russian_road_sign.svg/50px-3.27_Russian_road_sign.svg.png
    ProhibitoryNoParkingOrStopping,

    // 3.20 https://upload.wikimedia.org/wikipedia/commons/thumb/2/2e/3.20_Russian_road_sign.svg/50px-3.20_Russian_road_sign.svg.png
    ProhibitoryNoOvertaking,

    // 3.21 https://upload.wikimedia.org/wikipedia/commons/thumb/e/e4/3.21_Russian_road_sign.svg/50px-3.21_Russian_road_sign.svg.png
    ProhibitoryEofNoOvertaking,

    // 3.22
    ProhibitoryNoOvertakingByHeavyVehicle,

    // 3.23
    ProhibitoryEofNoOvertakingByHeavyVehicle,

    // 3.3
    ProhibitoryNoMotorVehicles,

    // 3.5
    ProhibitoryNoMotorcycles,

    // 3.6
    ProhibitoryNoTractors,

    // 3.7
    ProhibitoryNoTrailer,

    // 3.8
    ProhibitoryNoHorseCarriages,

    // 3.9
    ProhibitoryNoBicycles,

    // 3.10
    ProhibitoryNoPedestrians,

    // 3.15
    ProhibitoryMaxLength,

    // 3.16
    ProhibitoryMinDistance,

    // 3.17.1
    ProhibitoryCustoms,

    // 3.17.2
    ProhibitoryDanger,

    // 3.17.3
    ProhibitoryStop,

    // 3.26
    ProhibitoryNoHornsOrMotorNoise,

    // 3.29
    ProhibitoryNoParkingOnOddDaysOfTheMonth,

    // 3.30
    ProhibitoryNoParkingOnEvenDaysOfTheMonth,

    // 3.31
    ProhibitoryEofAll,

    // 3.32
    ProhibitoryNoVehiclesCarryingDangerousGoods,

    // 3.33
    ProhibitoryNoVehiclesCarryingExplosives,

    // 4.1.1 https://upload.wikimedia.org/wikipedia/commons/thumb/5/5b/4.1.1_Russian_road_sign.svg/50px-4.1.1_Russian_road_sign.svg.png
    MandatoryProceedStraight,

    // 4.1.5 https://upload.wikimedia.org/wikipedia/commons/thumb/7/73/4.1.5_Russian_road_sign.svg/50px-4.1.5_Russian_road_sign.svg.png
    MandatoryProceedStraightOrTurnLeft,

    // 4.1.4 https://upload.wikimedia.org/wikipedia/commons/thumb/b/be/4.1.4_Russian_road_sign.svg/50px-4.1.4_Russian_road_sign.svg.png
    MandatoryProceedStraightOrTurnRight,

    // 4.3 https://upload.wikimedia.org/wikipedia/commons/thumb/f/f8/4.3_Russian_road_sign.svg/50px-4.3_Russian_road_sign.svg.png
    MandatoryRoundabout,

    // https://upload.wikimedia.org/wikipedia/commons/thumb/c/c0/Znak_C-3.svg/120px-Znak_C-3.svg.png
    MandatoryTurnLeft,

    // 4.1.3 https://upload.wikimedia.org/wikipedia/commons/thumb/4/46/4.1.3_Russian_road_sign.svg/50px-4.1.3_Russian_road_sign.svg.png
    MandatoryTurnLeftAhead,

    // https://upload.wikimedia.org/wikipedia/commons/thumb/9/92/Znak_C-1.svg/120px-Znak_C-1.svg.png
    MandatoryTurnRight,

    // 4.1.2 https://upload.wikimedia.org/wikipedia/commons/thumb/2/23/4.1.2_Russian_road_sign.svg/50px-4.1.2_Russian_road_sign.svg.png
    MandatoryTurnRightAhead,

    // 4.1.6
    MandatoryTurnRightOrLeft,

    // 4.2.1
    MandatoryDrivingDirectionR,

    // 4.2.2
    MandatoryDrivingDirectionL,

    // 4.2.3
    MandatoryDrivingDirectionRL,

    // 4.4.1
    MandatoryCycleRoute,

    // 4.4.2
    MandatoryEofCycleRoute,

    // 4.5.1
    MandatoryPedestrianPath,

    // 4.5.2
    MandatoryCyclistAndPedestriansCombinedPath,

    // 4.5.3
    MandatoryEofCyclistAndPedestriansCombinedPath,

    // 4.5.4
    MandatoryCyclistAndPedestriansSegregatedPath,

    // 4.5.5
    MandatoryPedestrianAndCyclistsSegregatedPath,

    // 4.5.6
    MandatoryEofCyclistAndPedestriansSegregatedPath,

    // 4.5.7
    MandatoryEofPedestrianAndCyclistsSegregatedPath,

    // 4.6
    MandatoryMinSpeed,

    // 4.7
    MandatoryEofMinSpeed,

    // 4.8.1
    MandatoryVehiclesCarryingExplosivesProceedStraight,

    // 4.8.2
    MandatoryVehiclesCarryingExplosivesTurnRightAhead,

    // 4.8.3
    MandatoryVehiclesCarryingExplosivesTurnLeftAhead,

    // 5.5 https://upload.wikimedia.org/wikipedia/commons/thumb/f/fb/5.5_Russian_road_sign.svg/50px-5.5_Russian_road_sign.svg.png
    PrescriptionOneWayRoad,

    // 5.6 https://upload.wikimedia.org/wikipedia/commons/thumb/7/7e/5.6_Russian_road_sign.svg/50px-5.6_Russian_road_sign.svg.png
    PrescriptionEofOneWayRoad,

    // 5.7.1 https://upload.wikimedia.org/wikipedia/commons/thumb/9/98/5.7.1_Russian_road_sign.svg/50px-5.7.1_Russian_road_sign.svg.png
    PrescriptionEntryToOneWayRoadOnTheRight,

    // 5.7.2 https://upload.wikimedia.org/wikipedia/commons/thumb/5/57/5.7.2_Russian_road_sign.svg/50px-5.7.2_Russian_road_sign.svg.png
    PrescriptionEntryToOneWayRoadOnTheLeft,

    // 5.11.1
    PrescriptionRoadWithBusLane,

    // 5.12.1
    PrescriptionEofRoadWithBusLane,

    // 5.14 https://upload.wikimedia.org/wikipedia/commons/thumb/7/76/5.14_Russian_road_sign.svg/50px-5.14_Russian_road_sign.svg.png
    PrescriptionBusLane,

    // 5.14.1 https://upload.wikimedia.org/wikipedia/commons/thumb/d/d1/5.14.1_Russian_road_sign.svg/50px-5.14.1_Russian_road_sign.svg.png
    PrescriptionEofBusLane,

    // 5.15.1 https://upload.wikimedia.org/wikipedia/commons/thumb/6/6c/5.15.1_Russian_road_sign.svg/100px-5.15.1_Russian_road_sign.svg.png
    PrescriptionLanesDirection,

    // 5.15.2_forward_left http://storage.mds.yandex.net/get-maps_mrc_public/224191/toloka/classification/sign-pics/5.15.2_forward_or_left.png
    PrescriptionLaneDirectionFL,

    // 5.15.2_forward_right https://upload.wikimedia.org/wikipedia/commons/thumb/f/f9/5.15.2_B_Russian_road_sign.svg/50px-5.15.2_B_Russian_road_sign.svg.png
    PrescriptionLaneDirectionFR,

    // 5.15.2_forward https://upload.wikimedia.org/wikipedia/commons/thumb/0/04/5.15.2_A_Russian_road_sign.svg/50px-5.15.2_A_Russian_road_sign.svg.png
    PrescriptionLaneDirectionF,

    // 5.15.2_left_45 http://storage.mds.yandex.net/get-maps_mrc_public/200212/toloka/classification/sign-pics/5.15.2_left_45.png
    PrescriptionLaneDirectionFl,

    // 5.15.2_right_45 https://upload.wikimedia.org/wikipedia/commons/thumb/c/cd/5.15.2_D_Russian_road_sign.svg/50px-5.15.2_D_Russian_road_sign.svg.png
    PrescriptionLaneDirectionFr,

    // 5.15.2_left http://storage.mds.yandex.net/get-maps_mrc_public/200212/toloka/classification/sign-pics/5.15.2_left.png
    PrescriptionLaneDirectionL,

    // 5.15.2_right_left http://storage.mds.yandex.net/get-maps_mrc_public/200212/toloka/classification/sign-pics/5.15.2_right_or_left.png
    PrescriptionLaneDirectionRL,

    // 5.15.2_right https://upload.wikimedia.org/wikipedia/commons/thumb/8/80/5.15.2_C_Russian_road_sign.svg/50px-5.15.2_C_Russian_road_sign.svg.png
    PrescriptionLaneDirectionR,

    // 5.15.2_other https://upload.wikimedia.org/wikipedia/commons/thumb/b/b9/5.15.2_E_Russian_road_sign.svg/50px-5.15.2_E_Russian_road_sign.svg.png
    PrescriptionLaneDirectionOther,

    // 5.15.3 https://upload.wikimedia.org/wikipedia/commons/thumb/4/42/5.15.3_A_Russian_road_sign.svg/50px-5.15.3_A_Russian_road_sign.svg.png
    PrescriptionStartNewLineRight,

    // 5.15.4 https://upload.wikimedia.org/wikipedia/commons/thumb/2/2e/5.15.4_A_Russian_road_sign.svg/50px-5.15.4_A_Russian_road_sign.svg.png
    PrescriptionStartNewLineLeft,

    // 5.15.5 https://upload.wikimedia.org/wikipedia/commons/thumb/5/5d/5.15.5_Russian_road_sign.svg/50px-5.15.5_Russian_road_sign.svg.png
    PrescriptionEndLineRight,

    // 5.15.6 https://upload.wikimedia.org/wikipedia/commons/thumb/a/aa/5.15.6_Russian_road_sign.svg/50px-5.15.6_Russian_road_sign.svg.png
    PrescriptionEndLineLeft,

    // 5.15.2_forward_or_right_45 http://storage.mds.yandex.net/get-maps_mrc_public/264598/toloka/classification/sign-pics/5.12.2_forward_or_right_45.png
    PrescriptionLaneDirectionFFr,

    // 5.15.2_forward_or_left_45 http://storage.mds.yandex.net/get-maps_mrc_public/266148/toloka/classification/sign-pics/5.12.2_forward_or_left_45.png
    PrescriptionLaneDirectionFFl,

    // 5.15.2_right_or_right_45 http://storage.mds.yandex.net/get-maps_mrc_public/265615/toloka/classification/sign-pics/5.12.2_right_or_right_45.png
    PrescriptionLaneDirectionFrR,

    // 5.15.2_right_45_or_left_45 http://storage.mds.yandex.net/get-maps_mrc_public/265536/toloka/classification/sign-pics/5.12.2_right_45_or_left_45.png
    PrescriptionLaneDirectionFrFl,

    // 5.15.2.B
    PrescriptionLaneDirectionB,

    // 5.15.2.L_FL
    PrescriptionLaneDirectionLFl,

    // 5.23.1
    PrescriptionBuiltUpArea,

    // 5.23.2
    PrescriptionBuiltUpAreaPictogram,

    // 5.24.1
    PrescriptionEofBuiltUpArea,

    // 5.24.2
    PrescriptionEofBuiltUpAreaPictogram,

    // 5.25
    PrescriptionBuiltUpAreaWoRules,

    // 5.26
    PrescriptionEofBuiltUpAreaWoRules,

    // 5.3
    PrescriptionLimitedAccessRoad,

    // 5.4
    PrescriptionEofLimitedAccessRoad,

    // 5.8
    PrescriptionTwoWayTraffic,

    // 5.9
    PrescriptionEofTwoWayTraffic,

    // 5.10
    PrescriptionEntryToTwoWayTraffic,

    // 5.11.2
    PrescriptionRoadWithCycleLane,

    // 5.12.2
    PrescriptionEofRoadWithCycleLane,

    // 5.13.1
    PrescriptionEntryToRoadWithBusLaneL,

    // 5.13.2
    PrescriptionEntryToRoadWithBusLaneR,

    // 5.13.3
    PrescriptionEntryToRoadWithCycleLaneL,

    // 5.13.4
    PrescriptionEntryToRoadWithCycleLaneR,

    // 5.14.2
    PrescriptionCycleLane,

    // 5.14.3
    PrescriptionEofCycleLane,

    // 5.15.7
    PrescriptionTwoWayLanesDirection,

    // 5.16
    PrescriptionStoppingPlaceForBus,

    // 5.17
    PrescriptionStoppingPlaceForTrams,

    // 5.18
    PrescriptionStoppingPlaceForTaxi,

    // 5.19.1
    PrescriptionPedestrianCrossing,

    // 5.20
    PrescriptionRoadHump,

    // 5.21
    PrescriptionLivingZone,

    // 5.22
    PrescriptionEofLivingZone,

    // 6.4 https://upload.wikimedia.org/wikipedia/commons/thumb/e/eb/6.4_Russian_road_sign.svg/50px-6.4_Russian_road_sign.svg.png
    InformationParking,

    // 6.2
    InformationAdvisorySpeedLimit,

    // 6.3.1
    InformationUTurnPlace,

    // 6.3.2
    InformationUTurnDistance,

    // 6.5
    InformationEmergencyStoppingLane,

    // 6.6
    InformationPedestrianSubway,

    // 6.7
    InformationFootbridge,

    // 6.8.1
    InformationCulDeSacF,

    // 6.8.2
    InformationCulDeSacR,

    // 6.8.3
    InformationCulDeSacL,

    // 6.10.1
    InformationDirectionCursor,

    //6.15.1 https://ru.wikipedia.org/wiki/Файл:6.15.1_Russian_road_sign.svg
    InformationDirectionHeavyVehicleForward,

    //6.15.2 https://ru.wikipedia.org/wiki/Файл:6.15.2_Russian_road_sign.svg
    InformationDirectionHeavyVehicleRight,

    //6.15.3 https://ru.wikipedia.org/wiki/Файл:6.15.3_Russian_road_sign.svg
    InformationDirectionHeavyVehicleLeft,

    // 6.16
    InformationStopLine,

    // 1.7
    WarningRoundaboutAhead,

    // 1.17
    WarningUnevenRoadAhead,

    // 1.20.1
    WarningRoadNarrowsOnBoth,

    // 1.20.2
    WarningRoadNarrowsOnRight,

    // 1.20.3
    WarningRoadNarrowsOnLeft,

    // 1.22
    WarningPedestrianCrossingAhead,

    // 1.23
    WarningChildren,

    //2.3.1
    WarningCrossroadsMinorRoadRL,

    //2.3.2
    WarningCrossroadsMinorRoadR,

    //2.3.3
    WarningCrossroadsMinorRoadL,

    //2.3.4
    WarningCrossroadsMinorRoadFr,

    //2.3.5
    WarningCrossroadsMinorRoadFl,

    //2.3.6
    WarningCrossroadsMinorRoadBr,

    //2.3.7
    WarningCrossroadsMinorRoadBl,

    //1.4.1
    WarningLevelCrossingCountdown3R,

    //1.4.2
    WarningLevelCrossingCountdown2R,

    //1.4.3
    WarningLevelCrossingCountdown1R,

    //1.4.4
    WarningLevelCrossingCountdown3L,

    //1.4.5
    WarningLevelCrossingCountdown2L,

    //1.4.6
    WarningLevelCrossingCountdown1L,

    //1.5
    WarningTramCrossingAhead,

    //1.6
    WarningCrossroadsWithPriorityToTheRight,

    //1.8
    WarningTrafficSignalsAhead,

    //1.9
    WarningOpeningOrSwingBridge,

    //1.10
    WarningUnprotectedQuaysideOrRiverbank,

    //1.11.1
    WarningDangerousBendR,

    //1.11.2
    WarningDangerousBendL,

    //1.12.1
    WarningDangerousBendsR,

    //1.12.2
    WarningDangerousBendsL,

    //1.13
    WarningSteepDescent,

    //1.14
    WarningSteepAscent,

    //1.15
    WarningSlipperyRoadSurface,

    //1.16
    WarningUnevenRoad,

    //1.18
    WarningLooseChippings,

    //1.19
    WarningDangerousRoadside,

    //1.21
    WarningTwoWayTraffic,

    //1.24
    WarningCyclists,

    //1.25
    WarningRoadworksAhead,

    //1.26
    WarningDomesticAnimals,

    //1.27
    WarningWildAnimals,

    //1.28
    WarningFallingRocksOrDebris,

    //1.29
    WarningDangerousCrosswinds,

    //1.30
    WarningLowFlyingAircraft,

    //1.31
    WarningTunnelAhead,

    //1.32
    WarningTrafficJam,

    //1.33
    WarningOtherDangers,

    //1.34.1
    WarningTurningDirectionR,

    //1.34.2
    WarningTurningDirectionL,

    //1.34.3
    WarningTurningDirectionRL,

    // 8.1.2
    InformationDistanceStopAhead,

    // 8.1.3
    InformationDistanceObjectRight,

    // 8.1.4
    InformationDistanceObjectLeft,

    // 8.2.2
    InformationStartZone,

    // 8.2.4
    InformationInZone,

    // 8.4.1
    InformationHeavyVehicle,

    // 8.4.3
    InformationLightVehicle,

    // 8.5.1
    InformationHoliday,

    // 8.5.2
    InformationWorkingDay,

    // 8.5.5
    InformationHolidayTime ,

    // 8.5.6
    InformationWorkingDayTime,

    // 8.8
    InformationPaidServices,

    // 7.1
    InformationFirstAidStation,

    // 7.2
    InformationHospital,

    // 7.3
    InformationGasStation,

    // 7.4
    InformationServiceStation,

    // 7.5
    InformationCarWash,

    // 7.6
    InformationPhone,

    // 7.7
    InformationFood,

    // 7.8
    InformationDrinkingWater,

    // 7.9
    InformationHotel,

    // 7.10
    InformationCamping,

    // 7.11
    InformationRestingPlace,

    // 7.12
    InformationTrafficPolice,

    // 7.13
    InformationPolice,

    // 7.14
    InformationCheckpointInternationalRoadTransport,

    // 7.15
    InformationRadio,

    // 7.16
    InformationRadioCommunicationWithEmergencyServices,

    // 7.17
    InformationPoolOrBeach,

    // 7.18
    InformationToilet,

    // 7.19
    InformationEmergencyTelephone,

    // 7.20
    InformationFireExtinguisher,

    // 8.1.1
    InformationDistanceObject,

    // 8.2.1
    InformationStartZoneHorizontal,

    // 8.2.3
    InformationEofZone,

    // 8.3.1
    InformationDirectionalR,

    // 8.3.2
    InformationDirectionalL,

    // 8.3.3
    InformationDirectionalRL,

    // 8.4.2
    InformationTrailer,

    // 8.4.4
    InformationBus,

    // 8.4.5
    InformationTractor,

    // 8.4.6
    InformationMotorcycle,

    // 8.4.7
    InformationBicycle,

    // 8.4.8
    InformationVehiclesCarryingDangerousGoods,

    // 8.4.9
    InformationExceptHeavyVehicle,

    // 8.4.10
    InformationExceptLightVehicle,

    // 8.4.11
    InformationExceptBus,

    // 8.4.12
    InformationExceptMotorcycle,

    // 8.4.13
    InformationExceptBicycle,

    // 8.4.14
    InformationExceptTaxi,

    // 8.5.4
    InformationTime,

    // 8.6.1
    InformationParkingMethod1,

    // 8.6.2
    InformationParkingMethod2,

    // 8.6.3
    InformationParkingMethod3,

    // 8.6.4
    InformationParkingMethod4,

    // 8.6.5
    InformationParkingMethod5,

    // 8.6.6
    InformationParkingMethod6,

    // 8.6.7
    InformationParkingMethod7,

    // 8.6.8
    InformationParkingMethod8,

    // 8.6.9
    InformationParkingMethod9,

    // 8.7
    InformationParkingDeadstick,

    // 8.9
    InformationParkingDurationLimit,

    // 8.10
    InformationPlaceForCarsInspection,

    // 8.11
    InformationLimitMaximumWeight,

    // 8.12
    InformationDangerousRoadside,

    // 8.13
    InformationCourseOfMainRoad,

    // 8.14
    InformationLane,

    // 8.15
    InformationBlindPedestrians,

    // 8.16
    InformationWetCovering,

    // 8.17
    InformationInvalids,

    // 8.18
    InformationExceptInvalids,

    // 8.22.1
    InformationObstacleL,

    // 8.22.2
    InformationObstacleR,

    // 8.22.3
    InformationObstacleF,

    // 8.23
    InformationSpeedCamera,

    // 8.24
    InformationTowTruck,

    // 1.1
    WarningLevelCrossingWithBarrierOrGateAhead,

    // 1.2
    WarningRailwayLevelCrossingWithoutBarrierOrGateAhead,

    // 5.1
    PrescriptionMotorway,

    // 5.2
    PrescriptionEofMotorway,

    /**
     * @see
     * https://upload.wikimedia.org/wikipedia/commons/thumb/a/a8/1.18_Russian_road_marking.svg/512px-1.18_Russian_road_marking.svg.png
     */
    RoadMarkingLaneDirectionF,  // 1.18.1
    RoadMarkingLaneDirectionR,  // 1.18.2
    RoadMarkingLaneDirectionL,  // 1.18.3
    RoadMarkingLaneDirectionFR, // 1.18.4
    RoadMarkingLaneDirectionFL, // 1.18.5
    RoadMarkingLaneDirectionRL, // 1.18.6
};

enum class TemporarySign {
    No,
    Yes,
};


std::string toString(TrafficSignCategory tsc);

std::ostream& operator<<(std::ostream& os, TrafficSignCategory tsc);

TrafficSignCategory stringToTrafficSignCategory(const std::string& str);

std::string toString(TrafficSign tsc);

std::ostream& operator<<(std::ostream& os, TrafficSign ts);

TrafficSign stringToTrafficSign(const std::string& str);

TrafficSignCategory toTrafficSignCategory(TrafficSign ts);

TrafficSign flipTrafficSignHorz(TrafficSign ts);

std::string toString(TemporarySign ts);

std::ostream& operator<<(std::ostream& os, TemporarySign ts);

TemporarySign stringToTemporarySign(const std::string& str);

bool canBeTemporary(TrafficSign ts);

constexpr bool isRoadMarking(TrafficSign type)
{
    return TrafficSign::RoadMarkingLaneDirectionF <= type
           && type <= TrafficSign::RoadMarkingLaneDirectionRL;
}

constexpr bool needRecognizeNumber(TrafficSign type)
{
    return  TrafficSign::ProhibitoryMaxWeight        == type ||
            TrafficSign::ProhibitoryMaxWeightPerAxle == type ||
            TrafficSign::ProhibitoryMaxHeight        == type ||
            TrafficSign::ProhibitoryMaxWidth         == type ||

            TrafficSign::ProhibitoryMaxSpeed         == type ||
            TrafficSign::ProhibitoryMaxSpeed5        == type ||
            TrafficSign::ProhibitoryMaxSpeed10       == type ||
            TrafficSign::ProhibitoryMaxSpeed20       == type ||
            TrafficSign::ProhibitoryMaxSpeed30       == type ||
            TrafficSign::ProhibitoryMaxSpeed40       == type ||
            TrafficSign::ProhibitoryMaxSpeed50       == type ||
            TrafficSign::ProhibitoryMaxSpeed60       == type ||
            TrafficSign::ProhibitoryMaxSpeed70       == type ||
            TrafficSign::ProhibitoryMaxSpeed80       == type ||
            TrafficSign::ProhibitoryMaxSpeed90       == type ||
            TrafficSign::ProhibitoryMaxSpeed100      == type ||
            TrafficSign::ProhibitoryMaxSpeed110      == type ||
            TrafficSign::ProhibitoryMaxSpeed120      == type ||
            TrafficSign::ProhibitoryMaxSpeed130      == type ||
            TrafficSign::ProhibitoryMaxSpeed15       == type ||
            TrafficSign::ProhibitoryMaxSpeed25       == type ||
            TrafficSign::ProhibitoryMaxSpeed35       == type ||
            TrafficSign::ProhibitoryMaxSpeed45       == type ||
            TrafficSign::ProhibitoryMaxSpeed55       == type ||

            TrafficSign::ProhibitoryEofMaxSpeed      == type ||
            TrafficSign::ProhibitoryEofMaxSpeed5     == type ||
            TrafficSign::ProhibitoryEofMaxSpeed10    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed15    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed20    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed25    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed30    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed35    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed40    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed45    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed50    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed55    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed60    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed70    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed80    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed90    == type ||
            TrafficSign::ProhibitoryEofMaxSpeed100   == type ||
            TrafficSign::ProhibitoryEofMaxSpeed110   == type ||
            TrafficSign::ProhibitoryEofMaxSpeed120   == type ||
            TrafficSign::ProhibitoryEofMaxSpeed130   == type ||

            TrafficSign::ProhibitoryMaxLength        == type ||
            TrafficSign::ProhibitoryMinDistance      == type;
}

constexpr bool isAdditionalTable(TrafficSign type)
{
    return TrafficSign::InformationDistanceObject                 == type ||
           TrafficSign::InformationDistanceStopAhead              == type ||
           TrafficSign::InformationDistanceObjectRight            == type ||
           TrafficSign::InformationDistanceObjectLeft             == type ||

           TrafficSign::InformationStartZoneHorizontal            == type ||
           TrafficSign::InformationStartZone                      == type ||
           TrafficSign::InformationEofZone                        == type ||
           TrafficSign::InformationInZone                         == type ||

           TrafficSign::InformationDirectionalR                   == type ||
           TrafficSign::InformationDirectionalL                   == type ||
           TrafficSign::InformationDirectionalRL                  == type ||

           TrafficSign::InformationHeavyVehicle                   == type ||
           TrafficSign::InformationTrailer                        == type ||
           TrafficSign::InformationLightVehicle                   == type ||
           TrafficSign::InformationBus                            == type ||
           TrafficSign::InformationTractor                        == type ||
           TrafficSign::InformationMotorcycle                     == type ||
           TrafficSign::InformationBicycle                        == type ||
           TrafficSign::InformationVehiclesCarryingDangerousGoods == type ||
           TrafficSign::InformationExceptHeavyVehicle             == type ||
           TrafficSign::InformationExceptLightVehicle             == type ||
           TrafficSign::InformationExceptBus                      == type ||
           TrafficSign::InformationExceptMotorcycle               == type ||
           TrafficSign::InformationExceptBicycle                  == type ||
           TrafficSign::InformationExceptTaxi                     == type ||

           TrafficSign::InformationHoliday                        == type ||
           TrafficSign::InformationWorkingDay                     == type ||

           TrafficSign::InformationTime                           == type ||
           TrafficSign::InformationHolidayTime                    == type ||
           TrafficSign::InformationWorkingDayTime                 == type ||

           TrafficSign::InformationParkingMethod1                 == type ||
           TrafficSign::InformationParkingMethod2                 == type ||
           TrafficSign::InformationParkingMethod3                 == type ||
           TrafficSign::InformationParkingMethod4                 == type ||
           TrafficSign::InformationParkingMethod5                 == type ||
           TrafficSign::InformationParkingMethod6                 == type ||
           TrafficSign::InformationParkingMethod7                 == type ||
           TrafficSign::InformationParkingMethod8                 == type ||
           TrafficSign::InformationParkingMethod9                 == type ||

           TrafficSign::InformationParkingDeadstick               == type ||

           TrafficSign::InformationPaidServices                   == type ||

           TrafficSign::InformationParkingDurationLimit           == type ||

           TrafficSign::InformationPlaceForCarsInspection         == type ||

           TrafficSign::InformationLimitMaximumWeight             == type ||

           TrafficSign::InformationDangerousRoadside              == type ||

           TrafficSign::InformationCourseOfMainRoad               == type ||

           TrafficSign::InformationLane                           == type ||

           TrafficSign::InformationBlindPedestrians               == type ||

           TrafficSign::InformationWetCovering                    == type ||

           TrafficSign::InformationInvalids                       == type ||

           TrafficSign::InformationExceptInvalids                 == type ||

//         такие таблички не используются, как дополнительные для знаков
//           TrafficSign::InformationObstacleL                      == type ||
//           TrafficSign::InformationObstacleR                      == type ||
//           TrafficSign::InformationObstacleF                      == type ||

           TrafficSign::InformationSpeedCamera                    == type ||
           TrafficSign::InformationTowTruck                       == type;
}

constexpr bool isLaneDirectionPrescription(TrafficSign type)
{
    return TrafficSign::PrescriptionLanesDirection     == type ||
           TrafficSign::PrescriptionLaneDirectionFL    == type ||
           TrafficSign::PrescriptionLaneDirectionFR    == type ||
           TrafficSign::PrescriptionLaneDirectionF     == type ||
           TrafficSign::PrescriptionLaneDirectionFl    == type ||
           TrafficSign::PrescriptionLaneDirectionFr    == type ||
           TrafficSign::PrescriptionLaneDirectionL     == type ||
           TrafficSign::PrescriptionLaneDirectionRL    == type ||
           TrafficSign::PrescriptionLaneDirectionR     == type ||
           TrafficSign::PrescriptionLaneDirectionFFr   == type ||
           TrafficSign::PrescriptionLaneDirectionFFl   == type ||
           TrafficSign::PrescriptionLaneDirectionFrR   == type ||
           TrafficSign::PrescriptionLaneDirectionFrFl  == type ||
           TrafficSign::PrescriptionLaneDirectionB     == type ||
           TrafficSign::PrescriptionLaneDirectionLFl   == type ||
           TrafficSign::PrescriptionLaneDirectionOther == type;
}

} // namespace traffic_signs
} // namespace mrc
} // namespace maps
