#include <yandex/maps/mrc/traffic_signs/signs.h>

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <util/generic/serialized_enum.h>
#include <util/string/cast.h>

#include <algorithm>
#include <iterator>

#include <boost/bimap.hpp>
#include <boost/bimap/multiset_of.hpp>
#include <boost/lexical_cast.hpp>

namespace maps {
namespace mrc {
namespace traffic_signs {
namespace {

using TrafficSignCategoryToTrafficSignMap
    = boost::bimap<boost::bimaps::multiset_of<TrafficSignCategory>, TrafficSign>;

const std::vector<TrafficSignCategoryToTrafficSignMap::value_type>
    ROAD_SIGN_CAT_ROAD_SIGN_MAP_VALUES{

        {TrafficSignCategory::YellowRhomb, TrafficSign::PriorityPriorityRoad},
        {TrafficSignCategory::YellowRhomb, TrafficSign::PriorityEofPriorityRoad},

        {TrafficSignCategory::RedCircle, TrafficSign::PriorityStop},
        {TrafficSignCategory::RedCircle, TrafficSign::PriorityStopGesture},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryNoEntry},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryNoVehicles},
        {TrafficSignCategory::RedCircle,
         TrafficSign::ProhibitoryNoHeavyGoodsVehicles},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryNoRightTurn},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryNoLeftTurn},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryNoUturn},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxWeight},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxWeightPerAxle},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxHeight},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxWidth},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed5},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed10},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed20},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed30},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed40},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed50},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed60},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed70},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed80},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed90},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed100},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed110},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed120},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed130},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed15 },
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed25 },
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed35 },
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed45 },
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryMaxSpeed55 },

        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryNoParking},
        {TrafficSignCategory::RedCircle,
         TrafficSign::ProhibitoryNoParkingOrStopping},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryNoOvertaking},
        {TrafficSignCategory::RedCircle, TrafficSign::ProhibitoryNoOvertakingByHeavyVehicle},

        {TrafficSignCategory::BlueCircle, TrafficSign::MandatoryProceedStraight},
        {TrafficSignCategory::BlueCircle,
         TrafficSign::MandatoryProceedStraightOrTurnLeft},
        {TrafficSignCategory::BlueCircle,
         TrafficSign::MandatoryProceedStraightOrTurnRight},
        {TrafficSignCategory::BlueCircle, TrafficSign::MandatoryRoundabout},
        {TrafficSignCategory::BlueCircle, TrafficSign::MandatoryTurnLeft},
        {TrafficSignCategory::BlueCircle, TrafficSign::MandatoryTurnLeftAhead},
        {TrafficSignCategory::BlueCircle, TrafficSign::MandatoryTurnRight},
        {TrafficSignCategory::BlueCircle, TrafficSign::MandatoryTurnRightAhead},

        {TrafficSignCategory::RedFlippedTriagnleWithWhiteBackgound,
         TrafficSign::PriorityGiveWay},

        {TrafficSignCategory::BlackCircleWithWhiteBackground,
         TrafficSign::ProhibitoryEofMaxSpeed},
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed5 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed10 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed20 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed30 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed40 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed50 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed60 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed70 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed80 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed90 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed100 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed110 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed120 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed130 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed15 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed25 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed35 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed45 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofMaxSpeed55 },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofNoOvertaking },
        {TrafficSignCategory::BlackCircleWithWhiteBackground, TrafficSign::ProhibitoryEofNoOvertakingByHeavyVehicle},

        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionOneWayRoad},
        {TrafficSignCategory::BlueRectangle,
         TrafficSign::PrescriptionEofOneWayRoad},
        {TrafficSignCategory::BlueRectangle,
         TrafficSign::PrescriptionEntryToOneWayRoadOnTheRight},
        {TrafficSignCategory::BlueRectangle,
         TrafficSign::PrescriptionEntryToOneWayRoadOnTheLeft},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionRoadWithBusLane},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionEofRoadWithBusLane},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionBusLane},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionEofBusLane},
        {TrafficSignCategory::BlueRectangle,
         TrafficSign::PrescriptionLanesDirection},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionFL},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionFR},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionF},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionFl},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionFr},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionL},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionRL},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionR},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionFFr},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionFFl},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionFrR},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionFrFl},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionB},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionLFl},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionLaneDirectionOther},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionStartNewLineRight},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionStartNewLineLeft},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionEndLineRight},
        {TrafficSignCategory::BlueRectangle, TrafficSign::PrescriptionEndLineLeft},
        {TrafficSignCategory::BlueRectangle, TrafficSign::InformationParking},

        {TrafficSignCategory::Plate, TrafficSign::PrescriptionBuiltUpArea},
        {TrafficSignCategory::Plate, TrafficSign::PrescriptionEofBuiltUpArea},

        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningRoundaboutAhead},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningUnevenRoadAhead},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningRoadNarrowsOnBoth},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningRoadNarrowsOnRight},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningRoadNarrowsOnLeft},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningPedestrianCrossingAhead},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningChildren},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningCrossroadsMinorRoadRL},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningCrossroadsMinorRoadR},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningCrossroadsMinorRoadL},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningCrossroadsMinorRoadFr},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningCrossroadsMinorRoadFl},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningCrossroadsMinorRoadBr},
        {TrafficSignCategory::RedTriangleWithWhiteBackground, TrafficSign::WarningCrossroadsMinorRoadBl},

        {TrafficSignCategory::Plate, TrafficSign::InformationDistanceStopAhead},
        {TrafficSignCategory::Plate, TrafficSign::InformationDistanceObjectRight},
        {TrafficSignCategory::Plate, TrafficSign::InformationDistanceObjectLeft},
        {TrafficSignCategory::Plate, TrafficSign::InformationStartZone},
        {TrafficSignCategory::Plate, TrafficSign::InformationInZone},
        {TrafficSignCategory::Plate, TrafficSign::InformationHeavyVehicle},
        {TrafficSignCategory::Plate, TrafficSign::InformationLightVehicle},
        {TrafficSignCategory::Plate, TrafficSign::InformationHoliday},
        {TrafficSignCategory::Plate, TrafficSign::InformationWorkingDay},
        {TrafficSignCategory::Plate, TrafficSign::InformationHolidayTime},
        {TrafficSignCategory::Plate, TrafficSign::InformationWorkingDayTime},
        {TrafficSignCategory::Plate, TrafficSign::InformationPaidServices}};

const TrafficSignCategoryToTrafficSignMap ROAD_SIGN_CATEGORY_TO_ROAD_SIGN_MAP{
    ROAD_SIGN_CAT_ROAD_SIGN_MAP_VALUES.begin(),
    ROAD_SIGN_CAT_ROAD_SIGN_MAP_VALUES.end()};

const std::map<TrafficSign, TrafficSign> TrafficSignFlipHorzMap = {
    {TrafficSign::PriorityPriorityRoad, TrafficSign::PriorityPriorityRoad},
    {TrafficSign::PriorityGiveWay, TrafficSign::PriorityGiveWay},

    {TrafficSign::ProhibitoryNoEntry, TrafficSign::ProhibitoryNoEntry},
    {TrafficSign::ProhibitoryNoVehicles, TrafficSign::ProhibitoryNoVehicles},
    {TrafficSign::ProhibitoryNoParkingOrStopping, TrafficSign::ProhibitoryNoParkingOrStopping},
    {TrafficSign::ProhibitoryNoMotorVehicles, TrafficSign::ProhibitoryNoMotorVehicles},
    {TrafficSign::ProhibitoryNoVehiclesCarryingDangerousGoods, TrafficSign::ProhibitoryNoVehiclesCarryingDangerousGoods },
    {TrafficSign::ProhibitoryNoVehiclesCarryingExplosives, TrafficSign::ProhibitoryNoVehiclesCarryingExplosives },

    {TrafficSign::MandatoryProceedStraight, TrafficSign::MandatoryProceedStraight},
    {TrafficSign::MandatoryProceedStraightOrTurnLeft, TrafficSign::MandatoryProceedStraightOrTurnRight},
    {TrafficSign::MandatoryProceedStraightOrTurnRight, TrafficSign::MandatoryProceedStraightOrTurnLeft},
    {TrafficSign::MandatoryTurnLeft,  TrafficSign::MandatoryTurnRight},
    {TrafficSign::MandatoryTurnRight, TrafficSign::MandatoryTurnLeft},
    {TrafficSign::MandatoryTurnLeftAhead,  TrafficSign::MandatoryTurnRightAhead},
    {TrafficSign::MandatoryTurnRightAhead, TrafficSign::MandatoryTurnLeftAhead},

    {TrafficSign::MandatoryTurnRightOrLeft, TrafficSign::MandatoryTurnRightOrLeft},

    {TrafficSign::MandatoryDrivingDirectionR, TrafficSign::MandatoryDrivingDirectionL},
    {TrafficSign::MandatoryDrivingDirectionL, TrafficSign::MandatoryDrivingDirectionR},
    {TrafficSign::MandatoryDrivingDirectionRL, TrafficSign::MandatoryDrivingDirectionRL},

    {TrafficSign::MandatoryVehiclesCarryingExplosivesProceedStraight, TrafficSign::MandatoryVehiclesCarryingExplosivesProceedStraight},
    {TrafficSign::MandatoryVehiclesCarryingExplosivesTurnRightAhead, TrafficSign::MandatoryVehiclesCarryingExplosivesTurnLeftAhead},
    {TrafficSign::MandatoryVehiclesCarryingExplosivesTurnLeftAhead, TrafficSign::MandatoryVehiclesCarryingExplosivesTurnRightAhead},

    {TrafficSign::PrescriptionOneWayRoad, TrafficSign::PrescriptionOneWayRoad},
    {TrafficSign::PrescriptionEntryToOneWayRoadOnTheRight, TrafficSign::PrescriptionEntryToOneWayRoadOnTheLeft},
    {TrafficSign::PrescriptionEntryToOneWayRoadOnTheLeft, TrafficSign::PrescriptionEntryToOneWayRoadOnTheRight},

    {TrafficSign::PrescriptionBusLane, TrafficSign::PrescriptionBusLane},

    {TrafficSign::PrescriptionLanesDirection, TrafficSign::PrescriptionLanesDirection },

    {TrafficSign::PrescriptionLaneDirectionFL, TrafficSign::PrescriptionLaneDirectionFR},
    {TrafficSign::PrescriptionLaneDirectionFR, TrafficSign::PrescriptionLaneDirectionFL},
    {TrafficSign::PrescriptionLaneDirectionF, TrafficSign::PrescriptionLaneDirectionF},
    {TrafficSign::PrescriptionLaneDirectionFl, TrafficSign::PrescriptionLaneDirectionFr},
    {TrafficSign::PrescriptionLaneDirectionFr, TrafficSign::PrescriptionLaneDirectionFl},
    {TrafficSign::PrescriptionLaneDirectionL, TrafficSign::PrescriptionLaneDirectionR},
    {TrafficSign::PrescriptionLaneDirectionR, TrafficSign::PrescriptionLaneDirectionL},
    {TrafficSign::PrescriptionLaneDirectionRL, TrafficSign::PrescriptionLaneDirectionRL},
    {TrafficSign::PrescriptionLaneDirectionFFl, TrafficSign::PrescriptionLaneDirectionFFr},
    {TrafficSign::PrescriptionLaneDirectionFFr, TrafficSign::PrescriptionLaneDirectionFFl},
    {TrafficSign::PrescriptionLaneDirectionFrFl, TrafficSign::PrescriptionLaneDirectionFrFl},
    {TrafficSign::PrescriptionLaneDirectionB, TrafficSign::PrescriptionLaneDirectionB},
    {TrafficSign::PrescriptionLaneDirectionLFl, TrafficSign::PrescriptionLaneDirectionFrR},
    {TrafficSign::PrescriptionLaneDirectionFrR, TrafficSign::PrescriptionLaneDirectionLFl},

    {TrafficSign::PrescriptionStartNewLineRight, TrafficSign::PrescriptionStartNewLineLeft},
    {TrafficSign::PrescriptionStartNewLineLeft, TrafficSign::PrescriptionStartNewLineRight},
    {TrafficSign::PrescriptionEndLineRight, TrafficSign::PrescriptionEndLineLeft},
    {TrafficSign::PrescriptionEndLineLeft, TrafficSign::PrescriptionEndLineRight},

    {TrafficSign::PrescriptionLimitedAccessRoad, TrafficSign::PrescriptionLimitedAccessRoad},

    {TrafficSign::PrescriptionTwoWayTraffic, TrafficSign::PrescriptionTwoWayTraffic},
    {TrafficSign::PrescriptionEntryToTwoWayTraffic, TrafficSign::PrescriptionEntryToTwoWayTraffic},

    {TrafficSign::PrescriptionCycleLane, TrafficSign::PrescriptionCycleLane},

    {TrafficSign::PrescriptionRoadHump, TrafficSign::PrescriptionRoadHump},

    {TrafficSign::InformationCulDeSacF, TrafficSign::InformationCulDeSacF},
    {TrafficSign::InformationCulDeSacR, TrafficSign::InformationCulDeSacL},
    {TrafficSign::InformationCulDeSacL, TrafficSign::InformationCulDeSacR},

    {TrafficSign::WarningUnevenRoadAhead, TrafficSign::WarningUnevenRoadAhead},
    {TrafficSign::WarningRoadNarrowsOnBoth, TrafficSign::WarningRoadNarrowsOnBoth},
    {TrafficSign::WarningRoadNarrowsOnRight, TrafficSign::WarningRoadNarrowsOnLeft},
    {TrafficSign::WarningRoadNarrowsOnLeft, TrafficSign::WarningRoadNarrowsOnRight},

    {TrafficSign::WarningCrossroadsMinorRoadRL, TrafficSign::WarningCrossroadsMinorRoadRL},
    {TrafficSign::WarningCrossroadsMinorRoadR, TrafficSign::WarningCrossroadsMinorRoadL},
    {TrafficSign::WarningCrossroadsMinorRoadL, TrafficSign::WarningCrossroadsMinorRoadR},
    {TrafficSign::WarningCrossroadsMinorRoadFr, TrafficSign::WarningCrossroadsMinorRoadFl},
    {TrafficSign::WarningCrossroadsMinorRoadFl, TrafficSign::WarningCrossroadsMinorRoadFr},
    {TrafficSign::WarningCrossroadsMinorRoadBr, TrafficSign::WarningCrossroadsMinorRoadBl},
    {TrafficSign::WarningCrossroadsMinorRoadBl, TrafficSign::WarningCrossroadsMinorRoadBr},

    {TrafficSign::WarningLevelCrossingCountdown3R, TrafficSign::WarningLevelCrossingCountdown3L},
    {TrafficSign::WarningLevelCrossingCountdown2R, TrafficSign::WarningLevelCrossingCountdown2L},
    {TrafficSign::WarningLevelCrossingCountdown1R, TrafficSign::WarningLevelCrossingCountdown1L},
    {TrafficSign::WarningLevelCrossingCountdown3L, TrafficSign::WarningLevelCrossingCountdown3R},
    {TrafficSign::WarningLevelCrossingCountdown2L, TrafficSign::WarningLevelCrossingCountdown2R},
    {TrafficSign::WarningLevelCrossingCountdown1L, TrafficSign::WarningLevelCrossingCountdown1R},

    {TrafficSign::WarningTramCrossingAhead, TrafficSign::WarningTramCrossingAhead},
    {TrafficSign::WarningCrossroadsWithPriorityToTheRight, TrafficSign::WarningCrossroadsWithPriorityToTheRight},
    {TrafficSign::WarningTrafficSignalsAhead, TrafficSign::WarningTrafficSignalsAhead},
    {TrafficSign::WarningDangerousBendR, TrafficSign::WarningDangerousBendL},
    {TrafficSign::WarningDangerousBendL, TrafficSign::WarningDangerousBendR},
    {TrafficSign::WarningDangerousBendsR, TrafficSign::WarningDangerousBendsL},
    {TrafficSign::WarningDangerousBendsL, TrafficSign::WarningDangerousBendsR},
    {TrafficSign::WarningUnevenRoad, TrafficSign::WarningUnevenRoad},
    {TrafficSign::WarningOtherDangers, TrafficSign::WarningOtherDangers},

    {TrafficSign::WarningTurningDirectionR, TrafficSign::WarningTurningDirectionL},
    {TrafficSign::WarningTurningDirectionL, TrafficSign::WarningTurningDirectionR},
    {TrafficSign::WarningTurningDirectionRL, TrafficSign::WarningTurningDirectionRL},

    {TrafficSign::InformationInZone, TrafficSign::InformationInZone},
    {TrafficSign::InformationHoliday, TrafficSign::InformationHoliday},
    {TrafficSign::InformationWorkingDay, TrafficSign::InformationWorkingDay},

    {TrafficSign::InformationFirstAidStation, TrafficSign::InformationFirstAidStation},
    {TrafficSign::InformationCarWash, TrafficSign::InformationCarWash},
    {TrafficSign::InformationCamping, TrafficSign::InformationCamping},

    {TrafficSign::InformationEofZone, TrafficSign::InformationEofZone},
    {TrafficSign::InformationDirectionalR, TrafficSign::InformationDirectionalL},
    {TrafficSign::InformationDirectionalL, TrafficSign::InformationDirectionalR},
    {TrafficSign::InformationDirectionalRL, TrafficSign::InformationDirectionalRL},
    {TrafficSign::InformationVehiclesCarryingDangerousGoods, TrafficSign::InformationVehiclesCarryingDangerousGoods},

    {TrafficSign::InformationLane, TrafficSign::InformationLane},

    {TrafficSign::InformationObstacleL, TrafficSign::InformationObstacleR},
    {TrafficSign::InformationObstacleR, TrafficSign::InformationObstacleL},
    {TrafficSign::InformationObstacleF, TrafficSign::InformationObstacleF}
};

const std::set<TrafficSign> TrafficSignCanBeTemporary = {
    TrafficSign::ProhibitoryMaxSpeed,
    TrafficSign::ProhibitoryMaxSpeed5,
    TrafficSign::ProhibitoryMaxSpeed10,
    TrafficSign::ProhibitoryMaxSpeed20,
    TrafficSign::ProhibitoryMaxSpeed30,
    TrafficSign::ProhibitoryMaxSpeed40,
    TrafficSign::ProhibitoryMaxSpeed50,
    TrafficSign::ProhibitoryMaxSpeed60,
    TrafficSign::ProhibitoryMaxSpeed70,
    TrafficSign::ProhibitoryMaxSpeed80,
    TrafficSign::ProhibitoryMaxSpeed90,
    TrafficSign::ProhibitoryMaxSpeed100,
    TrafficSign::ProhibitoryMaxSpeed110,
    TrafficSign::ProhibitoryMaxSpeed120,
    TrafficSign::ProhibitoryMaxSpeed130,
    TrafficSign::ProhibitoryMaxSpeed15,
    TrafficSign::ProhibitoryMaxSpeed25,
    TrafficSign::ProhibitoryMaxSpeed35,
    TrafficSign::ProhibitoryMaxSpeed45,
    TrafficSign::ProhibitoryMaxSpeed55
};

// Camel case to lower with underscore
template <typename Enum>
std::ostream& serialize(std::ostream& os, Enum e)
{
    bool noPrev = true;
    bool prevIsDigit = false;
    for (const auto& c : ToString(e)) {
        if (isdigit(c)) {
            if (!prevIsDigit) {
                os << '_';
            }
            os << c;
            prevIsDigit = true;
        }
        else {
            if (isupper(c)) {
                if (!noPrev) {
                    os << '_';
                }
                os << (char)tolower(c);
            }
            else {
                os << c;
            }
            prevIsDigit = false;
        }
        noPrev = false;
    }
    return os;
}

// Lower with underscore to camel case
template <typename Enum>
Enum deserialize(const std::string& str) try {
    std::ostringstream os;
    bool toUpper = true;
    for (const auto& c : str) {
        if (toUpper) {
            os << (char)toupper(c);
            toUpper = false;
        }
        else if (c == '_') {
            toUpper = true;
        }
        else {
            os << c;
        }
    }
    return FromString<Enum>(os.str());
}
catch (const std::exception&) {
    auto enums = GetEnumAllValues<Enum>();
    std::vector<std::string> strs;
    std::transform(enums.begin(), enums.end(), std::back_inserter(strs),
                   [](const auto& item) { return toString(item); });
    std::sort(strs.begin(), strs.end());
    throw maps::Exception{} << "Invalid enum value: " << str << " ("
                            << wiki::common::join(strs, ",") << ").";
}

} // namespace

std::string toString(TrafficSignCategory tsc)
{
    return boost::lexical_cast<std::string>(tsc);
}

TrafficSignCategory stringToTrafficSignCategory(const std::string& str)
{
    return deserialize<TrafficSignCategory>(str);
}

std::string toString(TrafficSign ts)
{
    return boost::lexical_cast<std::string>(ts);
}

std::ostream& operator<<(std::ostream& os, TrafficSign ts) {
    return serialize(os, ts);
}

TrafficSign stringToTrafficSign(const std::string& str)
{
    return deserialize<TrafficSign>(str);
}

TrafficSignCategory toTrafficSignCategory(TrafficSign ts)
{
    auto it = ROAD_SIGN_CATEGORY_TO_ROAD_SIGN_MAP.right.find(ts);
    REQUIRE(it != ROAD_SIGN_CATEGORY_TO_ROAD_SIGN_MAP.right.end(),
            "Invalid TrafficSign: "
                << static_cast<std::underlying_type<TrafficSign>::type>(ts));
    return it->second;
}


std::ostream& operator<<(std::ostream& os, TrafficSignCategory tsc) {
    return serialize(os, tsc);
}

TrafficSign flipTrafficSignHorz(TrafficSign ts) {
    auto result = TrafficSignFlipHorzMap.find(ts);
    if (result != TrafficSignFlipHorzMap.end())
        return result->second;
    return TrafficSign::Unknown;
}

std::string toString(TemporarySign ts)
{
    return boost::lexical_cast<std::string>(ts);
}

std::ostream& operator<<(std::ostream& os, TemporarySign ts) {
    return serialize(os, ts);
}

TemporarySign stringToTemporarySign(const std::string& str)
{
    return deserialize<TemporarySign>(str);
}

bool canBeTemporary(TrafficSign ts)
{
    return (TrafficSignCanBeTemporary.find(ts) != TrafficSignCanBeTemporary.end());
}


} // namespace traffic_signs
} // namespace mrc
} // namespace maps
