#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/common/include/exception.h>
#include <yandex/maps/mrc/traffic_signs/signs.h>

namespace maps {
namespace mrc {
namespace traffic_signs {

namespace tests {

TEST(tests, traffic_sign_category_conversion)
{
    EXPECT_TRUE(!toString(TrafficSignCategory::Unknown).empty());

    const int WRONG_ENUM_VALUE = 100500;
    EXPECT_THROW(toString(static_cast<TrafficSignCategory>(WRONG_ENUM_VALUE)),
                      std::exception);

    auto rsc = TrafficSignCategory::YellowRhomb;
    EXPECT_EQ(stringToTrafficSignCategory(toString(rsc)), rsc);
}

TEST(tests, traffic_sign_conversion)
{
    EXPECT_TRUE(!toString(TrafficSign::Unknown).empty());

    const int WRONG_ENUM_VALUE = 100500;
    EXPECT_THROW(toString(static_cast<TrafficSign>(WRONG_ENUM_VALUE)),
                      std::exception);

    const std::unordered_map<TrafficSign, std::string> CASES
        = {{TrafficSign::ProhibitoryMaxSpeed15,
            "prohibitory_max_speed_15"},
           {TrafficSign::PrescriptionLaneDirectionFR,
            "prescription_lane_direction_f_r"},
           {TrafficSign::PrescriptionLaneDirectionFr,
            "prescription_lane_direction_fr"},
           {TrafficSign::PrescriptionLaneDirectionFrFl,
            "prescription_lane_direction_fr_fl"},
           {TrafficSign::WarningLevelCrossingCountdown3R,
            "warning_level_crossing_countdown_3_r"}};
    for (auto & [ key, val ] : CASES) {
        EXPECT_EQ(toString(key), val);
        EXPECT_EQ(stringToTrafficSign(val), key);
    }
}

TEST(tests, traffic_sign_category_calculation)
{
    EXPECT_EQ(toTrafficSignCategory(TrafficSign::InformationParking),
        TrafficSignCategory::BlueRectangle);

    const int WRONG_ENUM_VALUE = 100500;
    EXPECT_THROW(toTrafficSignCategory(static_cast<TrafficSign>(WRONG_ENUM_VALUE)),
                      maps::Exception);
}

TEST(tests, temporary_sign_conversion)
{
    const int WRONG_ENUM_VALUE = 100500;
    EXPECT_THROW(toString(static_cast<TrafficSign>(WRONG_ENUM_VALUE)),
                        std::exception);

    for (auto value : {TemporarySign::Yes, TemporarySign::No}) {
        EXPECT_EQ(stringToTemporarySign(toString(value)), value);
    }

}

TEST(tests, flip_trafic_sign_horizontal)
{
    EXPECT_EQ(flipTrafficSignHorz(TrafficSign::PriorityPriorityRoad), TrafficSign::PriorityPriorityRoad);
    EXPECT_EQ(flipTrafficSignHorz(TrafficSign::PriorityEofPriorityRoad), TrafficSign::Unknown);
}

TEST(tests, check_traffic_signs_need_recognize_number)
{
    EXPECT_EQ(needRecognizeNumber(TrafficSign::PriorityPriorityRoad), false);
    EXPECT_EQ(needRecognizeNumber(TrafficSign::ProhibitoryEofMaxSpeed), true);
}

TEST(tests, check_traffic_signs_additional_table)
{
    EXPECT_EQ(isAdditionalTable(TrafficSign::PriorityPriorityRoad), false);
    EXPECT_EQ(isAdditionalTable(TrafficSign::InformationHeavyVehicle), true);
}
} // namespace test
} // namespace traffic_signs
} // namespace mrc
} // namespace maps
