#pragma once

#include <maps/libs/xml/include/xml.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/concurrent/include/background_thread.h>
#include <maps/libs/http/include/request.h>
#include <maps/libs/process/include/process.h>
#include <boost/lexical_cast.hpp>

#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/common/network.h>

#include <chrono>
#include <optional>
#include <thread>

#include <sys/socket.h>
#include <arpa/inet.h>
#ifdef htonl
#undef htonl
#endif

namespace maps::mrc::unittest {

const std::string LOCAL_HOST = "127.0.0.1";

struct WaitForTestServer {
    int socket;

    WaitForTestServer(std::uint16_t port)
    {
        ::sockaddr_in testServerEp;
        ::memset(&testServerEp, 0, sizeof(testServerEp));
        testServerEp.sin_family = AF_INET;
        testServerEp.sin_port = htons(port);
        testServerEp.sin_addr.s_addr
            = ::inet_addr(LOCAL_HOST.c_str());
        socket = ::socket(PF_INET, SOCK_STREAM, 0);
        // Try to connect to the test server but not longer that for 60 s
        std::size_t attempts = 10 * 60;
        while (::connect(socket, (sockaddr*)&testServerEp,
                         sizeof(testServerEp)) != 0) {
            using namespace std::literals::chrono_literals;
            std::this_thread::sleep_for(100ms);
            REQUIRE(--attempts, "Local test server is dead");
        }
    }

    ~WaitForTestServer() noexcept(false)
    {
        REQUIRE(::shutdown(socket, SHUT_RDWR) == 0,
                "Cannot shutdown connection with the test server");
        REQUIRE(::close(socket) == 0, "Cannot close the socket");
    }
};

using CommandProvider = std::function<process::Command(uint16_t /*port*/)>;

class TestServer {
public:
    /**
     * @param port local port to start server at
     * @param serverBinaryPath path to the binary, from Arcadia root
     */
    TestServer(CommandProvider commandProvider)
        : port_(NTesting::GetFreePort())
        , serverProcess_(process::run(commandProvider(port_)))
    {
        WaitForTestServer waitForTestServer(port_);
    }

    ~TestServer()
    {
        if (serverProcess_.finished()) {
            return;
        }

        serverProcess_.sendSignal(SIGTERM);

        int guardCount = 0;
        concurrent::BackgroundThread guard(
            [&]() {
                if (guardCount == 1) {
                    serverProcess_.sendSignal(SIGKILL);
                }
                ++guardCount;
            },
            std::chrono::seconds(30)
        );
        guard.start();
        serverProcess_.syncWait();
    }

    const std::string& getHost() const {return LOCAL_HOST;}
    std::uint16_t getPort() const { return port_; }
    std::string getPortStr() const { return std::to_string(port_); }

private:
    NTesting::TPortHolder port_;
    process::Process serverProcess_;
};

// To use the MDS stub fixture you must add the following line to a test's
// ya.make:
// DEPENDS(maps/wikimap/mapspro/services/mrc/libs/unittest/stubs/mds)

struct MdsStubServerProvider {
    auto operator()(uint16_t port) {
        const std::string mdsStub = BinaryPath("maps/wikimap/mapspro/services/mrc/libs/unittest/stubs/mds/mds-stub");

        std::vector<std::string> command {mdsStub, "--port", std::to_string(port)};
        if (storage) {
            command.push_back("--storage");
            command.push_back(*storage);
        }

        return process::Command(command);
    }

    std::optional<std::string> storage;
};

const std::string MDS_GROUP_ID = "100500";

class MdsStubFixture : private TestServer {
public:
    MdsStubFixture(): TestServer(MdsStubServerProvider{}) {}

    explicit MdsStubFixture(
            xml3::Doc& configDoc,
            const std::optional<std::string>& storage = std::nullopt)
        : TestServer(MdsStubServerProvider{storage})
    {
        INFO() << "MDS stub is running on " << getHost() << ":" << getPort();

        const char* const MDS_XPATH = "/config/external-services/mds";
        const char* const MDS_PUBLIC_XPATH = "/config/external-services/mds-public";
        {
            auto node = configDoc.node(MDS_XPATH);
            node.setAttr("host", getHost());
            node.setAttr("write-port", getPortStr());
            node.setAttr("read-port", getPortStr());
        }
        {
            auto node = configDoc.node(MDS_PUBLIC_XPATH);
            node.setAttr("read-host", getHost());
            node.setAttr("write-host", getHost());
            node.setAttr("write-port", getPortStr());
            node.setAttr("read-port", getPortStr());
        }
    }

    /// Clears all the data
    void clearMds()
    {
        INFO() << "Clearing MDS stub";
        http::Client client;
        http::Request request(client, http::POST, "http://" + getMdsHost()
            + ":" + std::to_string(getMdsPort()) + "/_clear");
        auto response = request.perform();
        REQUIRE(response.status() == 200, "MDS stub responded with status " << response.status());
    }

    std::uint16_t getMdsPort() const { return getPort(); }
    const std::string& getMdsHost() const { return getHost(); }
};

// To use the MDS stub fixture you must add the following line to a test's
// ya.make:
// DEPENDS(maps/wikimap/mapspro/services/mrc/libs/unittest/stubs/social)
const auto SOCIAL_STUB_SERVER_PROVIDER = [](uint16_t port) {
    return process::Command(
        {BinaryPath("maps/wikimap/mapspro/services/mrc/libs/unittest/"
                    "stubs/social/social-stub"),
         LOCAL_HOST,
         std::to_string(port)});
};

class SocialStubFixture : private TestServer {
public:
    SocialStubFixture()
        : TestServer(SOCIAL_STUB_SERVER_PROVIDER)
    { }

    explicit SocialStubFixture(xml3::Doc& configDoc)
        : TestServer(SOCIAL_STUB_SERVER_PROVIDER)
    {
        INFO() << "Social stub is running on " << getHost() << ":" << getPort();

        static const char* const SOCIAL_XPATH = "/config/external-services/social-backoffice";
        {
            auto node = configDoc.node(SOCIAL_XPATH);
            node.setAttr("url", "http://localhost:" + getPortStr());
        }
    }

    std::uint16_t getSocialPort() const { return getPort(); }
    const std::string& getSocialHost() const { return getHost(); }
};

// To use the YAVISION stub fixture you must add the following line to a test's
// ya.make:
// DEPENDS(maps/wikimap/mapspro/services/mrc/libs/unittest/stubs/yavision)
const auto YAVISION_STUB_SERVER_PROVIDER = [](uint16_t port) {
    return process::Command(
        {BinaryPath("maps/wikimap/mapspro/services/mrc/libs/unittest/"
                    "stubs/yavision/yavision-stub"),
         LOCAL_HOST,
         std::to_string(port)});
};

class YavisionStubFixture : private TestServer {
public:
    YavisionStubFixture()
        : TestServer(YAVISION_STUB_SERVER_PROVIDER)
    { }

    explicit YavisionStubFixture(xml3::Doc& configDoc)
        : TestServer(YAVISION_STUB_SERVER_PROVIDER)
    {
        INFO() << "Yavision stub is running on " << getHost() << ":" << getPort();

        const char* const YAVISION_XPATH = "/config/external-services/yavision";
        {
            auto node = configDoc.node(YAVISION_XPATH);
            node.setAttr("url", http::URL(node.attr("url")).setPort(getPort()).toString());
        }
    }

    std::uint16_t getYavisionPort() const { return getPort(); }
    const std::string& getYavisionHost() const { return getHost(); }
};

} // namespace maps::mrc::unittest
