#include <maps/libs/common/include/exception.h>

extern "C" {
#include <libavformat/avformat.h>
#include <libavcodec/avcodec.h>
#include <libavutil/avutil.h>
#include <libavutil/pixdesc.h>
#include <libswscale/swscale.h>
}

namespace maps::mrc::video {

inline void checkReturnCode(int ret, const std::string& functionName)
{
    REQUIRE(ret >= 0, "Error in function " << functionName << ": " << av_err2str(ret));
}

// RAII Wrapper for FFMpeg type AVFormatContext
class AVFormatContextWrapper {
public:
    void open(const std::string& uri)
    {
        close();
        int ret = avformat_open_input(&wrapped_, uri.c_str(), nullptr, nullptr);
        checkReturnCode(ret, "avformat_open_input");
    }

    void close()
    {
        if (wrapped_) {
            avformat_close_input(&wrapped_);
            wrapped_ = nullptr;
        }
    }

    bool isOpened() const { return wrapped_ != nullptr; }

    AVFormatContext* get() { return wrapped_; }
    AVFormatContext* operator->() { return wrapped_; }

    ~AVFormatContextWrapper() { close(); }

private:
    AVFormatContext* wrapped_ = nullptr;
};

// RAII Wrapper for FFMpeg type AVStream
class AVStreamWrapper {
public:
    void wrap(AVStream* avStream)
    {
        close();
        wrapped_ = avStream;
    }

    void open(AVCodec* avCodec)
    {
        auto ret = avcodec_open2(wrapped_->codec, avCodec, nullptr);
        checkReturnCode(ret, "avcodec_open2");
    }

    void close()
    {
        if (wrapped_) {
            avcodec_close(wrapped_->codec);
            wrapped_ = nullptr;
        }
    }

    AVStream* get() { return wrapped_; }
    AVStream* operator->() { return wrapped_; }

    ~AVStreamWrapper() { close(); }

private:
    AVStream* wrapped_ = nullptr;
};


// RAII Wrapper for FFMpeg type AVPacket
class AVPacketWrapper {
public:
    AVPacketWrapper()
    {
        av_init_packet(&wrapped_);
    }

    ~AVPacketWrapper()
    {
        av_free_packet(&wrapped_);
    }

    AVPacket& get() { return wrapped_; }

private:
    AVPacket wrapped_;
};

// RAII Wrapper for FFMpeg type AVFrame
class AVFrameWrapper {
public:
    AVFrameWrapper() : wrapped_(av_frame_alloc()) {}

    ~AVFrameWrapper() { av_frame_free(&wrapped_); }

    AVFrame* get() { return wrapped_; }

private:
    AVFrame* wrapped_;
};

// RAII Wrapper for FFMpeg type SwsContext
class SwsContextWrapper {
public:
    ~SwsContextWrapper() { sws_freeContext(wrapped_); }

    SwsContext* get() { return wrapped_; }

    SwsContextWrapper& operator=(SwsContext* ctx)
    {
        wrapped_ = ctx;
        return *this;
    }

private:
    SwsContext* wrapped_{nullptr};
};

} // maps::mrc::video
