#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/serialization.h>

#include <string_view>
#include <unordered_map>
#include <utility>

namespace maps::mrc::yt {

namespace {

struct YTPool {
    std::string tree;
    std::string pool;
};

const std::unordered_map<PoolType, std::vector<YTPool>>
POOL_TYPE_TO_CPU_POOLS {
    {
        PoolType::AdHoc,
        {
            {
                .tree = "yamaps",
                .pool = "maps-core-nmaps-mrc-ad_hoc"
            },
        }
    },
    {
        PoolType::Processing,
        {
            {
                .tree = "yamaps",
                .pool = "maps-core-nmaps-mrc-processing"
            },
        }
    },
};


const std::unordered_map<PoolType, std::vector<YTPool>>
POOL_TYPE_TO_GPU_POOLS {
    {
        PoolType::AdHoc,
        {
            {
                .tree = "gpu_geforce_1080ti",
                .pool = "maps-mrc-ad_hoc"
            },
            {
                .tree = "gpu_tesla_v100",
                .pool = "research_gpu"
            },
        }
    },
    {
        PoolType::Processing,
        {
            {
                .tree = "gpu_geforce_1080ti",
                .pool = "maps-mrc-processing"
            },
            {
                .tree = "gpu_tesla_v100",
                .pool = "research_gpu"
            },
        }
    },
};

void addSchedulingOptions(NYT::TNode* specPtr, const std::vector<YTPool>& pools) {
    NYT::TNode& spec = *specPtr;

    spec["scheduling_options_per_pool_tree"] = NYT::TNode::CreateMap();
    spec["pool_trees"] = NYT::TNode::CreateList();

    for (const auto& [tree, pool] : pools) {
        spec["scheduling_options_per_pool_tree"](
            TString(tree), NYT::TNode::CreateMap()
                ("pool", TString(pool))
        );
        spec["pool_trees"].Add(NYT::TNode(TString(tree)));
    }
}

} // namespace

void addCpuSchedulingOptions(NYT::TNode* specPtr, std::optional<PoolType> poolType) {
    if (!poolType.has_value()) {
        return;
    }

    addSchedulingOptions(specPtr, POOL_TYPE_TO_CPU_POOLS.at(poolType.value()));
}

void addGpuSchedulingOptions(NYT::TNode* specPtr, std::optional<PoolType> poolType) {
    if (!poolType.has_value()) {
        return;
    }

    addSchedulingOptions(specPtr, POOL_TYPE_TO_GPU_POOLS.at(poolType.value()));
}

NYT::TNode baseCpuOperationSpec(std::string_view title, std::optional<PoolType> poolType)
{
    NYT::TNode result = NYT::TNode::CreateMap();

    result["title"] = TString(title);
    addCpuSchedulingOptions(&result, poolType);

    return result;
}

NYT::TNode baseGpuOperationSpec(std::string_view title, std::optional<PoolType> poolType)
{
    NYT::TNode result = NYT::TNode::CreateMap();

    result["title"] = TString(title);
    addGpuSchedulingOptions(&result, poolType);

    return result;
}

NYT::TNode baseWorkerSpec(const std::optional<std::string>& portoLayerPath)
{
    if (not portoLayerPath) {
        return NYT::TNode::CreateMap();
    }

    return NYT::TNode::CreateMap()
        ("layer_paths", NYT::TNode::CreateList()
            .Add(TString(*portoLayerPath))
        );
}

} // namespace maps::mrc::yt
