#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/type_traits.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/exif.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/image_box.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/sql_chemistry/include/column.h>

#include <library/cpp/type_info/type.h>
#include <mapreduce/yt/interface/common.h>

#include <chrono>
#include <cstddef>
#include <string>
#include <tuple>

namespace maps::mrc::yt {

template<class T>
NTi::TTypePtr getTypeOf();

namespace impl {

// Use trick with fake pointer argument to specialize the function template

/* Base types */

inline NTi::TTypePtr getTypeOf(const bool*) { return NTi::Bool(); }

inline NTi::TTypePtr getTypeOf(const int8_t*) { return NTi::Int8(); }
inline NTi::TTypePtr getTypeOf(const uint8_t*) { return NTi::Uint8(); }

inline NTi::TTypePtr getTypeOf(const int16_t*) { return NTi::Int16(); }
inline NTi::TTypePtr getTypeOf(const uint16_t*) { return NTi::Uint16(); }

inline NTi::TTypePtr getTypeOf(const int32_t*) { return NTi::Int32(); }
inline NTi::TTypePtr getTypeOf(const uint32_t*) { return NTi::Uint32(); }

inline NTi::TTypePtr getTypeOf(const int64_t*) { return NTi::Int64(); }
inline NTi::TTypePtr getTypeOf(const uint64_t*) { return NTi::Uint64(); }

inline NTi::TTypePtr getTypeOf(const float*) { return NTi::Double(); }
inline NTi::TTypePtr getTypeOf(const double*) { return NTi::Double(); }

inline NTi::TTypePtr getTypeOf(const std::string*) { return NTi::String(); }

/* Complex types */

inline NTi::TTypePtr getTypeOf(const json::Value*) { return NTi::String(); }

template<class Clock, class Duration>
NTi::TTypePtr getTypeOf(const std::chrono::time_point<Clock, Duration>*)
{
    return yt::getTypeOf<uint64_t>();
}

template<class Rep, class Period>
NTi::TTypePtr getTypeOf(const std::chrono::duration<Rep, Period>*)
{
    return yt::getTypeOf<int64_t>();
}


inline NTi::TTypePtr getTypeOf(const common::ImageOrientation*) { return yt::getTypeOf<int16_t>(); }

inline NTi::TTypePtr getTypeOf(const common::ImageBox*) { return NTi::List(yt::getTypeOf<uint64_t>()); }

inline NTi::TTypePtr getTypeOf(const geolib3::Point2*)
{
    const auto type = yt::getTypeOf<double>();
    return NTi::Struct({{"x", type}, {"y", type}});
}

template<typename T>
inline typename std::enable_if<std::is_enum_v<T>, NTi::TTypePtr>::type getTypeOf(const T*)
{
    return NTi::String();
}

template<typename T>
inline NTi::TTypePtr getTypeOf(const std::optional<T>*) {
    return NTi::Optional(yt::getTypeOf<T>());
}

template<typename T>
inline NTi::TTypePtr getTypeOf(const std::vector<T>*) { return NTi::List(yt::getTypeOf<T>()); }

/* Table column */

template<typename T>
inline NTi::TTypePtr getTypeOf(const sql_chemistry::Nullable<T>*) { return NTi::Optional(yt::getTypeOf<T>()); }

template<typename T>
inline NTi::TTypePtr getTypeOf(const sql_chemistry::Column<T>*) { return yt::getTypeOf<T>(); }

/* Serializabel object */

template<typename T>
inline typename std::enable_if<isTableSerializabel<T>, NTi::TTypePtr>::type getTypeOf(const T*)
{
    using Table = typename db::TableTraits<T>::Table;

    std::vector<NTi::TStructType::TOwnedMember> members;
    introspection::genericForEach(
        Table::columns_(),
        [&members](const auto& column) {
            members.emplace_back(TString(column.name()), impl::getTypeOf(&column));
        }
    );

    return NTi::Struct(members);
}

template<class ...Args>
inline auto collectTypeOf(const std::tuple<Args...>*)
{
    return std::make_tuple(
        yt::getTypeOf<typename std::remove_cvref<Args>::type>()...
    );
}

template<class T>
inline auto collectTypeOf() { return collectTypeOf(static_cast<const T*>(nullptr)); }

template<typename T>
inline typename std::enable_if<isSerializabel<T>, NTi::TTypePtr>::type getTypeOf(const T*)
{
    std::vector<NTi::TStructType::TOwnedMember> members;
    introspection::tupleForEach(
        T::columns(),
        impl::collectTypeOf<introspection::introspected_tuple_t<T>>(),
        [&members](const auto& name, const auto& type) {
            members.emplace_back(name, type);
        }
    );

    return NTi::Struct(members);
}

} // namespace impl

template<class T>
inline NTi::TTypePtr getTypeOf() { return impl::getTypeOf(static_cast<const T*>(nullptr)); }

inline NYT::TColumnSchema makeSchemaColumn(std::string_view name, const NTi::TTypePtr& type)
{
    return NYT::TColumnSchema().Name(TString(name)).TypeV3(type);
}

template<class T>
inline NYT::TColumnSchema makeSchemaColumn(std::string_view name)
{
    return makeSchemaColumn(name, getTypeOf<T>());
}

template<class Column>
inline NYT::TColumnSchema makeSchemaColumn(const Column& column)
{
    return makeSchemaColumn(column.name(), getTypeOf<Column>());
}

template<class T>
typename std::enable_if<isTableSerializabel<T>, NYT::TTableSchema>::type getSchemaOf()
{
    using Table = typename db::TableTraits<T>::Table;

    NYT::TTableSchema schema;
    introspection::genericForEach(
        Table::columns_(),
        [&schema](const auto& column) {
            schema.AddColumn(makeSchemaColumn(column));
        }
    );

    return schema;
}

template<class T>
typename std::enable_if<isSerializabel<T>, NYT::TTableSchema>::type getSchemaOf()
{
    NYT::TTableSchema schema;

    introspection::tupleForEach(
        T::columns(),
        impl::collectTypeOf<introspection::introspected_tuple_t<T>>(),
        [&schema](const auto& name, const auto& type) {
            schema.AddColumn(makeSchemaColumn(name, type));
        }
    );

    return schema;
}

} // namespace maps::mrc::yt
