#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/serialization_access.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/type_traits.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/exif.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/image_box.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/types.h>

#include <maps/libs/json/include/value.h>

#include <maps/libs/geolib/include/heading.h>
#include <maps/libs/geolib/include/linear_ring.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/polyline.h>

#include <yandex/maps/mrc/traffic_signs/signs.h>

#include <mapreduce/yt/interface/node.h>

#include <opencv2/opencv.hpp>

#include <chrono>
#include <optional>
#include <sstream>
#include <string>
#include <string_view>
#include <vector>

namespace maps::mrc::yt {

/* Base types */

template<class T>
inline typename std::enable_if<std::is_arithmetic_v<T>, NYT::TNode>::type serialize(T value)
{
    return NYT::TNode(value);
}

inline NYT::TNode serialize(std::string_view value) { return TStringBuf(value); }

inline NYT::TNode serialize(const std::string& value) { return TString(value); }

/* Complex types */

template<class T>
typename std::enable_if<std::is_enum_v<T>, NYT::TNode>::type serialize(T value)
{
    std::ostringstream out;
    out << value;
    return serialize(out.str());
}

NYT::TNode serialize(const json::Value& value);

NYT::TNode serialize(const common::ImageBox& box);

inline NYT::TNode serialize(const common::ImageOrientation& orientation)
{
    return serialize(static_cast<int>(orientation));
}

inline NYT::TNode serialize(geolib3::Heading heading)
{
    return serialize(heading.value());
}

NYT::TNode serialize(const cv::Point2f& point);

template<class Clock, class Duration>
NYT::TNode serialize(std::chrono::time_point<Clock, Duration> value)
{
    return serialize(static_cast<uint64_t>(value.time_since_epoch().count()));
}

template<class Rep, class Period>
NYT::TNode serialize(std::chrono::duration<Rep, Period> value)
{
    return serialize(static_cast<int64_t>(value.count()));
}


inline NYT::TNode serialize(const Bytes& bytes)
{
    return TString(reinterpret_cast<const TString::TChar*>(bytes.data()), bytes.size());
}

NYT::TNode serialize(const geolib3::Point2& point);

NYT::TNode serialize(const geolib3::Polyline2& polyline);

NYT::TNode serialize(const geolib3::LinearRing2& lineRing);

NYT::TNode serialize(const geolib3::Polygon2& polygon);

/* Compound types (declaration) */

template<class T>
NYT::TNode serialize(const std::optional<T>& value);

template<class T>
NYT::TNode serialize(const std::vector<T>& values);

/* Serializabel types */

template<class E>
typename std::enable_if<isTableSerializabel<E>, NYT::TNode>::type serialize(const E& entity)
{
    using Table = typename db::TableTraits<E>::Table;

    auto node = NYT::TNode::CreateMap();
    introspection::tupleForEach(
        Table::columns_(),
        sql_chemistry::GatewayAccess<E>::fields(entity),
        [&node](const auto& column, const auto& field) {
            node(TString(column.name()), yt::serialize(field));
        }
    );

    return node;
}

template<class E>
typename std::enable_if<isSerializabel<E>, NYT::TNode>::type serialize(const E& entity)
{
    auto node = NYT::TNode::CreateMap();
    introspection::tupleForEach(
        E::columns(),
        SerializationAccess<E>::fields(entity),
        [&node](const auto& column, const auto& field) {
            node(column, yt::serialize(field));
        }
    );

    return node;
}

template<typename T>
NYT::TNode serialize(const FeatureWithImage<T>& value)
{
    auto result = serialize(value.feature);

    auto image = NYT::TNode::CreateEntity();
    if (!value.image.empty()) {
        Bytes bytes;
        REQUIRE(cv::imencode(".jpg", value.image, bytes), "Impossible serialize image!");
        image = serialize(bytes);
    }

    result("image", image);

    return result;
}

/* Compound types (definition) */

template<class T>
NYT::TNode serialize(const std::optional<T>& value)
{
    return value ? serialize(*value) : NYT::TNode::CreateEntity();
}

template<class T>
NYT::TNode serialize(const std::vector<T>& values)
{
    auto result = NYT::TNode::CreateList();

    for (const auto& value: values) {
        result.Add(serialize(value));
    }

    return result;
}

template<typename T>
T deserialize(const NYT::TNode& node);

namespace impl {

// Use trick with fake pointer argument to specialize the function template

/* Base types */

inline bool deserialize(const NYT::TNode& node, const bool*) { return node.AsBool(); }

template<typename T>
inline typename std::enable_if<std::is_integral<T>::value, T>::type deserialize(const NYT::TNode& node, const T*)
{
    if constexpr(std::is_signed<T>::value) {
        return node.IntCast<int64_t>();
    } else {
        return node.IntCast<uint64_t>();
    }
}

template<typename T>
inline typename std::enable_if<std::is_floating_point<T>::value, T>::type deserialize(const NYT::TNode& node, const T*)
{
    return node.AsDouble();
}

inline std::string deserialize(const NYT::TNode& node, const std::string*) { return node.AsString(); }

/* Complex types */

template<class T>
typename std::enable_if<std::is_enum_v<T>, T>::type deserialize(const NYT::TNode& node, const T*)
{
    std::istringstream in(node.AsString());

    T value;
    in >> value;

    REQUIRE(in, "Failed to parse value '" << node.AsString() << "'");

    return value;
}

// Specialize, no operator >>
traffic_signs::TrafficSign deserialize(const NYT::TNode& node, const traffic_signs::TrafficSign*);

geolib3::Heading deserialize(const NYT::TNode& node, const geolib3::Heading*);

template<class Clock, class Duration>
std::chrono::time_point<Clock, Duration>
deserialize(const NYT::TNode& node, const std::chrono::time_point<Clock, Duration>*)
{
    return std::chrono::time_point<Clock, Duration>(
        Duration(yt::deserialize<uint64_t>(node))
    );
}

template<class Rep, class Period>
std::chrono::duration<Rep, Period>
deserialize(const NYT::TNode& node, const std::chrono::duration<Rep, Period>*)
{
    return std::chrono::duration<Rep, Period>(yt::deserialize<int64_t>(node));
}


Bytes deserialize(const NYT::TNode& node, const Bytes*);

json::Value deserialize(const NYT::TNode& node, const json::Value*);

common::ImageOrientation deserialize(const NYT::TNode& node, const common::ImageOrientation*);

common::ImageBox deserialize(const NYT::TNode& node, const common::ImageBox*);

geolib3::Point2 deserialize(const NYT::TNode& node, const geolib3::Point2*);

geolib3::Polyline2 deserialize(const NYT::TNode& node, const geolib3::Polyline2*);

geolib3::LinearRing2 deserialize(const NYT::TNode& node, const geolib3::LinearRing2*);

geolib3::Polygon2 deserialize(const NYT::TNode& node, const geolib3::Polygon2*);

cv::Point2f deserialize(const NYT::TNode& node, const cv::Point2f*);

/* Compound types (declaration) */

template<typename T>
std::optional<T> deserialize(const NYT::TNode& node, const std::optional<T>*);

template<typename T>
std::vector<T> deserialize(const NYT::TNode& node, const std::vector<T>*);

/* Serializabel types */

template<class E>
typename std::enable_if<isTableSerializabel<E>, E>::type deserialize(const NYT::TNode& node, const E*)
{
    using Table = typename db::TableTraits<E>::Table;

    auto entity = sql_chemistry::GatewayAccess<E>::construct();

    introspection::tupleForEach(
        Table::columns_(),
        sql_chemistry::GatewayAccess<E>::fields(entity),
        [&node](const auto& column, auto& field) {
            using T = typename std::decay<decltype(field)>::type;

            TStringBuf name(column.name());
            field = yt::deserialize<T>(node[name]);
        }
    );

    return entity;
}

template<class E>
typename std::enable_if<isSerializabel<E>, E>::type deserialize(const NYT::TNode& node, const E*)
{
    auto entity = yt::SerializationAccess<E>::construct();

    introspection::tupleForEach(
        E::columns(),
        yt::SerializationAccess<E>::fields(entity),
        [&node](const auto& column, auto& field) {
            using T = typename std::decay<decltype(field)>::type;
            field = yt::deserialize<T>(node[column]);
        }
    );

    return entity;
}

template<typename T>
FeatureWithImage<T> deserialize(const NYT::TNode& node, const FeatureWithImage<T>*)
{
    auto image = [](const auto& node) {
        if (node.IsNull()) {
            return cv::Mat();
        }

        return common::decodeImage(node.AsString());
    };

    return {yt::deserialize<T>(node), image(node["image"])};
}

/* Compound types (definition) */

template<typename T>
std::optional<T> deserialize(const NYT::TNode& node, const std::optional<T>*)
{
    if (node.IsUndefined() || node.IsNull()) {
        return std::nullopt;
    }

    return yt::deserialize<T>(node);
}

template<typename T>
std::vector<T> deserialize(const NYT::TNode& node, const std::vector<T>*)
{
    std::vector<T> result;

    for (size_t i = 0; i < node.Size(); ++i) {
        result.push_back(yt::deserialize<T>(node[i]));
    }

    return result;
}

} // namespace impl

template<typename T>
T deserialize(const NYT::TNode& node)
{
    return impl::deserialize(node, static_cast<const T*>(nullptr));
}

} // namespace maps::mrc::yt
