#include "context.h"
#include "tools.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/for_each_passage.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/mds_file_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/track_point_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/assignment_object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>
#include <maps/libs/geolib/include/distance.h>

#include <algorithm>
#include <numeric>
#include <sstream>

namespace maps {
namespace mrc {
namespace img_qa {

Context::Context(const common::Config& cfg,
                 const auth::TvmtoolSettings& tvm)
    : poolHolder_(cfg.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                                     maps::mrc::common::LONG_READ_POOL_ID))
    , mds_(cfg.makeMdsClient())
    , publicMds_(cfg.makePublicMdsClient())
    , mrcBrowserUrl_(cfg.externals().mrcBrowserUrl())
    , blackboxClient_(tvm)
{
}

pgpool3::TransactionHandle Context::masterReadOnlyTransaction()
{
    return poolHolder_.pool().masterReadOnlyTransaction();
}

pgpool3::Pool& Context::pool()
{
    return poolHolder_.pool();
}

db::TIds Context::loadActiveAndCompletedAssignmentIds()
{
    using namespace db::ugc;
    auto txn = poolHolder_.pool().slaveTransaction();
    // Should be already unique.
    return AssignmentGateway{*txn}.loadIds(
        table::Assignment::status == AssignmentStatus::Active ||
        table::Assignment::status == AssignmentStatus::Completed);
}

bool Context::existsUnprocessedPhotos(db::TId assignmentId)
{
    auto txn = poolHolder_.pool().slaveTransaction();
    return db::FeatureGateway{*txn}.exists(
        db::table::Feature::assignmentId.equals(assignmentId) &&
        db::table::Feature::processedAt.isNull());
}

db::Features Context::loadProcessedPhotos(db::TId assignmentId)
{
    auto txn = poolHolder_.pool().slaveTransaction();
    return db::FeatureGateway{*txn}.load(
        db::table::Feature::assignmentId.equals(assignmentId)
        && db::table::Feature::processedAt.isNotNull());
}

db::Features
Context::loadProcessedPhotos(db::TId assignmentId,
                             db::CameraDeviation cameraDeviation)
{
    auto txn = poolHolder_.pool().slaveTransaction();
    if (cameraDeviation == db::CameraDeviation::Front) {
        return db::FeatureGateway{*txn}.load(
            db::table::Feature::assignmentId.equals(assignmentId)
            && db::table::Feature::processedAt.isNotNull()
            && (db::table::Feature::cameraDeviation.equals(
                    toIntegral(db::CameraDeviation::Front))
                || db::table::Feature::cameraDeviation.isNull()));
    }
    else {
        return db::FeatureGateway{*txn}.load(
            db::table::Feature::assignmentId.equals(assignmentId)
            && db::table::Feature::processedAt.isNotNull()
            && db::table::Feature::cameraDeviation.equals(
                   toIntegral(cameraDeviation)));
    }
}

db::FeatureQaTasks
Context::loadFeatureQaTasks(const FeatureBatch& photos)
{
    auto txn = poolHolder_.pool().slaveTransaction();

    db::TIds photoIds;
    for (const auto& photo : photos) {
        photoIds.push_back(photo.id());
    }

    return db::FeatureQaTaskGateway{*txn}.load(
        db::table::FeatureQaTask::featureId.in(photoIds));
}

db::TrackPoints
Context::loadAssignmentTrackPoints(db::TId assignmentId)
{
    auto txn = poolHolder_.pool().slaveTransaction();
    return db::TrackPointGateway{*txn}.load(
        db::table::TrackPoint::assignmentId.equals(assignmentId));
}

db::ugc::Assignment Context::loadAssignment(db::TId assignmentId)
{
    auto txn = poolHolder_.pool().slaveTransaction();
    db::ugc::AssignmentGateway gtw(*txn);
    return gtw.loadById(assignmentId);
}

db::ugc::Task Context::loadTask(db::TId taskId,
                                db::ugc::LoadNames loadNames,
                                db::ugc::LoadTargets loadTargets)
{
    auto txn = poolHolder_.pool().slaveTransaction();
    db::ugc::TaskGateway gtw{*txn};
    auto task = gtw.loadById(taskId);
    gtw.loadAlso(task, loadNames, loadTargets);
    return task;
}

std::optional<db::ugc::AssignmentReview>
Context::loadAssignmentReview(db::TId assignmentId,
                              db::CameraDeviation cameraDeviation)
{
    auto txn = poolHolder_.pool().slaveTransaction();
    db::ugc::AssignmentReviewGateway gtw(*txn);
    return gtw.tryLoadOne(
        db::ugc::table::AssignmentReview::assignmentId.equals(assignmentId)
        && db::ugc::table::AssignmentReview::cameraDeviation.equals(
               toIntegral(cameraDeviation)));
}

toloka::ImageQualityC12nTasks Context::getImageQualityC12nTasks(
    const db::FeatureQaTasks& photoTasks)
{
    if (photoTasks.empty()) {
        return {};
    }
    db::TIds taskIds;
    for (const auto& photoTask : photoTasks) {
        taskIds.push_back(photoTask.tolokaId());
    }

    auto txn = poolHolder_.pool().slaveTransaction();
    return toloka::getTasks<toloka::ImageQualityC12nTask>(*txn, taskIds);
}

std::vector<mds::Key>
Context::postToPublicMdsWithCorrectOrientation(FeatureBatch photos)
{
    using namespace std::literals;
    constexpr std::chrono::minutes TTL = 60min * 24 * 14;

    auto now = std::chrono::system_clock::now();
    auto epoch = now.time_since_epoch();
    std::ostringstream os;
    os << "assignment_inspector/"
       << std::chrono::duration_cast<std::chrono::milliseconds>(epoch).count()
       << "/";
    auto prefix = os.str();

    std::vector<mds::Key> result;
    for (const auto& photo : photos) {
        auto photoId = photo.id();
        auto url = makeReadUrl(photo);
        auto img = loadFeatureImage(photo);
        img = common::transformByImageOrientation(img, photo.orientation());
        auto resp = publicMds_.post(prefix + std::to_string(photoId), img, TTL);
        result.push_back(resp.key());
    }
    return result;
}

void Context::deleteMdsFiles(const std::vector<mds::Key>& mdsKeys)
{
    for (const mds::Key& mdsKey : mdsKeys) {
        publicMds_.del(mdsKey);
    }
}

toloka::ImageQualityC12nTasks
Context::createImageQualityC12nTasks(FeatureBatch photos)
{
    std::vector<mds::Key> mdsKeys;
    try {
        mdsKeys = postToPublicMdsWithCorrectOrientation(photos);
        toloka::ImageQualityC12nInputs inputs;
        for (const mds::Key& mdsKey : mdsKeys) {
            inputs.emplace_back(publicMds_.makeReadUrl(mdsKey));
        }

        auto txn = poolHolder_.pool().masterWriteableTransaction();
        auto tasks = toloka::createTasks<toloka::ImageQualityC12nTask>(
                *txn, db::toloka::Platform::Toloka, inputs);
        txn->commit();

        return tasks;
    } catch (const std::exception& e) {
        deleteMdsFiles(mdsKeys);
        throw;
    }
}

bool Context::isVisible(const db::ugc::AssignmentObject& assignmentObject)
{
    static const double LOCALITY_METERS = 100;
    auto bbox = geolib3::BoundingBox{
        geolib3::fastGeoShift(assignmentObject.geodeticPos(),
                              {-LOCALITY_METERS, -LOCALITY_METERS}),
        geolib3::fastGeoShift(assignmentObject.geodeticPos(),
                              {+LOCALITY_METERS, +LOCALITY_METERS})};
    return db::FeatureGateway{*masterReadOnlyTransaction()}.exists(
               db::table::Feature::shouldBePublished.is(true)
               && db::table::Feature::pos.intersects(
                      geolib3::convertGeodeticToMercator(bbox))
               && db::table::Feature::assignmentId
                      == assignmentObject.assignmentId());
}

db::AssignmentObjectFeedbackTasks
Context::createAssignmentObjectFeedbackTasks(const db::ugc::Assignment& assignment)
{
    // Use the master DB instance to avoid theoretically possible desync with
    // slaves.
    auto assignmentObjects
        = db::ugc::AssignmentObjectGateway{*masterReadOnlyTransaction()}
              .loadByAssignmentId(assignment.id());

    if (assignment.status() == db::ugc::AssignmentStatus::Active) {
        auto it = std::partition(assignmentObjects.begin(),
                                 assignmentObjects.end(),
                                 [this](const auto& assignmentObject) {
                                     return isVisible(assignmentObject);
                                 });
        assignmentObjects.erase(it, assignmentObjects.end());
    }

    db::TIds assignmentObjectIds;
    assignmentObjectIds.reserve(assignmentObjects.size());
    for (const auto& assignmentObject : assignmentObjects) {
        assignmentObjectIds.push_back(assignmentObject.objectId());
    }

    using FeedbackTaskTable = db::table::AssignmentObjectFeedbackTask;
    auto existingIds
        = db::AssignmentObjectFeedbackTaskGateway{*masterReadOnlyTransaction()}
              .loadIds(
                  FeedbackTaskTable::objectId.in(assignmentObjectIds),
                  sql_chemistry::orderBy(FeedbackTaskTable::objectId).asc());

    db::AssignmentObjectFeedbackTasks feedbackTasks;
    feedbackTasks.reserve(assignmentObjectIds.size() - existingIds.size());
    for (auto assignmentObjectId : assignmentObjectIds) {
        if (std::binary_search(existingIds.begin(), existingIds.end(),
                               assignmentObjectId)) {
            continue; // AssignmentObjectFeedbackTask already exists.
        }
        feedbackTasks.emplace_back(assignmentObjectId);
    }

    return feedbackTasks;
}

void Context::insert(db::FeatureQaTasks& photoTasks)
{
    auto txn = poolHolder_.pool().masterWriteableTransaction();
    db::FeatureQaTaskGateway{*txn}.insert(photoTasks);
    txn->commit();
}

void Context::update(db::Features& features)
{
    auto txn = poolHolder_.pool().masterWriteableTransaction();
    db::FeatureGateway{*txn}.update(features, db::UpdateFeatureTxn::Yes);
    txn->commit();
}

void Context::save(db::ugc::AssignmentReview& review)
{
    auto txn = poolHolder_.pool().masterWriteableTransaction();
    db::ugc::AssignmentReviewGateway gtw(*txn);
    gtw.upsert(review);
    txn->commit();
}

void Context::update(db::ugc::Assignment& assignment,
                     db::AssignmentObjectFeedbackTasks&& feedbackTasks)
{
    auto txn = poolHolder_.pool().masterWriteableTransaction();
    db::ugc::AssignmentGateway{*txn}.update(assignment);
    db::AssignmentObjectFeedbackTaskGateway{*txn}.insert(std::move(feedbackTasks));
    txn->commit();
}

std::string Context::makeReadUrl(const db::Feature& feature)
{
    return mds_.makeReadUrl(feature.mdsKey());
}

std::string Context::makeBrowserUrl(const db::Feature& feature)
{
    std::ostringstream path;
    path << "/feature/" << feature.id() << "/image";
    auto result = mrcBrowserUrl_;
    result.setPath(path.str());
    return result.toString();
}

std::string Context::loadUserLogin(db::TId assignmentId)
{
    auto assignedTo = loadAssignment(assignmentId).assignedTo();
    try {
        auto uid = boost::lexical_cast<blackbox_client::Uid>(assignedTo);
        auto uidToLoginMap = blackboxClient_.uidToLoginMap({uid});
        auto it = uidToLoginMap.find(uid);
        REQUIRE(it != uidToLoginMap.end(), "blackbox response is empty");
        return it->second;
    }
    catch (const std::exception& e) {
        WARN() << "driver " << assignedTo << " not found (" << e.what() << ")";
        return assignedTo;
    }
}

common::Blob Context::loadFeatureImage(const db::Feature& feature)
{
    return mds_.get(feature.mdsKey());
}


db::ugc::AssignmentObjects Context::loadAssignmentObjects(db::TId assignmentId)
{
    auto txn = poolHolder_.pool().slaveTransaction();
    return db::ugc::AssignmentObjectGateway{*txn}.loadByAssignmentId(assignmentId);
}

bool Context::isAssignmentUpdated(db::TId assignmentId)
{
    auto txn = poolHolder_.pool().slaveTransaction();
    auto reviews = db::ugc::AssignmentReviewGateway{*txn}.load(
        db::ugc::table::AssignmentReview::assignmentId.equals(assignmentId));
    size_t processedPhotos = std::accumulate(
        reviews.begin(), reviews.end(), 0, [](size_t result, auto& review) {
            return result + review.processedPhotos().value_or(0);
        });
    size_t processedPoints = std::accumulate(
        reviews.begin(), reviews.end(), 0, [](size_t result, auto& review) {
            return result + review.processedPoints().value_or(0);
        });
    size_t photos = db::FeatureGateway{*txn}.count(
        db::table::Feature::assignmentId.equals(assignmentId) &&
        db::table::Feature::processedAt.isNotNull());
    size_t points = db::TrackPointGateway{*txn}.count(
        db::table::TrackPoint::assignmentId.equals(assignmentId));
    return processedPhotos != photos
        || processedPoints != reviews.size() * points;
}

Strings Context::loadEmails(db::TId tasksGroupId)
{
    auto result = Strings{};
    auto tasksGroupEmails =
        db::ugc::TasksGroupEmailGateway{*poolHolder_.pool().slaveTransaction()}
            .load(db::ugc::table::TasksGroupEmail::tasksGroupId ==
                  tasksGroupId);
    for (const auto& tasksGroupEmail : tasksGroupEmails) {
        result.push_back(tasksGroupEmail.email());
    }
    return result;
}

} // img_qa
} // mrc
} // maps
