#include "geojson.h"

#include "tools.h"

#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/json/include/value.h>

#include <chrono>
#include <fstream>

namespace maps {
namespace mrc {
namespace img_qa {
namespace {

std::string
createGeojsonLinestringFeature(size_t& id,
                               const geolib3::Polyline2& linestring,
                               const LineStyle& params)
{
    json::Builder builder;
    builder << [&](json::ObjectBuilder b) {
        b["type"] = "Feature";
        b["id"] = id;
        b["geometry"] << geolib3::geojson(linestring);
        b["properties"] = [&](json::ObjectBuilder b) {
            b["description"] = params.description;
            b["stroke"] = params.color;
            b["stroke-width"] = params.width;
            b["stroke-opacity"] = params.opacity;
        };
    };
    id++;
    return builder.str();
}

std::vector<std::string>
getGeojsonPolylines(size_t& id,
                    const geolib3::PolylinesVector& lines,
                    const LineStyle& params)
{
    std::vector<std::string> result;
    result.reserve(lines.size());
    for (const auto& line : lines) {
        result.push_back(createGeojsonLinestringFeature(id, line, params));
    }
    return result;
}

} // anonymous namespace

void GeojsonBuilder::addTrack(geolib3::PolylinesVector lines,
                              LineStyle params)
{
    tracks.push_back({std::move(lines), std::move(params)});
}

void GeojsonBuilder::addAssignmentObjects(
    const db::ugc::AssignmentObjects& assignmentObjects_)
{
    assignmentObjects.insert(assignmentObjects.end(),
                             assignmentObjects_.begin(),
                             assignmentObjects_.end());
}

void GeojsonBuilder::writeToFile(const std::string& filename)
{
    std::ofstream outFile(filename);
    outFile << generateGeojsonString();
}

std::string GeojsonBuilder::generateGeojsonString()
{
    size_t id = 0;
    json::Builder builder;
    builder << [&](json::ObjectBuilder b) {
        b["type"] = "FeatureCollection";
        b["features"] << [&](json::ArrayBuilder b) {
            for (const auto& track : tracks) {
                std::vector<std::string> polylines
                    = getGeojsonPolylines(id, track.lines, track.params);
                for (std::string polyline : polylines) {
                    b << json::Verbatim(polyline);
                }
            }
            for (const auto& assignmentObject : assignmentObjects) {
                b << [&](json::ObjectBuilder b) {
                    b["type"] = "Feature";
                    b["id"] = id++;
                    b["geometry"]
                        << geolib3::geojson(assignmentObject.geodeticPos());
                    b["properties"] = [&](json::ObjectBuilder b) {
                        auto desc = boost::lexical_cast<std::string>(
                            assignmentObject.objectType());
                        if (assignmentObject.comment()) {
                            desc += ":" + *assignmentObject.comment();
                        }
                        b["description"] = desc;
                    };
                };
            }
        };
    };
    return builder.str();
}

} // img_qa
} // mrc
} // maps
