#include "mail.h"

#include "geojson.h"
#include "inspect_quality.h"
#include "tools.h"

#include <maps/libs/concurrent/include/scoped_guard.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/process/include/process.h>
#include <util/generic/serialized_enum.h>

#include <Poco/Net/FilePartSource.h>
#include <Poco/Net/MailMessage.h>
#include <Poco/Net/MailRecipient.h>
#include <Poco/Net/StringPartSource.h>

#include <boost/filesystem.hpp>

#include <chrono>
#include <cmath>
#include <ctime>
#include <string>

namespace bf = boost::filesystem;

namespace maps {
namespace mrc {
namespace img_qa {
namespace {

const int NUMBER_OF_EXAMPLE_PHOTOS = 20;

void sendEmail(const std::string& from,
               const std::string& to,
               const std::string& subject,
               const std::string& text,
               const std::vector<bf::path>& attachmentFiles)
{
    Poco::Net::MailMessage msg;
    msg.setSender(from);
    msg.addRecipient(Poco::Net::MailRecipient(
            Poco::Net::MailRecipient::PRIMARY_RECIPIENT, to));

    msg.setSubject(msg.encodeWord(subject, "UTF-8"));
    /// The MailMessage takes ownership of the PartSource
    msg.addContent(new Poco::Net::StringPartSource(text, "text/html"));

    for (const bf::path& path : attachmentFiles) {
        /// The MailMessage takes ownership of the PartSource
        msg.addAttachment(
            path.filename().string(),
            new Poco::Net::FilePartSource(path.string(), "application/json"));
    }

    auto process = process::runInteractive(
        process::Command({"/usr/lib/sendmail", "-t"}));
    if (process.valid() && !process.finished()) {
        msg.write(process.inputStream());
        process.closeInputStream();
        auto status = process.syncWait();
        if (!status.exited() || status.exitStatus() != 0) {
            throw maps::Exception() << "Can't run sendmail: " << status;
        }
    }
    else {
        throw maps::Exception("Can't run sendmail");
    }
}

std::pair<std::string, std::string> getDrivingTimeRange(Context& ctx,
                                                        db::TId assignmentId)
{
    std::vector<chrono::TimePoint> timePoints;
    for (const auto& trackPoint :
         ctx.loadAssignmentTrackPoints(assignmentId)) {
        timePoints.push_back(trackPoint.timestamp());
    }
    for (const auto& photo : ctx.loadProcessedPhotos(assignmentId)) {
        timePoints.push_back(photo.timestamp());
    }

    if (timePoints.empty()) {
        return {};
    }

    auto minMaxTime
        = std::minmax_element(timePoints.begin(), timePoints.end());

    using clock = std::chrono::system_clock;
    std::time_t minTime = clock::to_time_t(
            std::chrono::time_point_cast<clock::time_point::duration>(
                *minMaxTime.first));
    std::time_t maxTime = clock::to_time_t(
            std::chrono::time_point_cast<clock::time_point::duration>(
                *minMaxTime.second));

    std::string minTimeStr = std::ctime(&minTime);
    std::string maxTimeStr = std::ctime(&maxTime);
    return std::make_pair(minTimeStr, maxTimeStr);
}

struct Photo {
    std::string url;
    double quality;
};

struct Photos {
    size_t numberOfPhotos;
    size_t goodPhotosPercentage;
    std::vector<Photo> randomGoodPhotos;
    std::vector<Photo> randomBadPhotos;
};

Photos getPhotos(Context& ctx, db::TId assignmentId)
{
    auto assignmentPhotos = ctx.loadProcessedPhotos(assignmentId);
    Photos result;
    result.numberOfPhotos = assignmentPhotos.size();
    if (result.numberOfPhotos == 0) {
        result.goodPhotosPercentage = 0;
        return result;
    }
    std::random_shuffle(assignmentPhotos.begin(), assignmentPhotos.end());
    auto photoToStatusMap = loadPhotoToTolokaStatusMap(
        ctx, {assignmentPhotos.begin(), assignmentPhotos.end()});
    int numberOfGoodPhotos = 0;

    for (const auto& feature : assignmentPhotos) {
        if (isPhotoGood(feature, photoToStatusMap)) {
            numberOfGoodPhotos++;
            if (result.randomGoodPhotos.size() < NUMBER_OF_EXAMPLE_PHOTOS) {
                result.randomGoodPhotos.push_back(
                    Photo{feature.isPublished() ? ctx.makeBrowserUrl(feature)
                                                : ctx.makeReadUrl(feature),
                          feature.quality()});
            }
        }
        else {
            if (result.randomBadPhotos.size() < NUMBER_OF_EXAMPLE_PHOTOS) {
                result.randomBadPhotos.push_back(
                    Photo{feature.isPublished() ? ctx.makeBrowserUrl(feature)
                                                : ctx.makeReadUrl(feature),
                          feature.quality()});
            }
        }
    }
    result.goodPhotosPercentage
        = round((100.0 * numberOfGoodPhotos) / result.numberOfPhotos);
    return result;
}

std::string
composeEmailText(Context& ctx,
                 db::TId assignmentId,
                 const std::vector<CoverageStatistics>& allCoverageStats)
{
    auto drivingTimeRange = getDrivingTimeRange(ctx, assignmentId);
    Photos photos = getPhotos(ctx, assignmentId);

    std::ostringstream text;
    text << "<html><body>"
         << "Driver: " << ctx.loadUserLogin(assignmentId)
         << "<br>"
         << "First gps point time: " << drivingTimeRange.first << "<br>"
         << "Last gps point time: " << drivingTimeRange.second << "<br>"
         << "Task length: "
         << ctx.loadTask(ctx.loadAssignment(assignmentId).taskId(),
                         db::ugc::LoadNames::No,
                         db::ugc::LoadTargets::No).distanceInMeters() / 1000.0
         << " km";

    for (const auto& coverageStats: allCoverageStats) {
        text << "<br>"
             << "Camera deviation: "
             << ToString(coverageStats.cameraDeviation)
             << "<br>"
             << "Coverage: "
             << coverageStats.allPhotosCoverage.coverageFraction
             << "<br>"
             << "Good photos coverage: "
             << coverageStats.goodPhotosCoverage.coverageFraction << "<br>"
             << "Ride time: " << coverageStats.trackDuration.count() / 60.0
             << " min<br>"
             << "Ride track length: "
             << coverageStats.trackDistanceInMeters / 1000.0
             << " km";
    }

    text << "<br>"
         << "Number of photos: " << photos.numberOfPhotos << " ("
         << photos.goodPhotosPercentage << "% good)";

    text << "<br><br>"
         << "Random good photos:<br><br>";
    for (Photo photo : photos.randomGoodPhotos) {
        text << photo.quality << "<br>"
             << "<img src=\"" + photo.url + "\"><br>";
    }

    text << "<br><br>"
         << "Random bad photos:<br><br>";
    for (Photo photo : photos.randomBadPhotos) {
        text << photo.quality << "<br>"
             << "<img src=\"" + photo.url + "\"><br>";
    }
    text << "</body></html>";
    return text.str();
}

std::vector<bf::path> generateAttachmentFiles(
    Context& ctx,
    db::TId assignmentId,
    const std::vector<CoverageStatistics>& allCoverageStats)
{
    std::vector<bf::path> attachments;

    bf::path tempDir = bf::temp_directory_path();
    std::string assignmentSuffix
        = "assignment_" + std::to_string(assignmentId);
    auto assignmentObjects = ctx.loadAssignmentObjects(assignmentId);
    for (const auto& coverageStats: allCoverageStats) {
        std::string deviationSuffix
            = assignmentSuffix + "_"
              + ToString(coverageStats.cameraDeviation);
        {
            GeojsonBuilder builder;
            bf::path filepath
                = tempDir / bf::path("all_" + deviationSuffix + ".geojson");
            {
                LineStyle params;
                params.description = "ride track";
                params.color = "#0000ff";
                params.width = 2;
                params.opacity = 1.;
                auto trackPoints = ctx.loadAssignmentTrackPoints(assignmentId);
                sortUniqueByTime(trackPoints);
                builder.addTrack(toPolylines(getSegments(makePath(trackPoints))),
                                 params);
            }
            {
                LineStyle params;
                params.description = "covered task";
                params.color = "#00ff00";
                params.width = 3;
                params.opacity = 0.9;
                builder.addTrack(
                    toEquidistantPolylines(
                        coverageStats.allPhotosCoverage.coveredParts),
                    params);
            }
            {
                LineStyle params;
                params.description = "uncovered task";
                params.color = "#ff0000";
                params.width = 3;
                params.opacity = 0.9;
                builder.addTrack(
                    toEquidistantPolylines(
                        coverageStats.allPhotosCoverage.uncoveredParts),
                    params);
            }
            builder.addAssignmentObjects(assignmentObjects);
            builder.writeToFile(filepath.string());
            attachments.push_back(filepath);
        }
        {
            GeojsonBuilder builder;
            bf::path filepath
                = tempDir
                  / bf::path("covered_" + deviationSuffix + ".geojson");
            {
                LineStyle params;
                params.description = "covered task";
                params.color = "#00ff00";
                params.width = 3;
                params.opacity = 0.9;
                builder.addTrack(
                    toEquidistantPolylines(
                        coverageStats.allPhotosCoverage.coveredParts),
                    params);
            }
            builder.writeToFile(filepath.string());
            attachments.push_back(filepath);
        }
        {
            GeojsonBuilder builder;
            bf::path filepath
                = tempDir
                  / bf::path("uncovered_" + deviationSuffix + ".geojson");
            {
                LineStyle params;
                params.description = "uncovered task";
                params.color = "#ff0000";
                params.width = 3;
                params.opacity = 0.9;
                builder.addTrack(
                    toEquidistantPolylines(
                        coverageStats.allPhotosCoverage.uncoveredParts),
                    params);
            }
            builder.writeToFile(filepath.string());
            attachments.push_back(filepath);
        }
        {
            GeojsonBuilder builder;
            bf::path filepath
                = tempDir
                  / bf::path("good_images_" + deviationSuffix + ".geojson");
            {
                LineStyle params;
                params.description = "good images";
                params.color = "#00ff00";
                params.width = 3;
                params.opacity = 0.9;
                builder.addTrack(
                    toEquidistantPolylines(
                        coverageStats.goodPhotosCoverage.coveredParts),
                    params);
            }
            builder.writeToFile(filepath.string());
            attachments.push_back(filepath);
        }
        {
            GeojsonBuilder builder;
            bf::path filepath
                = tempDir / bf::path("uncovered_and_bad_images_"
                                     + deviationSuffix + ".geojson");
            {
                LineStyle params;
                params.description = "uncovered and bad images";
                params.color = "#ff0000";
                params.width = 3;
                params.opacity = 0.9;
                builder.addTrack(
                    toEquidistantPolylines(
                        coverageStats.goodPhotosCoverage.uncoveredParts),
                    params);
            }
            builder.writeToFile(filepath.string());
            attachments.push_back(filepath);
        }
    }
    if (!assignmentObjects.empty()) {
        GeojsonBuilder builder;
        bf::path filepath
            = tempDir
              / bf::path("object_points_" + assignmentSuffix + ".geojson");
        builder.addAssignmentObjects(assignmentObjects);
        builder.writeToFile(filepath.string());
        attachments.push_back(filepath);
    }
    return attachments;
}

} // anonymous namespace

void sendEmailReport(
    const Strings& emails,
    Context& ctx,
    db::TId assignmentId,
    const std::vector<CoverageStatistics>& allCoverageStats)
{
    INFO() << "sending email report for assignment " << assignmentId;
    std::string emailText
        = composeEmailText(ctx, assignmentId, allCoverageStats);
    std::string taskName
        = ctx.loadTask(ctx.loadAssignment(assignmentId).taskId(),
                       db::ugc::LoadNames::Yes, db::ugc::LoadTargets::No)
              .names()
              .begin()
              ->second;
    std::string subject = "assignment" + std::to_string(assignmentId) + " "
                          + taskName;

    std::vector<bf::path> attachments
        = generateAttachmentFiles(ctx, assignmentId, allCoverageStats);
    concurrent::ScopedGuard guard([&] {
        for (bf::path attachmentPath : attachments) {
            bf::remove(attachmentPath);
        }
    });
    for (const auto& email : emails) {
        try {
            sendEmail(email, email, subject, emailText, attachments);
        }
        catch (const std::exception& e) {
            ERROR() << "Can't send email to " << email << ": " << e.what();
        }
    }
}

} // img_qa
} // mrc
} // maps
