#include "process_assignment.h"

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/environment.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/threadpool_wrapper.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>

namespace db = maps::mrc::db;
namespace img_qa = maps::mrc::img_qa;

int main(int argc, char* argv[]) try {
    maps::cmdline::Parser parser(
        "Tool sends photos of completed tasks to Toloka for quality "
        "estimation. It stops if enough defective photos or reached the "
        "limit. In addition, on the first call, it calculates the positions "
        "of the photos and the coverage fraction.");
    auto syslog = parser.string("syslog-tag")
                      .help("redirect log output to syslog with given tag");
    auto configPath = parser.string("config").help("path to configuration");
    auto secretVersion = parser.string("secret-version")
        .help("version for secrets from yav.yandex-team.ru");
    auto assignmentId
        = parser.num("assignment").help("assignment ID to inspect");
    auto modifyData = parser.flag("force").defaultValue(false).help(
        "create Toloka tasks and modify Postgres database if necessary");
    auto disableEmailReports
        = parser.flag("disable-email-reports")
              .defaultValue(false)
              .help("disable email reporting (by default image inspector "
                    "sends email report"
                    " for each assigment if assignment results are changed)");
    parser.parse(argc, argv);

    if (syslog.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslog));
    }

    auto cfg =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, configPath);
    switch (maps::common::getYandexEnvironment()) {
        case maps::common::Environment::Stable:
        case maps::common::Environment::Testing:
            INFO() << "Enable TVM suppport";
            cfg.enableTvmClient();
            break;
        default:
            break;
    }

    INFO() << "Starting";

    img_qa::Context ctx{cfg};

    maps::pgp3utils::PgAdvisoryXactMutex mutex(ctx.pool(),
        static_cast<int64_t>(maps::mrc::common::LockId::ImageInspector));
    if (modifyData && !mutex.try_lock()) {
        INFO() << "Another process is ongoing";
        return EXIT_SUCCESS;
    }

    db::TIds assignmentIds;
    if (assignmentId.defined()) {
        assignmentIds.push_back(assignmentId);
    } else {
        assignmentIds = ctx.loadActiveAndCompletedAssignmentIds();
    }

    INFO() << "There are " << assignmentIds.size() << " assignments to process";

    maps::mrc::common::ThreadpoolWrapper threadPool{
        4 /* threadsNumber */, 4 /* queueSize */};
    for (auto id : assignmentIds) {
        auto assignment = ctx.loadAssignment(id);
        if (db::ugc::AssignmentStatus::Active == assignment.status()
                && !ctx.isAssignmentUpdated(id))
        {
            INFO() << "Assignment " << id << " is skipped";
            continue;
        }
        INFO() << "Processing assignment: " << id;
        threadPool->add([&, assignment = std::move(assignment) ]() mutable {
            processAssignment(
                assignment, cfg, modifyData, disableEmailReports, ctx);
        });
    }
    threadPool->drain();
    threadPool.checkExceptions();

    INFO() << "Done";
    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
