#include "coverage.h"
#include "inspect_quality.h"
#include "mail.h"
#include "process_assignment.h"
#include "share_images.h"
#include "tools.h"

#include <maps/libs/log8/include/log8.h>

namespace maps::mrc::img_qa {
namespace {

/// to avoid case when photos arrive in DB later than the user's complete
const std::chrono::hours REVIEW_AGING_TO_ACCEPT{24};

}  // anonymous namespace

void processAssignment(db::ugc::Assignment& assignment,
                       const common::Config& cfg,
                       bool modifyData,
                       bool disableEmailReports,
                       Context& ctx)
{
    const auto now = currentTimePoint();
    bool mayAccept = !ctx.existsUnprocessedPhotos(assignment.id());
    bool shouldSendEmailNotification = false;
    std::vector<CoverageStatistics> allCoverageStats;

    auto task = ctx.loadTask(
        assignment.taskId(), db::ugc::LoadNames::No, db::ugc::LoadTargets::No);
    for (auto cameraDeviation : task.cameraDeviations()) {
        INFO() << "Camera deviation: " << toIntegral(cameraDeviation);
        auto review =
            ctx.loadAssignmentReview(assignment.id(), cameraDeviation);
        if (!review) {
            review = db::ugc::AssignmentReview(assignment.id());
            review->setCameraDeviation(cameraDeviation);
        }
        auto coverageStat =
            inspectCoverage(ctx, assignment.id(), cameraDeviation);
        INFO() << toString(coverageStat);
        allCoverageStats.push_back(coverageStat);

        if (areRideResultsChanged(*review, coverageStat)) {
            review->setActualizationDate(currentTimePoint());
            if (!disableEmailReports) {
                shouldSendEmailNotification = true;
            }
        }
        fillReview(coverageStat, *review);

        auto tolokaStat =
            inspectQuality(ctx,
                           assignment.id(),
                           cameraDeviation,
                           coverageStat.goodPhotosCoverage.uncoveredParts,
                           modifyData);
        INFO() << toString(tolokaStat);
        review->setTolokaStatus(status(tolokaStat));

        // legacy support
        if (!review->actualizationDate()) {
            review->setActualizationDate(currentTimePoint());
        }
        if (modifyData) {
            ctx.save(*review);
        }
        if (review->tolokaStatus() == db::ugc::TolokaStatus::InProgress ||
            (now - *review->actualizationDate()) < REVIEW_AGING_TO_ACCEPT) {
            mayAccept = false;
        }
    }
    if (shouldSendEmailNotification) {
        auto emails = task.tasksGroupId().has_value()
                          ? ctx.loadEmails(task.tasksGroupId().value())
                          : Strings{};
        emails.push_back(cfg.rideReportsEmailAddress());
        sendEmailReport(emails, ctx, assignment.id(), allCoverageStats);
    }
    shareImages(ctx, assignment.id(), modifyData);
    if (modifyData) {
        if (assignment.status() == db::ugc::AssignmentStatus::Completed &&
            assignment.submittedAt() &&
            (now - *assignment.submittedAt()) > REVIEW_AGING_TO_ACCEPT &&
            mayAccept) {
            INFO() << "Assignment " << assignment.id() << " is accepted";
            assignment.markAsAccepted();
        }
        ctx.update(assignment,
                   ctx.createAssignmentObjectFeedbackTasks(assignment));
    }
}

}  // namespace maps::mrc::img_qa