#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/auth/include/test_utils.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/json/include/prettify.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/assignment_object_feedback_task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/assignment_inspector/coverage.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/assignment_inspector/geojson.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/assignment_inspector/inspect_quality.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/assignment_inspector/process_assignment.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/assignment_inspector/share_images.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/assignment_inspector/tools.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/units_literals.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>
#include <yandex/maps/mrc/unittest/local_server.h>

#include <boost/format.hpp>
#include <boost/range/algorithm/count_if.hpp>
#include <boost/range/irange.hpp>

#include <fstream>

namespace maps {
namespace mrc {
namespace img_qa {
namespace tests {
namespace {

using namespace geolib3::literals;

const db::TId ASSIGNMENT_ID = 1;
const std::string CURRENT_DIR = ".";
constexpr db::CameraDeviation FRONT = db::CameraDeviation::Front;
constexpr db::CameraDeviation RIGHT = db::CameraDeviation::Right;

class TestFixture
    : public testing::Test,
      public unittest::WithUnittestConfig<unittest::DatabaseFixture,
                                          unittest::MdsStubFixture> {

    auth::TvmtoolRecipeHelper tvmtoolRecipeHelper_;

public:
    Context ctx;

    TestFixture()
        : tvmtoolRecipeHelper_("maps/libs/auth/tests/tvmtool.recipe.conf")
        , ctx(config(), tvmtoolRecipeHelper_.tvmtoolSettings())
    {
        static const std::string SQL_DATA_PATH
            = ArcadiaSourceRoot() + "/maps/wikimap/mapspro/services/mrc/"
                                    "long_tasks/assignment_inspector/tests/data.sql";
        auto txn = pool().masterWriteableTransaction();
        txn->exec(maps::common::readFileToString(SQL_DATA_PATH));
        txn->commit();

        auto mds = config().makeMdsClient();
        for (auto imageNum : boost::irange(1, 12)) {
            mds.post(std::to_string(imageNum),
                     common::getTestImage<std::string>());
        }
    }
};

db::ugc::AssignmentObjects makeAssignmentObjects()
{
    return {
        db::ugc::AssignmentObject{
            ASSIGNMENT_ID, chrono::parseSqlDateTime("2017-05-17 11:03:17+03"),
            geolib3::Point2(0, 1), db::ugc::AssignmentObjectType::Barrier},
        db::ugc::AssignmentObject{
            ASSIGNMENT_ID, chrono::parseSqlDateTime("2017-05-19 11:03:17+03"),
            geolib3::Point2(2, 4), db::ugc::AssignmentObjectType::Barrier}
            .setComment("comment")};
}

db::Feature makeFeature(const geolib3::Point2& pos,
                        const geolib3::Direction2& dir)
{
    return db::Feature{"source",
                       pos,
                       dir.heading(),
                       "2017-05-19 11:03:17+03",
                       mds::Key{"group", "key"},
                       db::Dataset::Agents}
        .setSize({6, 9})
        .setAutomaticShouldBePublished(true)
        .setIsPublished(true);
}

} // anonymous namespace

TEST_F(TestFixture, test_inspect_quality)
{
    auto coverageStat = inspectCoverage(ctx, ASSIGNMENT_ID, FRONT);
    auto allUncovered = coverageStat.allPhotosCoverage.uncoveredParts;
    auto goodUncovered = coverageStat.goodPhotosCoverage.uncoveredParts;

    /**
     * The same photo which was already included in the coverage should not
     * bring additional coverage (so should not be sent to toloka)
     */
    auto stat = inspectQuality(ctx, ASSIGNMENT_ID, FRONT, allUncovered,
                               /*createTolokaTasks*/ false);
    INFO() << toString(stat);
    EXPECT_TRUE(status(stat) == db::ugc::TolokaStatus::Accepted);
    EXPECT_EQ(stat.total, 6u);
    EXPECT_EQ(stat.tolokaAccept, 0u);
    EXPECT_EQ(stat.tolokaReject, 0u);
    EXPECT_EQ(stat.tolokaInProgress, 0u);
    EXPECT_EQ(stat.tolokaComing, 0u);

    // a bad photo is going to toloka
    stat = inspectQuality(ctx, ASSIGNMENT_ID, FRONT, goodUncovered,
                          /*createTolokaTasks*/ true);
    INFO() << toString(stat);
    EXPECT_TRUE(status(stat) == db::ugc::TolokaStatus::InProgress);
    EXPECT_EQ(stat.total, 6u);
    EXPECT_EQ(stat.tolokaAccept, 0u);
    EXPECT_EQ(stat.tolokaReject, 0u);
    EXPECT_EQ(stat.tolokaInProgress, 0u);
    EXPECT_EQ(stat.tolokaComing, 1u);

    // the bad photo is already in toloka
    stat = inspectQuality(ctx, ASSIGNMENT_ID, FRONT, goodUncovered,
                          /*createTolokaTasks*/ true);
    INFO() << toString(stat);
    EXPECT_TRUE(status(stat) == db::ugc::TolokaStatus::InProgress);
    EXPECT_EQ(stat.total, 6u);
    EXPECT_EQ(stat.tolokaAccept, 0u);
    EXPECT_EQ(stat.tolokaReject, 0u);
    EXPECT_EQ(stat.tolokaInProgress, 1u);
    EXPECT_EQ(stat.tolokaComing, 0u);

    // emulate toloka processes
    auto acceptPhotos = [&]() {
        auto txn = pool().masterWriteableTransaction();
        db::toloka::TaskGateway gtw{*txn};
        auto ids = gtw.loadIdsByType(
            db::toloka::Platform::Toloka,
            db::toloka::TaskType::ImageQualityClassification);
        for (auto id : ids) {
            auto task = gtw.loadById(id);
            task.setStatus(db::toloka::TaskStatus::Finished)
                .setOutputValues(R"({"state":"ok"})");
            gtw.updatex(task);
        }
        txn->commit();
    };
    acceptPhotos();

    // the bad photo passed toloka
    stat = inspectQuality(ctx, ASSIGNMENT_ID, FRONT, goodUncovered,
                          /*createTolokaTasks*/ true);
    INFO() << toString(stat);
    EXPECT_TRUE(status(stat) == db::ugc::TolokaStatus::Accepted);
    EXPECT_EQ(stat.total, 6u);
    EXPECT_EQ(stat.tolokaAccept, 1u);
    EXPECT_EQ(stat.tolokaReject, 0u);
    EXPECT_EQ(stat.tolokaInProgress, 0u);
    EXPECT_EQ(stat.tolokaComing, 0u);
}

TEST(tests, test_coverage_without_db_simple)
{
    using namespace geolib3;
    const double COVERAGE = 0.66;
    constexpr double TOLERANCE_PERCENT = 3.;
    const Point2 origin{37.25, 55.1};

    Segments task = {{origin, fastGeoShift(origin, {0., 90 /*meters*/})}};
    db::Features photos{
        makeFeature(origin, geolib3::Direction2(89_deg)),
        makeFeature(fastGeoShift(origin, {EPS, 30 /*meters*/}), geolib3::Direction2(91_deg)),
        // out of visibility
        makeFeature(fastGeoShift(origin, {-EPS, 180 /*meters*/}), geolib3::Direction2(91_deg))
    };
    auto coverage = getCoverage(task, photos);
    EXPECT_NEAR(coverage.coverageFraction, COVERAGE, TOLERANCE_PERCENT);
}

TEST(tests, test_coverage_without_db_extra)
{
    /**
     * using comfortable units for tests
     *
     *            /
     *         /
     *      /   3*d
     *      -  -  -   0.5PI camera FOV
     *      \
     *         \
     *            \
     */

    using namespace geolib3;
    const double d = 10 /*meters*/;
    Point2 p0(37.61, 55.75); // Moscow

    // dx and dy are d meters shifts in lat,lon coordinates
    Vector2 dx = fastGeoShift(p0, Vector2(d, 0)) - p0;
    Vector2 dy = fastGeoShift(p0, Vector2(0, d)) - p0;

    auto photo1 = makeFeature(p0, Direction2(Vector2(1, 0)));
    auto photo2 = makeFeature(p0 + 4 * dx + dy, Direction2(Vector2(1, 0)));
    auto photo3 = makeFeature(p0 + 9 * dx, Direction2(Vector2(-1, 0)));
    auto photo4 = makeFeature(p0 + 10 * dx - dy, Direction2(Vector2(1, 0)));

    auto photos = db::Features{photo1, photo2, photo3, photo4};

    Polyline2 polyline1(PointsVector{p0 + dx, p0 + 3 * dx, p0 + 12 * dx});
    Polyline2 polyline1r = polyline1;
    polyline1r.reverse();

    Polyline2 polyline2(PointsVector{p0 + dx + 5 * dy, p0 + dx - 5 * dy});
    Polyline2 polyline2r = polyline2;
    polyline2r.reverse();

    Polyline2 polyline3(PointsVector{p0 + 4.1 * dy, p0 + 20 * dx + 4.1 * dy});
    Polyline2 polyline3r = polyline3;
    polyline3r.reverse();

    auto result1 = getCoverage(getSegments(polyline1), photos);
    EXPECT_NEAR(result1.coverageFraction, 5.0 / 11.0, 0.1);

    auto result2 = getCoverage(getSegments(polyline1r), photos);
    EXPECT_NEAR(result2.coverageFraction, 3.0 / 11.0, 0.1);

    auto result3 = getCoverage(getSegments({polyline1, polyline1r}), photos);
    EXPECT_NEAR(result3.coverageFraction, 8.0 / 22.0, 0.1);

    auto result4 = getCoverage(getSegments({polyline2, polyline2r}), photos);
    EXPECT_NEAR(result4.coverageFraction, 0, 0.1);

    auto result5 = getCoverage(getSegments({polyline3, polyline3r}), photos);
    EXPECT_NEAR(result5.coverageFraction, 0, 0.1);

    auto result6 = getCoverage(getSegments({polyline1,
                                            polyline2,
                                            polyline3,
                                            polyline1r,
                                            polyline2r,
                                            polyline3r}),
                               photos);
    EXPECT_NEAR(result6.coverageFraction, 8.0 / (22.0 + 20.0 + 40.0),
                      0.1);

    auto coverage = getCoverage(getSegments(polyline1), photos);
    auto uncovered = coverage.uncoveredParts;
    auto covered = coverage.coveredParts;
    EXPECT_EQ(uncovered.size(), 2u);
    EXPECT_EQ(covered.size(), 3u);

    auto compare =
        [&](const geolib3::Segment2& lhs, const geolib3::Segment2& rhs) {
            return geolib3::geoDistance(p0, lhs.start())
                   < geolib3::geoDistance(p0, rhs.start());
        };
    std::sort(uncovered.begin(), uncovered.end(), compare);
    std::sort(covered.begin(), covered.end(), compare);

    EXPECT_NEAR(uncovered[0].start().y(), p0.y(), 0.01);
    EXPECT_NEAR(uncovered[0].end().y(), p0.y(), 0.01);
    EXPECT_NEAR(uncovered[1].start().y(), p0.y(), 0.01);
    EXPECT_NEAR(uncovered[1].end().y(), p0.y(), 0.01);

    EXPECT_NEAR(covered[0].start().y(), p0.y(), 0.01);
    EXPECT_NEAR(covered[0].end().y(), p0.y(), 0.01);
    EXPECT_NEAR(covered[1].start().y(), p0.y(), 0.01);
    EXPECT_NEAR(covered[1].end().y(), p0.y(), 0.01);
    EXPECT_NEAR(covered[2].start().y(), p0.y(), 0.01);
    EXPECT_NEAR(covered[2].end().y(), p0.y(), 0.01);

    EXPECT_NEAR(uncovered[0].start().x(), (p0 + 3 * dx).x(), 0.01);
    EXPECT_NEAR(uncovered[0].end().x(), (p0 + 5 * dx).x(), 0.01);
    EXPECT_NEAR(uncovered[1].start().x(), (p0 + 7 * dx).x(), 0.01);
    EXPECT_NEAR(uncovered[1].end().x(), (p0 + 11 * dx).x(), 0.01);

    EXPECT_NEAR(covered[0].start().x(), (p0 + dx).x(), 0.01);
    EXPECT_NEAR(covered[0].end().x(), (p0 + 3 * dx).x(), 0.01);
    EXPECT_NEAR(covered[1].start().x(), (p0 + 5 * dx).x(), 0.01);
    EXPECT_NEAR(covered[1].end().x(), (p0 + 7 * dx).x(), 0.01);
    EXPECT_NEAR(covered[2].start().x(), (p0 + 11 * dx).x(), 0.01);
    EXPECT_NEAR(covered[2].end().x(), (p0 + 12 * dx).x(), 0.01);
}

TEST_F(TestFixture, test_inspect_coverage)
{
    constexpr double COVERAGE = 0.35;
    constexpr double TRACK_DISTANCE_IN_METERS = 89.;
    constexpr std::chrono::seconds TRACK_DURATION{13};
    constexpr size_t PROCESSED_PHOTOS = 6;
    constexpr size_t PROCESSED_POINTS = 9;
    constexpr double TOLERANCE_PERCENT = 5.;

    auto coverageStat = inspectCoverage(ctx, ASSIGNMENT_ID, FRONT);
    INFO() << toString(coverageStat);
    EXPECT_NEAR(coverageStat.allPhotosCoverage.coverageFraction,
                      COVERAGE, TOLERANCE_PERCENT);
    EXPECT_NEAR(coverageStat.goodPhotosCoverage.coverageFraction,
                      COVERAGE, TOLERANCE_PERCENT);
    EXPECT_NEAR(coverageStat.trackDistanceInMeters,
                      TRACK_DISTANCE_IN_METERS, TOLERANCE_PERCENT);
    EXPECT_TRUE(coverageStat.trackDuration == TRACK_DURATION);
    EXPECT_EQ(coverageStat.processedPhotos, PROCESSED_PHOTOS);
    EXPECT_EQ(coverageStat.processedPoints, PROCESSED_POINTS);
}

TEST(tests, test_create_geojson)
{
    LineStyle params;
    params.description = "test1";
    params.color = "#0f0f0f";
    params.width = 2;
    params.opacity = 1;

    GeojsonBuilder builder;
    builder.addTrack(
        {geolib3::Polyline2{geolib3::PointsVector{geolib3::Point2(0, 1),
                                                  geolib3::Point2(2, 4),
                                                  geolib3::Point2(8, 16)}},
         geolib3::Polyline2{geolib3::PointsVector{geolib3::Point2(32, 64),
                                                  geolib3::Point2(64, 32)}}},
        params);

    std::string geojsonStr = builder.generateGeojsonString();

    std::string expectedStr
        = "{\"type\":\"FeatureCollection\",\"features\":["
          "{\"type\":\"Feature\",\"id\":0,\"geometry\":{\"type\":"
          "\"LineString\","
          "\"coordinates\":[[0,1],[2,4],[8,16]]},"
          "\"properties\":{\"description\":\"test1\",\"stroke\":\"#0f0f0f\","
          "\"stroke-width\":2,\"stroke-opacity\":1}},"
          "{\"type\":\"Feature\",\"id\":1,\"geometry\":{\"type\":"
          "\"LineString\","
          "\"coordinates\":[[32,64],[64,32]]},"
          "\"properties\":{\"description\":\"test1\",\"stroke\":\"#0f0f0f\","
          "\"stroke-width\":2,\"stroke-opacity\":1}}]}";

    EXPECT_EQ(geojsonStr, expectedStr);
}

TEST(tests, test_geojson_object_points)
{
    GeojsonBuilder builder;
    builder.addAssignmentObjects(makeAssignmentObjects());
    const std::string geojsonStr = builder.generateGeojsonString();
    const std::string EXPECTED = R"({
    "type": "FeatureCollection",
    "features": [
        {
            "type": "Feature",
            "id": 0,
            "geometry": {
                "type": "Point",
                "coordinates": [
                    0,
                    1
                ]
            },
            "properties": {
                "description": "barrier"
            }
        },
        {
            "type": "Feature",
            "id": 1,
            "geometry": {
                "type": "Point",
                "coordinates": [
                    2,
                    4
                ]
            },
            "properties": {
                "description": "barrier:comment"
            }
        }
    ]
})";
    EXPECT_EQ(json::prettifyJson(geojsonStr), EXPECTED);
}

TEST_F(TestFixture, test_assignment_object_feedback_task)
{
    {
        auto txn = pool().masterWriteableTransaction();
        const auto feedbackTasks
            = db::AssignmentObjectFeedbackTaskGateway{*txn}.loadWithoutFeedback();
        EXPECT_EQ(feedbackTasks.size(), 0u);
    }

    // Accept the assignment
    {
        auto assignment = ctx.loadAssignment(ASSIGNMENT_ID);
        assignment.markAsAccepted();
        ctx.update(assignment,
                 ctx.createAssignmentObjectFeedbackTasks(assignment));

        auto txn = pool().masterWriteableTransaction();
    }

    // Check feedback tasks which were posted earlier aren't overwritten in case
    // an assignment state is cleared. E.g. to recalculate feature positions.
    const std::string FEEDBACK_TASK_ID = "feedback task id";
    {
        auto txn = ctx.pool().masterWriteableTransaction();
        auto feedbackTasks = db::AssignmentObjectFeedbackTaskGateway{
            *txn}.loadWithoutFeedback();
        // Check all feedback tasks were inserted
        EXPECT_EQ(feedbackTasks.size(), 4u);

        // Mark the 1st feedback task as posted
        feedbackTasks.at(0).setFeedbackTaskId(FEEDBACK_TASK_ID);
        db::AssignmentObjectFeedbackTaskGateway{*txn}.update(
            feedbackTasks.at(0));
        txn->commit();
    }

    {
        // Reaccept assignment
        auto assignment = ctx.loadAssignment(ASSIGNMENT_ID);
        ctx.update(assignment,
                 ctx.createAssignmentObjectFeedbackTasks(assignment));

        auto txn = ctx.pool().masterWriteableTransaction();

        auto feedbackTasks
            = db::AssignmentObjectFeedbackTaskGateway{*txn}.loadWithoutFeedback();
        EXPECT_EQ(feedbackTasks.size(), 3u);

        feedbackTasks
            = db::AssignmentObjectFeedbackTaskGateway{*txn}.loadWithFeedback();
        EXPECT_EQ(feedbackTasks.size(), 1u);
        EXPECT_EQ(*feedbackTasks.at(0).feedbackTaskId(), FEEDBACK_TASK_ID);
    }
}

TEST_F(TestFixture, test_camera_deviation)
{
    auto photos = ctx.loadProcessedPhotos(ASSIGNMENT_ID);
    EXPECT_TRUE(!photos.empty());
    for (const auto& photo : photos) {
        EXPECT_TRUE(photo.hasCameraDeviation());
    }

    const db::TId ASSIGNMENT_ID_MULTIDIRECTIONAL = 3;
    for (auto cameraDeviation : {FRONT, RIGHT}) {
        auto coverageStat = inspectCoverage(
            ctx, ASSIGNMENT_ID_MULTIDIRECTIONAL, cameraDeviation);
        EXPECT_TRUE(coverageStat.goodPhotosCoverage.coverageFraction > 0);
        auto tolokaStat = inspectQuality(
            ctx, ASSIGNMENT_ID_MULTIDIRECTIONAL, cameraDeviation,
            coverageStat.goodPhotosCoverage.uncoveredParts,
            /* modifyData */ true);
        EXPECT_TRUE(status(tolokaStat) == db::ugc::TolokaStatus::Accepted);
    }
}

TEST_F(TestFixture, test_process_assignment)
{
    // avoid toloka
    {
        auto txn = pool().masterWriteableTransaction();
        db::FeatureGateway{*txn}.remove(db::table::Feature::quality < 0.5);
        txn->commit();
    }

    // image_analyzer_yt in progress
    EXPECT_TRUE(ctx.existsUnprocessedPhotos(ASSIGNMENT_ID));

    auto run = [&] {
        auto assignment = ctx.loadAssignment(ASSIGNMENT_ID);
        processAssignment(assignment,
                          config(),
                          true /* modifyData */,
                          true /* disableEmailReports */,
                          ctx);
    };

    run();

    // make review old
    {
        using namespace std::literals::chrono_literals;
        auto review =
            ctx.loadAssignmentReview(ASSIGNMENT_ID, db::CameraDeviation::Front);
        review->setActualizationDate(chrono::TimePoint::clock::now() - 24h);
        auto txn = pool().masterWriteableTransaction();
        db::ugc::AssignmentReviewGateway{*txn}.upsert(*review);
        txn->commit();
    }

    run();

    EXPECT_TRUE(ctx.loadAssignment(ASSIGNMENT_ID).status() ==
                db::ugc::AssignmentStatus::Completed);

    // image_analyzer_yt is done
    {
        auto txn = pool().masterWriteableTransaction();
        db::FeatureGateway{*txn}.remove(
            db::table::Feature::quality.isNull() ||
            db::table::Feature::roadProbability.isNull() ||
            db::table::Feature::forbiddenProbability.isNull());
        txn->commit();
    }
    EXPECT_TRUE(!ctx.existsUnprocessedPhotos(ASSIGNMENT_ID));

    run();

    EXPECT_TRUE(ctx.loadAssignment(ASSIGNMENT_ID).status() ==
                db::ugc::AssignmentStatus::Accepted);
}

} // tests
} // img_qa
} // mrc
} // maps
