#include "tools.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/common/include/math.h>
#include <maps/libs/deprecated/localeutils/include/localemapper.h>

#include <boost/range/algorithm/copy.hpp>

#include <algorithm>
#include <sstream>

namespace maps {
namespace mrc {
namespace img_qa {
namespace {

const std::string GROUP_ID_FIELD = "group_id";
const std::string PATH_FIELD = "path";

chrono::TimePoint::duration
operator*(chrono::TimePoint::duration duration, double times)
{
    return duration *= times;
}

struct GeoLength {
    double operator()(const geolib3::Segment2& segment) const
    {
        return geolib3::geoLength(segment);
    }
};

geolib3::Polyline2 equidistant(const geolib3::Polyline2& geoLine)
{
    /**
     * The interface of geolib3::Polyline2 is ugly.
     * It's impossible to edit points in-place.
     */

    constexpr double RIGHT_HAND_TRAFFIC_SHIFT_METERS = 10;

    geolib3::PointsVector mercPoints(geoLine.points().size());
    std::transform(geoLine.points().begin(), geoLine.points().end(),
                   mercPoints.begin(), geolib3::geoPoint2Mercator);

    auto mercLine = geolib3::equidistant(
        geolib3::Polyline2(std::move(mercPoints)),
        RIGHT_HAND_TRAFFIC_SHIFT_METERS, geolib3::Orientation::Clockwise);

    geolib3::PointsVector geoPoints(mercLine.points().size());
    std::transform(mercLine.points().begin(), mercLine.points().end(),
                   geoPoints.begin(), geolib3::mercator2GeoPoint);

    return geolib3::Polyline2(std::move(geoPoints));
}

auto makeTuple(const db::TrackPoint& trackPoint)
{
    return std::make_tuple(trackPoint.sourceId(), trackPoint.timestamp());
}

} // anonymous namespace

TolokaStatistics
makeTolokaStatistics(db::TId assignmentId,
                     size_t total,
                     const PhotoIdToTolokaStatusMap& tolokaPhotoToStatusMap)
{
    TolokaStatistics stat{};
    stat.assignmentId = assignmentId;
    stat.total = total;
    for (const auto& photoToStatus : tolokaPhotoToStatusMap) {
        switch (photoToStatus.second) {
        case db::ugc::TolokaStatus::Accepted:
            ++stat.tolokaAccept;
            break;
        case db::ugc::TolokaStatus::Rejected:
            ++stat.tolokaReject;
            break;
        case db::ugc::TolokaStatus::InProgress:
            ++stat.tolokaInProgress;
            break;
        }
    }

    auto posted = stat.tolokaReject + stat.tolokaAccept
                  + stat.tolokaInProgress;
    stat.tolokaComing
        = maps::common::clip<size_t>(stat.total - posted, 0, stat.total);
    return stat;
}

db::ugc::TolokaStatus status(const TolokaStatistics& stat)
{
    if (stat.tolokaInProgress == 0 && stat.tolokaComing == 0) {
        // Toloka tasks are done
        return db::ugc::TolokaStatus::Accepted;
    }
    return db::ugc::TolokaStatus::InProgress;
}

std::string toString(const TolokaStatistics& stat)
{
    std::ostringstream os;
    // clang-format off
    os << "assignment-id=" << stat.assignmentId
       << "; status=" << status(stat)
       << "; total=" << stat.total
       << "; accepted=" << stat.tolokaAccept
       << "; rejected=" << stat.tolokaReject
       << "; in-progress=" << stat.tolokaInProgress
       << "; coming=" << stat.tolokaComing;
    // clang-format on
    return os.str();
}

std::string toString(const CoverageStatistics& stat)
{
    std::ostringstream os;
    // clang-format off
    os << "assignment-id=" << stat.assignmentId
       << "; camera_deviation=" << toIntegral(stat.cameraDeviation)
       << "; coverage_fraction=" << stat.allPhotosCoverage.coverageFraction
       << "; good_coverage_fraction=" << stat.goodPhotosCoverage.coverageFraction
       << "; track_distance_in_meters=" << stat.trackDistanceInMeters
       << "; track_duration_in_seconds=" << stat.trackDuration.count()
       << "; processed_photos=" << stat.processedPhotos
       << "; good_photos=" << stat.goodPhotos
       << "; processed_points=" << stat.processedPoints;
    // clang-format on
    return os.str();
}

void sortByQualityDesc(db::Features& photos)
{
    std::sort(
        photos.begin(), photos.end(),
        [](const db::Feature& lhs, const db::Feature& rhs) {
            return lhs.quality() > rhs.quality();
        });
}

db::FeatureQaTasks
zip(FeatureBatch photos, const toloka::ImageQualityC12nTasks& tasks)
{
    size_t photosCount = std::distance(photos.begin(), photos.end());
    REQUIRE(tasks.size() == photosCount, "photos and tasks size mismatch");
    db::FeatureQaTasks photoTasks;
    photoTasks.reserve(photosCount);
    size_t offset = 0;
    for (const auto& photo : photos) {
        photoTasks.emplace_back(photo.id(), tasks[offset++].id());
    }
    return photoTasks;
}

chrono::TimePoint fromBoost(const boost::posix_time::ptime& time)
{
    return chrono::parseIsoDateTime(
        boost::posix_time::to_iso_extended_string(time));
}

boost::posix_time::ptime toBoost(const chrono::TimePoint& time)
{
    using clock = std::chrono::system_clock;
    return boost::posix_time::from_time_t(clock::to_time_t(
        std::chrono::time_point_cast<clock::time_point::duration>(time)));
}

void sortUniqueByTime(db::TrackPoints& trackPoints)
{
    std::sort(trackPoints.begin(),
              trackPoints.end(),
              [](const db::TrackPoint& lhs, const db::TrackPoint& rhs) {
                  return makeTuple(lhs) < makeTuple(rhs);
              });
    auto end = std::unique(
        trackPoints.begin(),
        trackPoints.end(),
        [](const db::TrackPoint& lhs, const db::TrackPoint& rhs) {
            return lhs.timestamp() == rhs.timestamp() ||
                   toBoost(rhs.timestamp()) == toBoost(lhs.timestamp());
        });
    trackPoints.erase(end, trackPoints.end());
}

TrackSegments makePath(chrono::TimePoint startTime,
                       chrono::TimePoint endTime,
                       const Segments& segments)
{
    std::vector<double> lengths(segments.size());
    auto sum = maps::common::partialSumKahan(segments.begin(), segments.end(), lengths.begin(), GeoLength{});
    TrackSegments result;
    if (sum > geolib3::EPS) { // to avoid division by zero
        result.reserve(segments.size());
        auto start = startTime;
        for (size_t i = 0; i < segments.size(); ++i) {
            auto end = startTime + (endTime - startTime) * (lengths[i] / sum);
            result.push_back({segments[i], start, end});
            start = end;
        }
    }
    return result;
}

TrackSegments makePath(const db::TrackPoints& trackPoints)
{
    static constexpr size_t MAX_DISTANCE_INTERVAL_IN_METERS(1000);
    static constexpr std::chrono::minutes MAX_TIME_INTERVAL(1);

    TrackSegments result;
    result.reserve(trackPoints.size());  // extra one
    std::adjacent_find(
        trackPoints.begin(),
        trackPoints.end(),
        [&result](const db::TrackPoint& lhs, const db::TrackPoint& rhs) {
            auto lengthInMeters =
                geolib3::fastGeoDistance(lhs.geodeticPos(), rhs.geodeticPos());
            auto duration = rhs.timestamp() - lhs.timestamp();
            if (lhs.sourceId() == rhs.sourceId() &&
                lengthInMeters < MAX_DISTANCE_INTERVAL_IN_METERS &&
                duration < MAX_TIME_INTERVAL) {
                result.push_back({{lhs.geodeticPos(), rhs.geodeticPos()},
                                  lhs.timestamp(),
                                  rhs.timestamp()});
            }
            return false;  // continue
        });
    return result;
}

Segments getSegments(const geolib3::Polyline2& polyline)
{
    Segments result;
    boost::range::copy(polyline.segments(), std::back_inserter(result));
    return result;
}

Segments
getSegments(const std::initializer_list<geolib3::Polyline2>& polylines)
{
    Segments result;
    for (const auto& polyline : polylines) {
        result += getSegments(polyline);
    }
    return result;
}

Segments getSegments(const db::ugc::Task& task)
{
    Segments result;
    for (const auto& target : task.targets()) {
        auto polyline = target.geodeticGeom();
        switch (target.direction()) {
        case db::ugc::Direction::Forward:
            result += getSegments(polyline);
            break;
        case db::ugc::Direction::Backward:
            polyline.reverse();
            result += getSegments(polyline);
            break;
        case db::ugc::Direction::Bidirectional:
            result += getSegments(polyline);
            polyline.reverse();
            result += getSegments(polyline);
            break;
        }
    }
    return result;
}

Segments getSegments(const TrackSegments& trackSegments)
{
    Segments result(trackSegments.size());
    std::transform(trackSegments.begin(), trackSegments.end(), result.begin(),
                   [](const TrackSegment& trackSegment) {
                       return trackSegment.segment;
                   });
    return result;
}

bool areRideResultsChanged(const db::ugc::AssignmentReview& review,
                           const CoverageStatistics& coverageStats)
{
    constexpr double TRACK_DISTANCE_CHANGED_THRESHOLD = 10;
    constexpr double COVERAGE_CHANGED_THRESHOLD = 0.01;

    if (fabs(coverageStats.allPhotosCoverage.coverageFraction
             - review.coverageFraction()) > COVERAGE_CHANGED_THRESHOLD) {
        return true;
    }

    if ((!review.goodCoverageFraction()
         && (coverageStats.goodPhotosCoverage.coverageFraction
             > COVERAGE_CHANGED_THRESHOLD))
        || (review.goodCoverageFraction()
            && (fabs(coverageStats.goodPhotosCoverage.coverageFraction
                     - *review.goodCoverageFraction())
                > COVERAGE_CHANGED_THRESHOLD))) {
        return true;
    }

    if ((!review.trackDistanceInMeters()
         && (coverageStats.trackDistanceInMeters
             > TRACK_DISTANCE_CHANGED_THRESHOLD))
        || (review.trackDistanceInMeters()
            && (fabs(coverageStats.trackDistanceInMeters
                     - *review.trackDistanceInMeters())
                > TRACK_DISTANCE_CHANGED_THRESHOLD))) {
        return true;
    }

    return false;
}

/// image quality based on tolokers opinions
bool isPhotoGood(const db::Feature& feature,
                 const PhotoIdToTolokaStatusMap& photoToStatusMap)
{
    if (photoToStatusMap.count(feature.id())) {
        switch (photoToStatusMap.at(feature.id())) {
        case db::ugc::TolokaStatus::InProgress:
            break;
        case db::ugc::TolokaStatus::Accepted:
            return true;
        case db::ugc::TolokaStatus::Rejected:
            return false;
        }
    }
    if (feature.cameraDeviation() != db::CameraDeviation::Front ||
        feature.privacy() != db::FeaturePrivacy::Public) {
        return feature.quality() >= BAD_QUALITY_THRESHOLD;
    }
    return feature.quality() >= GOOD_QUALITY_THRESHOLD;
}

geolib3::PolylinesVector toPolylines(const Segments& segments)
{
    static const double TOLERANCE_METERS = 0.5;

    geolib3::PolylinesVector result;
    geolib3::Polyline2 line;
    for (size_t i = 0; i < segments.size(); i++) {
        if (geolib3::geoLength(segments[i]) > TOLERANCE_METERS) {
            line.add(segments[i].start());
        }
        if (i == segments.size() - 1
            || geolib3::geoDistance(segments[i].end(),
                                    segments[i + 1].start())
                   > TOLERANCE_METERS
            || geolib3::angle(segments[i].vector(), segments[i + 1].vector())
                   > M_PI - geolib3::EPS) {
            line.add(segments[i].end());
            if (geolib3::geoLength(line) > TOLERANCE_METERS) {
                result.push_back(std::move(line));
            }
            line = geolib3::Polyline2{};
        }
    }
    return result;
}

geolib3::PolylinesVector toEquidistantPolylines(const Segments& segments)
{
    geolib3::PolylinesVector result;
    auto lines = img_qa::toPolylines(segments);
    result.reserve(lines.size());
    for (const auto& line : lines) {
        result.push_back(equidistant(line));
    }
    return result;
}

std::string getName(const db::ugc::Task& task, const Locale& locale)
{
    const auto& names = task.names();
    auto it = maps::findBestPair(names.begin(), names.end(), locale);
    return it == names.end() ? std::string{} : it->second;
}

void fillReview(const CoverageStatistics& coverageStat,
                db::ugc::AssignmentReview& review)
{
    review.setCoverageFraction(coverageStat.allPhotosCoverage.coverageFraction)
        .setGoodCoverageFraction(coverageStat.goodPhotosCoverage.coverageFraction)
        .setTrackDistanceInMeters(coverageStat.trackDistanceInMeters)
        .setTrackDuration(coverageStat.trackDuration)
        .setProcessedPhotos(coverageStat.processedPhotos)
        .setProcessedPoints(coverageStat.processedPoints)
        .setGoodPhotos(coverageStat.goodPhotos)
        .setCoveredGeodeticGeom(toEquidistantPolylines(
            coverageStat.allPhotosCoverage.coveredParts))
        .setUncoveredGeodeticGeom(toEquidistantPolylines(
            coverageStat.allPhotosCoverage.uncoveredParts))
        .setTrackGeodeticGeom(coverageStat.track);
    if (coverageStat.firstShotTime && coverageStat.lastShotTime) {
        review.setFirstShotTime(*coverageStat.firstShotTime)
            .setLastShotTime(*coverageStat.lastShotTime);
    }
}

std::string mdsKeyToJson(const mds::Key& key)
{
    json::Builder builder;
    builder << [&key](json::ObjectBuilder objectBuilder) {
        objectBuilder[GROUP_ID_FIELD] = key.groupId;
        objectBuilder[PATH_FIELD] = key.path;
    };
    return builder.str();
}

mds::Key jsonToMdsKey(const std::string& json)
{
    auto val = json::Value::fromString(json);
    return val.construct<mds::Key>(GROUP_ID_FIELD, PATH_FIELD);
}

} // img_qa
} // mrc
} // maps
