#include "emitter.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/assignment_object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/assignment_object_feedback_task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/retry.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>

#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>

namespace maps {
namespace mrc {
namespace social {

namespace {

const char* FEEDBACK_TASKS_OTHER_HANDLE = "/feedback/tasks";
const char* FEEDBACK_TASKS_BARRIER_HANDLE = "/feedback/tasks/absent-barrier";

char const* getHttpHandle(db::ugc::AssignmentObjectType objectType)
{
    if (objectType == db::ugc::AssignmentObjectType::Barrier) {
        return FEEDBACK_TASKS_BARRIER_HANDLE;
    }
    else {
        return FEEDBACK_TASKS_OTHER_HANDLE;
    }
}

char const* getSource(db::ugc::AssignmentObjectType objectType)
{
    switch (objectType) {
    case db::ugc::AssignmentObjectType::Barrier:
        return "mrc-privateapp-barrier";
    case db::ugc::AssignmentObjectType::Deadend:
        return "mrc-privateapp-deadend";
    case db::ugc::AssignmentObjectType::BadConditions:
        return "mrc-privateapp-bad_conditions";
    case db::ugc::AssignmentObjectType::NoEntry:
        return "mrc-privateapp-no_entry";
    }
    REQUIRE(false, "Unexpected object type");
}

std::string getDescription(db::ugc::AssignmentObjectType objectType)
{
    // TODO: The hardcoded russian text is to be localized.
    // UTF-8 encoding must be used
    switch (objectType) {
    case db::ugc::AssignmentObjectType::Deadend:
        return "На карте указан несуществующий проезд";
    case db::ugc::AssignmentObjectType::BadConditions:
        return "На карте ошибочно отмечено, что дорога имеет твердое "
               "покрытие";
    case db::ugc::AssignmentObjectType::NoEntry:
        return "На карте отсутствует запрещённый маневр";
    default:
        break;
    }
    REQUIRE(false, "Unexpected object type");
}

std::string buildRequestBody(const db::ugc::AssignmentObject& object)
{
    json::Builder bodyBuilder;
    bodyBuilder << [&](json::ObjectBuilder bodyBuilder) {
        bodyBuilder["workflow"] << "feedback";
        bodyBuilder["source"] << getSource(object.objectType());
        bodyBuilder["position"] << [&](json::ObjectBuilder positionBuilder) {
            positionBuilder["coordinates"]
                << [&](json::ArrayBuilder coordsBuilder) {
                       coordsBuilder << object.geodeticPos().x()
                                     << object.geodeticPos().y();
                   };
            positionBuilder["type"] << "Point";
        };

        // Note: for a barrier there is a specific HTTP handle that sets
        //       special task type and description.
        if (object.objectType() == db::ugc::AssignmentObjectType::Barrier
                                  && object.comment()) {
            bodyBuilder["userComment"] << *object.comment();
        }
        else if (object.objectType()
                 != db::ugc::AssignmentObjectType::Barrier) {
            bodyBuilder["type"] << "other";

            std::string description = getDescription(object.objectType());
            if (object.comment()) {
                description += "\n\nкомментарий пользователя:\n";
                description += *object.comment();
            }
            bodyBuilder["description"] << description;
        }
        bodyBuilder["hidden"] = true;

    };

    return bodyBuilder.str();
}

class HttpStatus500 : public Exception {
    using Exception::Exception;
};

} // namespace

AssignmentObjectFeedbackTasksEmitter::AssignmentObjectFeedbackTasksEmitter(
    const maps::mrc::common::Config& cfg)
    : socialUrl_{cfg.externals().socialBackofficeUrl()}
    , poolHolder_{cfg.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                                     maps::mrc::common::LONG_READ_POOL_ID)}
    , dbAccessMutex_(
        poolHolder_.pool(),
        static_cast<int64_t>(maps::mrc::common::LockId::AssignmentObjectFeedbackTasksEmitter)
    )
{
    httpClient_.setTimeout(std::chrono::seconds{1});
    if (!dbAccessMutex_.try_lock()) {
        throw AnotherProcessIsRunning();
    }
}

db::ugc::AssignmentObjects
AssignmentObjectFeedbackTasksEmitter::loadPublishableAssignmentObjects(
    const db::TIds& objectIds)
{
    auto txn = poolHolder_.pool().slaveTransaction();

    auto objectNotVisible
        = [&txn](const db::ugc::AssignmentObject& assignmentObject)
    {
        static const double LOCALITY_METERS = 100;
        auto bbox = geolib3::BoundingBox{
            geolib3::fastGeoShift(assignmentObject.geodeticPos(),
                                  {-LOCALITY_METERS, -LOCALITY_METERS}),
            geolib3::fastGeoShift(assignmentObject.geodeticPos(),
                                  {+LOCALITY_METERS, +LOCALITY_METERS})};
        return !db::FeatureGateway{*txn}.exists(
                   db::table::Feature::isPublished
                   && db::table::Feature::pos.intersects(
                          geolib3::convertGeodeticToMercator(bbox))
                   && db::table::Feature::assignmentId
                          == assignmentObject.assignmentId());
    };

    auto objects = db::ugc::AssignmentObjectGateway{*txn}.loadByIds(objectIds);

    objects.erase(
        std::remove_if(objects.begin(), objects.end(), objectNotVisible),
        objects.end());

    return objects;

}

db::TIds
AssignmentObjectFeedbackTasksEmitter::loadUnpostedAssignmentObjectIds()
{
    auto txn = poolHolder_.pool().slaveTransaction();
    return db::ugc::AssignmentObjectGateway{*txn}.loadIdsWithoutFeedback();
}

void AssignmentObjectFeedbackTasksEmitter::postFeedbackTasks(
    const db::ugc::AssignmentObjects& objects)
{
    db::AssignmentObjectFeedbackTasks feedbackTasks;
    feedbackTasks.reserve(objects.size());

    for (const auto& object : objects) {
        auto txn = poolHolder_.pool().masterWriteableTransaction();
        INFO() << "Posting a feedback task to the map editor for object "
               << object.objectId();

        auto feedbackTask = postOneFeedbackTask(object);

        REQUIRE(feedbackTask.feedbackTaskId(), "Feedback task ID for "
                                                   << feedbackTask.objectId()
                                                   << "wasn' set!");
        INFO() << "Feedback task " << *feedbackTask.feedbackTaskId()
               << " for object " << feedbackTask.objectId() << " is posted";

        db::AssignmentObjectFeedbackTaskGateway{*txn}.update(feedbackTask);
        txn->commit();

        DEBUG() << "object (" << feedbackTask.objectId()
                << ") <--> feedback task (" << *feedbackTask.feedbackTaskId()
                << ") relation is saved";
    }
}

db::AssignmentObjectFeedbackTask
AssignmentObjectFeedbackTasksEmitter::postOneFeedbackTask(
    const db::ugc::AssignmentObject& object)
{
    const std::string body = buildRequestBody(object);
    http::URL url = socialUrl_;
    url.setPath(getHttpHandle(object.objectType()));

    auto response = common::retryOnException<HttpStatus500>(
        common::RetryPolicy()
            .setInitialTimeout(std::chrono::seconds(1))
            .setMaxAttempts(3)
            .setTimeoutBackoff(2),
        [&]() {
            http::Request request(httpClient_, http::POST, url);
            if (tvmClient_) {
                static const TString SOCIAL_BACKOFFICE_TVM_ALIAS = "social-backoffice";
                auto tvmTicket = tvmClient_->GetServiceTicketFor(SOCIAL_BACKOFFICE_TVM_ALIAS);
                request.addHeader(auth::SERVICE_TICKET_HEADER, std::move(tvmTicket));
            }
            request.setContent(body);
            auto response = request.perform();
            if (response.status() >= 500) {
                throw HttpStatus500() << url
                                      << ", status: " << response.status();
            }
            REQUIRE(response.status() == 200, "Unexpected server response ["
                                                  << url << "]:\n"
                                                  << response.readBody());
            return response;
        });

    return db::AssignmentObjectFeedbackTask{object.objectId()}
        .setFeedbackTaskId(
            json::Value{response.body()}["feedbackTask"]["id"].as<std::string>()
        );
}

void
AssignmentObjectFeedbackTasksEmitter::initTvmClient()
{
    tvmClient_ = std::make_unique<NTvmAuth::TTvmClient>(auth::TvmtoolSettings().makeTvmClient());
}

} // namespace social
} // namespace mrc
} // namespace maps
