#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ride_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/ugc_event_logger/include/logger.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/async_ride_correction/lib/utility.h>

#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/geolib/include/vector.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>
#include <yandex/maps/mrc/unittest/local_server.h>
#include <yandex/maps/mrc/unittest/unittest_config.h>

#include <filesystem>

namespace maps::mrc::ride_correction::tests {

using namespace geolib3;
using namespace std::chrono_literals;

namespace {

const auto USER_ID = std::string{"Pinocchio"};
const auto SOURCE_ID = std::string{"wooden"};
const auto TIME = chrono::parseIsoDateTime("2021-02-09T10:40:00+03");

chrono::TimePoint constGetCurrentTimestamp()
{
    return chrono::TimePoint(chrono::TimePoint::duration(0));
};

const auto CLIENT_RIDE_IDS = std::vector<std::optional<std::string>>{
    std::nullopt,
    "b0c2d8c8-6fc6-45d0-9e8e-45e37bd29636"};

struct Fixture : testing::Test,
                 unittest::WithUnittestConfig<unittest::DatabaseFixture,
                                              unittest::MdsStubFixture> {
    Fixture()
        : ugcEventsLogPath("ugc_events.log")
        , ugcEventLogger(new ugc_event_logger::Logger(ugcEventsLogPath,
                                                      std::chrono::seconds(600),
                                                      constGetCurrentTimestamp))
        , userInfo{.ip = "127.0.0.1", .uid = "1"}
    {
    }

    ~Fixture() { std::filesystem::remove(ugcEventsLogPath); }

    db::Feature makePhoto(const std::string& userId,
                          const std::string& sourceId,
                          const std::optional<std::string>& clientRideId,
                          chrono::TimePoint time)
    {
        static auto jpg = maps::common::readFileToString("opencv_1.jpg");
        static auto pos = Point2{37.563593, 55.667642};
        static auto delta = Vector2{EPS, EPS};
        static auto counter = 0;
        auto mdsKey =
            config().makeMdsClient().post(std::to_string(++counter), jpg).key();
        auto result = sql_chemistry::GatewayAccess<db::Feature>::construct()
                          .setDataset(db::Dataset::Rides)
                          .setSourceId(sourceId)
                          .setTimestamp(time)
                          .setGeodeticPos(pos + delta * counter)
                          .setHeading(Heading{90})
                          .setMdsKey(mdsKey)
                          .setUserId(userId);
        if (clientRideId) {
            result.setClientRideId(*clientRideId);
        }
        return result;
    }

    std::string finishUgcLogAndGetContents()
    {
        ugcEventLogger.reset();
        return maps::common::readFileToString(ugcEventsLogPath);
    }

    std::string ugcEventsLogPath;
    std::unique_ptr<ugc_event_logger::Logger> ugcEventLogger;
    ugc_event_logger::UserInfo userInfo;
};

}  // namespace

TEST_F(Fixture, test_deleted_intervals)
{
    for (const auto& clientRideId : CLIENT_RIDE_IDS) {
        auto photos = db::Features{};
        for (int i = 0; i < 5; ++i) {
            photos.push_back(
                makePhoto(USER_ID, SOURCE_ID, clientRideId, TIME + 1min * i));
        }
        auto deletedIntervals = db::DeletedIntervals{};
        deletedIntervals.emplace_back(USER_ID,
                                      SOURCE_ID,
                                      clientRideId,
                                      photos[0].timestamp(),
                                      photos[1].timestamp());
        deletedIntervals.emplace_back(USER_ID,
                                      SOURCE_ID,
                                      clientRideId,
                                      photos[3].timestamp(),
                                      photos[4].timestamp());
        {
            auto txn = pool().masterWriteableTransaction();
            db::FeatureGateway{*txn}.insert(photos);
            db::DeletedIntervalGateway{*txn}.insert(deletedIntervals);
            txn->commit();
        }

        EXPECT_EQ(5u,
                  db::FeatureGateway{*pool().masterReadOnlyTransaction()}.count(
                      makeFeatureFilter(USER_ID,
                                        SOURCE_ID,
                                        clientRideId,
                                        photos.front().timestamp(),
                                        photos.back().timestamp()) &&
                      !db::table::Feature::deletedByUser.is(true)));

        updateRides(pool(),
                    USER_ID,
                    SOURCE_ID,
                    clientRideId,
                    photos.front().timestamp(),
                    photos.back().timestamp(),
                    std::nullopt,  //< showAuthorship
                    userInfo,
                    *ugcEventLogger);

        auto notDeletedPhotos =
            db::FeatureGateway{*pool().masterReadOnlyTransaction()}.load(
                makeFeatureFilter(USER_ID,
                                  SOURCE_ID,
                                  clientRideId,
                                  photos.front().timestamp(),
                                  photos.back().timestamp()) &&
                !db::table::Feature::deletedByUser.is(true));
        EXPECT_EQ(1u, notDeletedPhotos.size());
        EXPECT_EQ(notDeletedPhotos.front().id(), photos[2].id());
    }
}

TEST_F(Fixture, test_show_authorship)
{
    for (const auto& clientRideId : CLIENT_RIDE_IDS) {
        auto photo = makePhoto(USER_ID, SOURCE_ID, clientRideId, TIME);
        {
            auto txn = pool().masterWriteableTransaction();
            db::FeatureGateway{*txn}.insert(photo);
            txn->commit();
        }

        EXPECT_FALSE(
            db::FeatureGateway{*pool().masterReadOnlyTransaction()}.exists(
                makeFeatureFilter(
                    USER_ID, SOURCE_ID, clientRideId, TIME, TIME) &&
                db::table::Feature::showAuthorship.is(true)));

        updateRides(pool(),
                    USER_ID,
                    SOURCE_ID,
                    clientRideId,
                    TIME,
                    TIME,
                    true,  //< showAuthorship
                    userInfo,
                    *ugcEventLogger);

        EXPECT_TRUE(
            db::FeatureGateway{*pool().masterReadOnlyTransaction()}.exists(
                makeFeatureFilter(
                    USER_ID, SOURCE_ID, clientRideId, TIME, TIME) &&
                db::table::Feature::showAuthorship.is(true)));

        updateRides(pool(),
                    USER_ID,
                    SOURCE_ID,
                    clientRideId,
                    TIME,
                    TIME,
                    false,  //< showAuthorship
                    userInfo,
                    *ugcEventLogger);

        EXPECT_FALSE(
            db::FeatureGateway{*pool().masterReadOnlyTransaction()}.exists(
                makeFeatureFilter(
                    USER_ID, SOURCE_ID, clientRideId, TIME, TIME) &&
                db::table::Feature::showAuthorship.is(true)));
    }
}

TEST_F(Fixture, test_log_ugc_events)
{
    for (const auto& clientRideId : CLIENT_RIDE_IDS) {
        auto photo = makePhoto(USER_ID, SOURCE_ID, clientRideId, TIME);
        {
            auto txn = pool().masterWriteableTransaction();
            db::FeatureGateway{*txn}.insert(photo);
            txn->commit();
        }

        updateRides(pool(),
                    USER_ID,
                    SOURCE_ID,
                    clientRideId,
                    TIME,
                    TIME,
                    true,  //< showAuthorship
                    userInfo,
                    *ugcEventLogger);

        auto deletedIntervals = db::DeletedIntervals{};
        deletedIntervals.emplace_back(
            USER_ID, SOURCE_ID, clientRideId, TIME, TIME);
        {
            auto txn = pool().masterWriteableTransaction();
            db::DeletedIntervalGateway{*txn}.insert(deletedIntervals);
            txn->commit();
        }

        updateRides(pool(),
                    USER_ID,
                    SOURCE_ID,
                    clientRideId,
                    TIME,
                    TIME,
                    std::nullopt,  //< showAuthorship
                    userInfo,
                    *ugcEventLogger);
    }

    const std::string EXPECTED_LOG =
        R"({"timestamp":"0","user":{"ip":"127.0.0.1","uid":"1"},"object":{"type":"photo","id":"1","show_authorship":true},"action":"update"}
{"timestamp":"0","user":{"ip":"127.0.0.1","uid":"1"},"object":{"type":"photo","id":"1"},"action":"delete"}
{"timestamp":"0","user":{"ip":"127.0.0.1","uid":"1"},"object":{"type":"photo","id":"2","show_authorship":true},"action":"update"}
{"timestamp":"0","user":{"ip":"127.0.0.1","uid":"1"},"object":{"type":"photo","id":"2"},"action":"delete"}
)";
    EXPECT_EQ(finishUgcLogAndGetContents(), EXPECTED_LOG);
}

}  // namespace maps::mrc::ride_correction::tests
