#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/mongo/include/init.h>
#include <maps/tools/grinder/worker/include/api.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/wiki_config.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/async_takeout_uploader/lib/takeout.h>
#include <util/generic/yexception.h>

namespace {

const std::string GRINDER_WORKER_TYPE = "mrc-async-takeout-uploader";
}

int main(int argc, char* argv[]) try {
    using namespace maps;

    cmdline::Parser parser;
    auto configPath
        = parser.string("config").help("path to mrc configuration");
    auto syslogTag
        = parser.string("syslog-tag")
              .help("redirect log output to syslog with given tag");
    auto grinderConfigPath = parser.string("grinder-config")
                                 .help("path to grinder configuration file");
    auto concurrencyOpt = parser.size_t("concurrency")
            .defaultValue(5)
            .help("number of threads to process tasks");

    parser.parse(argc, argv);

    if (syslogTag.defined()) {
        log8::setBackend(maps::log8::toSyslog(syslogTag));
    }

    maps::mongo::init();

    auto nmapsConfigPtr = std::make_unique<wiki::common::ExtendedXmlDoc>(
        mrc::common::makeWikiConfig(mrc::common::WIKI_TEMPLATE_CONFIG_DIR));

    mrc::takeout::TakeoutUploader takeoutUploader{
        mrc::common::templateConfigFromCmdPath(configPath),
        *nmapsConfigPtr};

    grinder::worker::Options workerOpts;
    if (grinderConfigPath.defined()) {
        workerOpts.setConfigLocation(grinderConfigPath);
    }

    workerOpts
        .on(GRINDER_WORKER_TYPE,
            [&](const grinder::worker::Task& task) {
                MAPS_LOG_THREAD_PREFIX_APPEND(task.id());
                takeoutUploader.handleTakeoutUpload(
                    task.args()["uid"].as<std::string>(),
                    task.args()["job_id"].as<std::string>());
            })
        .setConcurrencyLevel(concurrencyOpt);

    grinder::worker::run(workerOpts);

    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
}
catch (const yexception& e) {
    FATAL() << "Worker failed: " << e.what();
    if (e.BackTrace()) {
        FATAL() << e.BackTrace()->PrintToString();
    }
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
