#pragma once
#include "strings.h"
#include "types.h"

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/geolib/include/heading.h>
#include <maps/libs/geolib/include/point.h>

#include <string>

namespace maps::mrc::drive_activity_stat {

/// Wrapper for rows in //home/logfeller/logs/maps-log* tables
struct ServiceLogRecord {
    std::optional<std::string> vhost;
    std::string isoEventtime;
    std::optional<std::string> request;
    std::optional<std::string> userAgent;

    chrono::TimePoint eventTimePoint() const {
        return chrono::parseSqlDateTime(isoEventtime);
    }

    template<class T>
    static auto introspect(T& t)
    {
        return std::tie(t.vhost, t.isoEventtime, t.request, t.userAgent);
    }

    static auto columns() {
        return std::make_tuple(VHOST, ISO_EVENTTIME, REQUEST, USER_AGENT);
    }
};


struct ParsedFirmwareUpdateRequest {
    chrono::TimePoint time;
    std::string deviceId;
    std::optional<std::string> firmwareVersion;

    template<class T>
    static auto introspect(T& t)
    {
        return std::tie(t.time, t.deviceId, t.firmwareVersion);
    }

    static auto columns() {
        return std::make_tuple(TIME, DEVICE_ID, FIRMWARE_VERSION);
    }
};

struct DeviceDailyActivityRecord {
    Date date;
    std::string deviceId;
    std::optional<std::string> firmwareVersion;
    std::chrono::minutes activePeriodDuration;

    template<class T>
    static auto introspect(T& t)
    {
        return std::tie(t.date, t.deviceId, t.firmwareVersion, t.activePeriodDuration);
    }

    static auto columns() {
        return std::make_tuple(DATE, DEVICE_ID, FIRMWARE_VERSION, ACTIVE_PERIOD_DURATION);
    }
};

struct DevicesRegistryRecord {
    std::string vin;
    std::string signalqSN;
    int64_t telematicsImei;

    template<class T>
    static auto introspect(T& t)
    {
        return std::tie(t.vin, t.signalqSN, t.telematicsImei);
    }

    static auto columns() {
        return std::make_tuple(VIN, SIGNALQ_SN, TELEMATICS_IMEI);
    }
};

struct ParsedTelematicsRecord {
    int64_t telematicsImei;
    chrono::TimePoint time;
    geolib3::Point2 geoPos;

    template<class T>
    static auto introspect(T& t)
    {
        return std::tie(t.telematicsImei, t.time, t.geoPos);
    }

    static auto columns() {
        return std::make_tuple(TELEMATICS_IMEI, TIME, POS);
    }
};

struct TelematicsDailyActivityRecord {
    Date date;
    int64_t telematicsImei;
    std::chrono::minutes activePeriodDuration;

    template<class T>
    static auto introspect(T& t)
    {
        return std::tie(t.date, t.telematicsImei, t.activePeriodDuration);
    }

    static auto columns() {
        return std::make_tuple(DATE, TELEMATICS_IMEI, ACTIVE_PERIOD_DURATION);
    }
};


} // namespace maps::mrc::drive_activity_stat
