#include "db.h"
#include "utility.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/hypothesis_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/metadata_gateway.h>

#include <boost/range/adaptors.hpp>
#include <boost/range/join.hpp>

#include <mutex>
#include <string>

namespace maps::mrc::excess_features_eraser {
namespace {

const std::string APP_NAME = "excess_features_eraser";
const std::string LAST_RUN_TIME = APP_NAME + ".timestamp";
constexpr auto READ_BATCH_SIZE = 10'000;
constexpr auto WRITE_BATCH_SIZE = 100;

}  // namespace

void updateLastRunTime(pgpool3::Pool& pool, chrono::TimePoint now)
{
    auto txn = pool.masterWriteableTransaction();
    db::MetadataGateway{*txn}.upsertByKey(LAST_RUN_TIME,
                                          chrono::formatSqlDateTime(now));
    txn->commit();
}

std::optional<chrono::TimePoint> loadLastRunTime(pqxx::transaction_base& txn)
{
    auto result = db::MetadataGateway{txn}.tryLoadByKey(LAST_RUN_TIME);
    if (!result) {
        return std::nullopt;
    }
    return chrono::parseSqlDateTime(*result);
}

Photos loadPhotos(pgpool3::Pool& pgPool,
                  const db::TIds& featureIds,
                  StringSet& stringSet,
                  chrono::TimePoint now)
{
    auto guard = std::mutex{};
    auto batchesNumber = 0;
    auto result = Photos{};
    result.reserve(featureIds.size());
    parallelForEachBatch<THREADS_NUMBER, READ_BATCH_SIZE>(
        featureIds, [&](std::span<const db::TId> batch) {
            auto features = db::FeatureGateway(*pgPool.slaveTransaction())
                                .loadByIds({batch.begin(), batch.end()});
            auto lock = std::lock_guard{guard};
            for (const auto& feature : features) {
                result.push_back(toPhoto(feature, stringSet, now));
            }
            if (++batchesNumber % 100 == 0) {
                INFO() << concat("loaded ", result.size(), " features");
            }
        });
    INFO() << concat("loaded ", result.size(), " features");
    return result;
}

db::TIds loadFeatureIdsFromHypotheses(pgpool3::Pool& pgPool,
                                      const db::TIds& featureIds)
{
    using boost::adaptors::keys;
    using boost::adaptors::values;

    auto guard = std::mutex{};
    auto batchesNumber = 0;
    auto result = db::TIds{};
    parallelForEachBatch<THREADS_NUMBER, READ_BATCH_SIZE>(
        featureIds, [&](std::span<const db::TId> batch) {
            auto txn = pgPool.slaveTransaction();

            auto frameToFeature =
                makeMap(db::eye::FeatureToFrameGateway{*txn}.load(
                            db::eye::table::FeatureToFrame::featureId.in(
                                makeFlatSet(batch))),
                        &db::eye::FeatureToFrame::frameId,
                        &db::eye::FeatureToFrame::featureId);
            if (frameToFeature.empty()) {
                return;
            }

            auto groupToFrame =
                makeMap(db::eye::DetectionGroupGateway{*txn}.load(
                            db::eye::table::DetectionGroup::frameId.in(
                                makeFlatSet(keys(frameToFeature)))),
                        &db::eye::DetectionGroup::id,
                        &db::eye::DetectionGroup::frameId);
            if (groupToFrame.empty()) {
                return;
            }

            auto detectionToGroup =
                makeMap(db::eye::DetectionGateway{*txn}.load(
                            db::eye::table::Detection::groupId.in(
                                makeFlatSet(keys(groupToFrame))) &&
                            db::eye::table::Detection::deleted.is(false)),
                        &db::eye::Detection::id,
                        &db::eye::Detection::groupId);
            if (detectionToGroup.empty()) {
                return;
            }

            auto primaryToDetection = makeMap(
                db::eye::PrimaryDetectionRelationGateway{*txn}.load(
                    db::eye::table::PrimaryDetectionRelation::detectionId.in(
                        makeFlatSet(keys(detectionToGroup))) &&
                    db::eye::table::PrimaryDetectionRelation::deleted.is(
                        false)),
                &db::eye::PrimaryDetectionRelation::primaryDetectionId,
                &db::eye::PrimaryDetectionRelation::detectionId);
            if (primaryToDetection.empty()) {
                return;
            }

            auto objectToPrimary = makeMap(
                db::eye::ObjectGateway{*txn}.load(
                    db::eye::table::Object::primaryDetectionId.in(makeFlatSet(
                        boost::range::join(keys(detectionToGroup),
                                           keys(primaryToDetection)))) &&
                    db::eye::table::Object::deleted.is(false)),
                &db::eye::Object::id,
                &db::eye::Object::primaryDetectionId);
            if (objectToPrimary.empty()) {
                return;
            }

            auto hypothesisToObjects = makeMap(
                db::eye::HypothesisObjectGateway{*txn}.load(
                    db::eye::table::HypothesisObject::objectId.in(
                        makeFlatSet(keys(objectToPrimary))) &&
                    db::eye::table::HypothesisObject::deleted.is(false)),
                &db::eye::HypothesisObject::hypothesisId,
                &db::eye::HypothesisObject::objectId);
            if (hypothesisToObjects.empty()) {
                return;
            }

            auto hypothesisIds = db::eye::HypothesisGateway{*txn}.loadIds(
                db::eye::table::Hypothesis::id.in(
                    makeFlatSet(keys(hypothesisToObjects))) &&
                db::eye::table::Hypothesis::deleted.is(false));
            if (hypothesisIds.empty()) {
                return;
            }

            auto feedbackToHypotheses =
                makeMap(db::eye::HypothesisFeedbackGateway{*txn}.load(
                            db::eye::table::HypothesisFeedback::hypothesisId.in(
                                hypothesisIds)),
                        &db::eye::HypothesisFeedback::feedbackId,
                        &db::eye::HypothesisFeedback::hypothesisId);
            if (feedbackToHypotheses.empty()) {
                return;
            }

            eraseFrom(hypothesisToObjects,
                      makeDifference(keys(hypothesisToObjects),
                                     values(feedbackToHypotheses)));
            eraseFrom(objectToPrimary,
                      makeDifference(keys(objectToPrimary),
                                     values(hypothesisToObjects)));
            eraseFrom(primaryToDetection,
                      makeDifference(keys(primaryToDetection),
                                     values(objectToPrimary)));
            eraseFrom(
                detectionToGroup,
                makeDifference(keys(detectionToGroup),
                               boost::range::join(values(objectToPrimary),
                                                  values(primaryToDetection))));
            eraseFrom(
                groupToFrame,
                makeDifference(keys(groupToFrame), values(detectionToGroup)));
            eraseFrom(
                frameToFeature,
                makeDifference(keys(frameToFeature), values(groupToFrame)));
            auto features = makeFlatSet(values(frameToFeature));
            if (features.empty()) {
                return;
            }

            auto lock = std::lock_guard{guard};
            std::move(
                features.begin(), features.end(), std::back_inserter(result));
            if (++batchesNumber % 10 == 0) {
                INFO() << concat("loaded ", result.size(), " hypothetic");
            }
        });
    INFO() << concat("loaded ", result.size(), " hypothetic");
    return result;
}

void unpublishFeatures(pgpool3::Pool& pgPool, const db::TIds& featureIds)
{
    auto guard = std::mutex{};
    auto batchesNumber = 0;
    auto affected = 0;
    parallelForEachBatch<THREADS_NUMBER, WRITE_BATCH_SIZE>(
        featureIds, [&](std::span<const db::TId> batch) {
            auto txn = pgPool.masterWriteableTransaction();
            auto features = db::FeatureGateway(*txn).load(
                db::table::Feature::id.in({batch.begin(), batch.end()}) &&
                db::table::Feature::isPublished &&
                db::table::Feature::dataset != db::Dataset::Rides);
            for (auto& feature : features) {
                feature.setAutomaticShouldBePublished(false);
            }
            db::FeatureGateway{*txn}.update(features,
                                            db::UpdateFeatureTxn::Yes);
            txn->commit();

            auto lock = std::lock_guard{guard};
            affected += features.size();
            if (++batchesNumber % 100 == 0) {
                INFO() << concat("affected ", affected, " features");
            }
        });
    INFO() << concat("affected ", affected, " features");
}

}  // namespace maps::mrc::excess_features_eraser
