#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/excess_features_eraser/lib/db.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/excess_features_eraser/lib/unpublish.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>

namespace eraser = maps::mrc::excess_features_eraser;

int main(int argc, char* argv[])
try {
    // clang-format off
    auto parser = maps::cmdline::Parser{};
    auto syslog = parser.string("syslog-tag")
                      .help("redirect log output to syslog with given tag");
    auto configPath = parser.string("config").help("path to configuration");
    auto secretVersion = parser.string("secret").help(
        "version for secrets from yav.yandex-team.ru");
    auto pedestrianGraphDir = parser.dir("pedestrian-graph")
        .help("yandex-maps-mrc-pedestrian-graph-pro dataset directory")
        .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-pedestrian-graph-pro");
    auto photoToPedestrianEdgeDir = parser.dir("photo-to-pedestrian-edge")
        .help("yandex-maps-mrc-photo-to-pedestrian-edge-pro dataset directory")
        .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-photo-to-pedestrian-edge-pro");
    auto roadGraphDir = parser.dir("road-graph")
        .help("yandex-maps-mrc-graph-pro dataset directory")
        .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-graph-pro");
    auto photoToRoadEdgeDir = parser.dir("photo-to-road-edge")
        .help("yandex-maps-mrc-photo-to-edge-pro dataset directory")
        .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-photo-to-edge-pro");
    auto dryRun = parser.flag("dry-run").help("do not update data");
    parser.parse(argc, argv);
    // clang-format on

    if (syslog.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslog));
    }
    auto config =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, configPath);
    auto postgres = config.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                                          maps::mrc::common::LONG_READ_POOL_ID);
    auto mutex = maps::pgp3utils::PgAdvisoryXactMutex{
        postgres.pool(),
        static_cast<int64_t>(maps::mrc::common::LockId::ExcessFeaturesEraser)};
    if (!dryRun && !mutex.try_lock()) {
        INFO() << "another process is ongoing";
        return EXIT_SUCCESS;
    }

    auto now = maps::chrono::TimePoint::clock::now();
    INFO() << "erasing photos, dryRun=" << dryRun;
    INFO() << "pedestrian graph";
    eraser::unpublishExcessFeaturesMatchedOnGraph(
        postgres.pool(),
        pedestrianGraphDir,
        photoToPedestrianEdgeDir,
        now,
        dryRun
    );
    INFO() << "road graph";
    eraser::unpublishExcessFeaturesMatchedOnGraph(
        postgres.pool(),
        roadGraphDir,
        photoToRoadEdgeDir,
        now,
        dryRun
    );
    if (!dryRun) {
        INFO() << "updating metadata";
        eraser::updateLastRunTime(postgres.pool(), now);
    }

    INFO() << "done";
    return EXIT_SUCCESS;
}
catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
}
catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
