#pragma once

#include "tools.h"

#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/object_in_photo.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/walk_object.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/features_reader.h>

#include <functional>
#include <unordered_map>

namespace maps::mrc::export_gen {

using ObjectsInPhotoMap = std::unordered_map<db::TId, db::ObjectsInPhoto>;
using FeaturesFn = std::function<void(db::Features&&, ObjectsInPhotoMap&&, Origin)>;
using WalkObjectsFn = std::function<void(db::WalkObjects&&)>;

struct ILoader {
    virtual ~ILoader() = default;
    virtual void forEachPublishedFeatures(size_t batchSize, FeaturesFn) const = 0;
    virtual void forEachWalkObjects(size_t batchSize, WalkObjectsFn) const = 0;
    virtual std::string_view fbVersion() const = 0;
};

class DbLoader : public ILoader {
public:
    explicit DbLoader(pgpool3::Pool& pool) : pool_(pool) {}
    void forEachPublishedFeatures(size_t batchSize, FeaturesFn) const override;
    void forEachWalkObjects(size_t batchSize, WalkObjectsFn) const override;
    std::string_view fbVersion() const override { return {}; }

private:
    pgpool3::Pool& pool_;
};

/**
 * The class loads data from a ecstatic datasets.
 * All changes after @param lastTxnId will be read from the database.
 */
class FbLoader : public ILoader {
public:
    FbLoader(pgpool3::Pool&,
             const std::string& mrcDatasetDir,
             const std::string& mrcFeaturesSecretDatasetDir,
             db::TId lastTxnId,
             EMappingMode);

    void forEachPublishedFeatures(size_t batchSize, FeaturesFn) const override;
    void forEachWalkObjects(size_t batchSize, WalkObjectsFn) const override;
    std::string_view fbVersion() const override;

private:
    pgpool3::Pool& pool_;
    fb::FeaturesReader defaultFeatures_;
    fb::FeaturesReader secretFeatures_;
    db::TId lastTxnId_;

    db::TIdSet loadNewPublishedFeatures(size_t batchSize, const FeaturesFn&) const;

    void loadOldPublishedFeatures(size_t batchSize,
                                  const FeaturesFn&,
                                  const db::TIdSet& newFeatureIds) const;
};

/// All changes after @param lastTxnId will be read from the database
std::unique_ptr<ILoader> makeLoader(
    pgpool3::Pool&,
    const std::string& mrcDatasetDir,
    const std::string& mrcFeaturesSecretDatasetDir,
    db::TId lastTxnId,
    EMappingMode = EMappingMode::Precharged);

}  // namespace maps::mrc::export_gen
