#include "tools.h"

#include <maps/libs/common/include/make_batches.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/metadata_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/visibility.h>

#include <set>

namespace maps::mrc::export_gen {

std::string_view insertIntoStringPool(const std::string& str)
{
    static auto pool = std::set<std::string>{};
    return *pool.insert(str).first;
}

Photo toPhoto(const db::Feature& feature, Origin origin)
{
    return Photo{
        .geodeticPos = feature.geodeticPos(),
        .sourceId = insertIntoStringPool(feature.sourceId()),
        .direction = db::direction(feature),
        .featureId = feature.id(),
        .timestamp = feature.timestamp(),
        .cameraDeviation = feature.cameraDeviation(),
        .dataset = feature.dataset(),
        .dayPart = common::getDayPart(feature.timestamp(),
                                      feature.geodeticPos().y(),
                                      feature.geodeticPos().x()),
        .graph = feature.graph(),
        .origin = origin,
        .privacy = feature.privacy(),
    };
}

void markAsPublished(const db::TIds& featureIds, pgpool3::Pool& pool)
{
    constexpr size_t BATCH_SIZE = 50000;
    size_t counter = 0;

    for (const auto& batch :
         maps::common::makeBatches(featureIds, BATCH_SIZE)) {
        auto txn = pool.masterWriteableTransaction();
        db::FeatureGateway gtw{*txn};
        auto features = gtw.loadByIds({batch.begin(), batch.end()});
        size_t skipped = 0;
        for (auto& feature : features) {
            try {
                feature.setIsPublished(true);
            }
            catch (const maps::Exception&) {
                ++skipped;
                WARN() << "feature " << feature.id() << " can't be published";
            }
        }
        gtw.update(features, db::UpdateFeatureTxn::Yes);
        txn->commit();
        counter += features.size();
        counter -= skipped;
        INFO() << counter << " features marked `published`";
    }
}

void updateMetadata(pgpool3::Pool& pool, db::TId txnId)
{
    INFO() << "updating metadata";
    auto txn = pool.masterWriteableTransaction();
    db::MetadataGateway{*txn}.upsertByKey(
        LAST_RUN_TIME,
        chrono::formatSqlDateTime(chrono::TimePoint::clock::now()));
    db::MetadataGateway{*txn}.upsertByKey(LAST_TXN_ID, std::to_string(txnId));
    txn->commit();
}

db::TId getLastFeatureTxnId(pgpool3::Pool& pool)
{
    auto txn = pool.slaveTransaction();
    auto snapshotId = sql_chemistry::SystemInformation{*txn}.getSnapshotId();
    auto featureTransactions = db::FeatureTransactionGateway{*txn}.load(
        db::table::FeatureTransaction::transactionId.less(
            snapshotId.minTransactionId),
        sql_chemistry::orderBy(db::table::FeatureTransaction::transactionId)
            .desc()
            .limit(1));
    return featureTransactions.empty()
               ? 0
               : featureTransactions.front().transactionId();
}

std::string format(size_t n)
{
    auto os = std::ostringstream{};
    os.imbue(std::locale(""));
    os << n;
    return os.str();
}

geolib3::Direction2 toDirection(const geolib3::Segment2& geoSegment)
{
    return geolib3::Direction2(convertGeodeticToMercator(geoSegment));
}

} //namespace maps::mrc::export_gen
