#include "social.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/social/magic_strings.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/parallel_for_each.h>
#include <maps/libs/common/include/make_batches.h>
#include <yandex/maps/wiki/social/feedback/commits.h>
#include <yandex/maps/wiki/social/feedback/gateway_ro.h>

using namespace std::string_view_literals;

namespace maps::mrc::feedback_stat::social {

FeedbackTasks loadFeedbackTasks(pgpool3::Pool& pool, const TIdSet& feedbackIds)
{
    FeedbackTasks result;
    auto batches = maps::common::makeBatches(feedbackIds, BATCH_SIZE);
    common::parallelForEach<MAX_THREADS>(
        batches.begin(),
        batches.end(),
        [&](std::mutex& guard, const auto& batch) {
            auto tasks =
                wiki::social::feedback::GatewayRO{*pool.slaveTransaction()}
                    .tasksByIds({batch.begin(), batch.end()});

            FeedbackTasks partialResult;
            for (const auto& task : tasks) {
                FeedbackTask feedbackTask{.id = static_cast<TId>(task.id()),
                                          .createdAt = task.createdAt(),
                                          .type = task.type()};
                if (auto taskResolved = task.resolved()) {
                    feedbackTask.verdict = taskResolved->resolution.verdict();
                }
                partialResult.push_back(feedbackTask);
            }

            std::lock_guard lock{guard};
            result.insert(
                result.end(), partialResult.begin(), partialResult.end());
            INFO() << wiki::social::sql::table::FEEDBACK_TASK << ": "
                   << result.size() << " tasks read";
        });
    return result;
}

TIdSet loadCommittedFeedbackIds(pgpool3::Pool& pool,
                                const FeedbackTasks& feedbackTasks)
{
    TIdSet feedbackIds;
    for (const auto& feedbackTask : feedbackTasks) {
        if (feedbackTask.verdict &&
            *feedbackTask.verdict ==
                wiki::social::feedback::Verdict::Accepted) {
            feedbackIds.insert(feedbackTask.id);
        }
    }

    TIdSet result;
    auto batches = maps::common::makeBatches(feedbackIds, BATCH_SIZE);
    common::parallelForEach<MAX_THREADS>(
        batches.begin(),
        batches.end(),
        [&](std::mutex& guard, const auto& batch) {
            auto commitFeedbackTasks =
                wiki::social::feedback::commitIdsByTaskIds(
                    *pool.slaveTransaction(), {batch.begin(), batch.end()});

            TIdSet partialResult;
            for (const auto& commitFeedbackTask : commitFeedbackTasks) {
                partialResult.insert(commitFeedbackTask.first);
            }

            std::lock_guard lock{guard};
            result.insert(partialResult.begin(), partialResult.end());
            INFO() << wiki::social::sql::table::COMMIT_FEEDBACK_TASK << ": "
                   << result.size() << " committed tasks read";
        });
    return result;
}

}  // namespace maps::mrc::feedback_stat::social
