#include "graph_coverage.h"
#include "strings.h"

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/algorithm/parallel_for_each.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/visibility.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/io.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/operation.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/serialization.h>

#include <maps/libs/chrono/include/days.h>
#include <maps/libs/edge_persistent_index/include/persistent_index.h>
#include <maps/libs/road_graph/include/graph.h>
#include <maps/libs/sql_chemistry/include/batch_load.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/make_batches.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/projection.h>
#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/log8/include/log8.h>

#include <mapreduce/yt/interface/client.h>
#include <pqxx/transaction_base>

#include <atomic>
#include <limits>
#include <string>
#include <unordered_set>

using namespace maps::chrono::literals;

namespace maps::mrc::graph_coverage_export {

using namespace maps::mrc::common::geometry;

namespace {

using FeaturesRange = boost::iterator_range<db::Features::iterator>;

// The input features range must be sorted according to the split predicate
template <typename SplitPredicate>
std::vector<FeaturesRange> makeRanges(const FeaturesRange& features,
                                      SplitPredicate splitPredicate)
{
    std::vector<FeaturesRange> ranges;

    auto begin = features.begin();
    while (begin != features.end()) {
        auto end = std::adjacent_find(begin, features.end(), splitPredicate);

        if (end != features.end()) {
            // Move to the second element of the pair
            ++end;
        }

        ranges.push_back(boost::make_iterator_range(begin, end));
        begin = end;
    }
    return ranges;
}

auto splitByCameraDeviation =
    [](const db::Feature& lhs, const db::Feature& rhs) {
        return lhs.cameraDeviation() != rhs.cameraDeviation();
    };

auto splitByDataset =
    [](const db::Feature& lhs, const db::Feature& rhs) {
        return lhs.dataset() != rhs.dataset();
    };

void sortByCameraDeviationAndDataset(db::Features& features)
{
    std::sort(features.begin(), features.end(),
            [](const db::Feature& lhs, const db::Feature& rhs) {
                auto lhsCamDev = db::toIntegral(lhs.cameraDeviation());
                auto rhsCamDev = db::toIntegral(rhs.cameraDeviation());

                if (lhsCamDev < rhsCamDev)
                    return true;
                if (rhsCamDev < lhsCamDev)
                    return false;
                return lhs.dataset() < rhs.dataset();
            });
}

class EdgeDataToEdgeCoverageMapper : public maps::mrc::yt::Mapper
{
public:
    void Do(maps::mrc::yt::Reader* reader, maps::mrc::yt::Writer* writer) override;

private:

    NYT::TNode toTNode(std::optional<uint8_t> dataset,
                       const db::CameraDeviation cameraDeviation,
                       int32_t geoId,
                       const EdgeData& edgeData,
                       const CoverageData& coverage,
                       double length) const;

    template <typename FeaturesIter>
    void writePolylineCoverage(std::optional<uint8_t> dataset,
                               const db::CameraDeviation cameraDeviation,
                               const EdgeData& edgeData,
                               FeaturesIter featuresBegin,
                               FeaturesIter featuresEnd,
                               maps::mrc::yt::Writer* writer)
    {
        auto mercatorGeom = geolib3::convertGeodeticToMercator(edgeData.geom);

        for (auto& coverageData
                : evalPolylineCoverage(edgeData.geom, featuresBegin, featuresEnd))
        {
            auto length = calcCoverageLength(mercatorGeom, coverageData.subpolyline);
            for (auto geoId : edgeData.geoIds) {
                writer->AddRow(toTNode(dataset, cameraDeviation, geoId,
                                       edgeData, coverageData, length));
            }
        }
    }
};

void EdgeDataToEdgeCoverageMapper::Do(maps::mrc::yt::Reader* reader, maps::mrc::yt::Writer* writer)
{
    for (; reader->IsValid(); reader->Next()) {
        auto row = reader->GetRow();
        auto edgeData = maps::mrc::yt::deserialize<EdgeData>(row);

        sortByCameraDeviationAndDataset(edgeData.features);

        auto features = boost::make_iterator_range(edgeData.features.begin(),
                                                   edgeData.features.end());

        for (const auto& range : makeRanges(features, splitByCameraDeviation)) {
            const auto cameraDeviation = range.begin()->cameraDeviation();
            // Cumulative coverage for all datasets
            writePolylineCoverage(std::nullopt, cameraDeviation, edgeData,
                                  range.begin(), range.end(), writer);

            // Coverage per dataset
            for (auto& subRange : makeRanges(range, splitByDataset)) {
                const auto& dataset = subRange.begin()->dataset();
                writePolylineCoverage(static_cast<uint8_t>(dataset), cameraDeviation, edgeData,
                                      subRange.begin(), subRange.end(), writer);
            }
        }
    }
}

NYT::TNode
EdgeDataToEdgeCoverageMapper::toTNode(std::optional<uint8_t> dataset,
                                      const db::CameraDeviation cameraDeviation,
                                      int32_t geoId,
                                      const EdgeData& edgeData,
                                      const CoverageData& coverage,
                                      double length) const
{
    return NYT::TNode::CreateMap()
        (col::DATASET, yt::serialize(dataset))
        (col::CAMERA_DEVIATION, db::toIntegral(cameraDeviation))
        (col::FC, edgeData.fc)
        (col::IS_TOLL, static_cast<uint8_t>(edgeData.isToll))
        (col::LENGTH, length)
        (col::GEO_ID, geoId)
        (col::PRIVATE_AREA, static_cast<uint8_t>(edgeData.privateArea))
        (col::ACTUALIZATION_DATE, coverage.actualizationDate.count())
        (col::FROM_DATE, coverage.from.count())
        (col::TO_DATE, coverage.to.count())
        (col::GRAPH_TYPE, static_cast<uint8_t>(edgeData.graphType));
}


REGISTER_MAPPER(EdgeDataToEdgeCoverageMapper);

class AggregateByAttributesReducer: public maps::mrc::yt::Reducer
{
public:
    void Do(maps::mrc::yt::Reader* reader, maps::mrc::yt::Writer* writer) override
    {
        double length = 0;
        std::optional<NYT::TNode> result;
        for (; reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();
            if (!result) {
                result = row;
            }
            length += row[col::LENGTH].As<double>();
        }
        ASSERT(result);
        result.value()[col::LENGTH] = length;
        writer->AddRow(result.value());
    }
};

REGISTER_REDUCER(AggregateByAttributesReducer);

geolib3::PointsVector
evalIntersectedPoints(const geolib3::Polygon2& convexPolygon,
                      const geolib3::Segment2& segment)
{
    std::set<geolib3::Point2> points;
    auto ring = convexPolygon.exteriorRing();
    for (size_t i = 0; i < ring.segmentsNumber(); ++i) {
        for (auto point : geolib3::intersection(ring.segmentAt(i), segment))
        {
            points.insert(point);
        }
    }
    return {points.begin(), points.end()};
}

std::optional<geolib3::Point2>
getPointNotEqualTo(const geolib3::PointsVector& points,
                   const geolib3::Point2& point)
{
    for (const auto& onePoint : points) {
        if (onePoint != point) {
            return onePoint;
        }
    }
    return std::nullopt;
}

} // namespace

double calcCoverageLength(
    const geolib3::Polyline2& mercatorGeom,
    const SubPolyline& subpolyline)
{
    const auto ratio = geolib3::MercatorRatio::fromMercatorPoint(mercatorGeom.pointAt(0));
    double length = 0.;
    if (subpolyline.begin().segmentIdx() == subpolyline.end().segmentIdx()) {
        length += ratio.toMeters(geolib3::length(mercatorGeom.segmentAt(subpolyline.begin().segmentIdx()))) *
            (subpolyline.end().segmentRelPosition() -
                subpolyline.begin().segmentRelPosition());
    } else {
        length += ratio.toMeters(geolib3::length(mercatorGeom.segmentAt(subpolyline.begin().segmentIdx()))) *
            (1. - subpolyline.begin().segmentRelPosition());

        for (size_t i = subpolyline.begin().segmentIdx() + 1;
                i < subpolyline.end().segmentIdx(); ++i)
        {
            length += ratio.toMeters(geolib3::length(mercatorGeom.segmentAt(i)));
        }

        length += ratio.toMeters(geolib3::length(mercatorGeom.segmentAt(subpolyline.end().segmentIdx()))) *
            subpolyline.end().segmentRelPosition();
    }

    return length;
}

db::Features
searchFeaturesForEdge(const geolib3::Polyline2& edgeGeometry,
                      const FeaturesIndex& featuresIndex)
{
    db::Features features;
    geolib3::BoundingBox searchBbox = db::addVisibilityMargins(edgeGeometry.boundingBox());
    db::TIds featureIds = featuresIndex.featureIdsInBbox(searchBbox);
    for (auto featureId : featureIds) {
        auto feature = featuresIndex.featureById(featureId);
        if (!db::isVisible(feature, edgeGeometry)) {
            continue;
        }
        features.push_back(std::move(feature));
    }
    return features;
}

std::optional<SubSegment>
evalSegmentCoverage(
    const geolib3::Segment2& geoSegment,
    geolib3::Direction2 direction,
    const geolib3::Polygon2& fieldOfView)
{
    const auto ANGLE_DIFF_THRESHOLD = geolib3::PI / 4;

    auto angle = geolib3::angleBetween(direction, geolib3::Direction2(geoSegment));
    if (angle >= ANGLE_DIFF_THRESHOLD) {
        return std::nullopt;
    }

    if (!geolib3::spatialRelation(fieldOfView, geoSegment, geolib3::Intersects)) {
        return std::nullopt ;
    }

    auto mercatorSegment = geolib3::convertGeodeticToMercator(geoSegment);
    auto mercatorView = geolib3::convertGeodeticToMercator(fieldOfView);

    bool intersectsStart =
        geolib3::spatialRelation(mercatorView, mercatorSegment.start(), geolib3::Intersects);
    bool intersectsEnd =
        geolib3::spatialRelation(mercatorView, mercatorSegment.end(), geolib3::Intersects);

    if (intersectsStart && intersectsEnd) {
        return SubSegment{MIN_REL_POS, MAX_REL_POS};
    }
    auto intersectionPoints = evalIntersectedPoints(mercatorView, mercatorSegment);
    if (intersectionPoints.empty()) {
        return std::nullopt;
    }

    if (intersectsStart) {
        auto otherPoint = getPointNotEqualTo(intersectionPoints, mercatorSegment.start());
        if (!otherPoint.has_value()) {
            return std::nullopt;
        }
        return SubSegment{MIN_REL_POS, geolib3::projectionFactor(mercatorSegment, otherPoint.value())};
    } else if (intersectsEnd) {
        auto otherPoint = getPointNotEqualTo(intersectionPoints, mercatorSegment.end());
        if (!otherPoint.has_value()) {
            return std::nullopt;
        }
        return SubSegment{geolib3::projectionFactor(mercatorSegment, otherPoint.value()), MAX_REL_POS};
    } else if (intersectionPoints.size() == 1) {
        return std::nullopt;
    }
    ASSERT(intersectionPoints.size() >= 2);
    double relPosA = geolib3::projectionFactor(mercatorSegment, intersectionPoints.front());
    double relPosB = geolib3::projectionFactor(mercatorSegment, intersectionPoints.back());
    return SubSegment{std::min(relPosA, relPosB), std::max(relPosA, relPosB)};
}

chrono::Days maxDate()
{
    return chrono::Days(std::numeric_limits<int16_t>::max());
}

CoverageDataVec evalPolylineCoverage(
    const geolib3::Polyline2& polyline,
    db::Features::const_iterator begin,
    db::Features::const_iterator end)
{
    std::map<chrono::Days, SubPolylines> dayToSubpolylines;

    for (auto featureIt = begin; featureIt != end; ++featureIt) {
        chrono::Days date = std::chrono::duration_cast<chrono::Days>(
                featureIt->timestamp().time_since_epoch());
        evalPolylineCoverage(
            polyline, *featureIt, std::back_inserter(dayToSubpolylines[date]));
    }

    CoverageDataVec result;
    std::list<CoverageData> current;

    for(auto& [date, subpolylines] : dayToSubpolylines) {
        subpolylines = merge(std::move(subpolylines));
        auto currentIt = current.begin();

        for (auto subpolyline : subpolylines) {
            if ((currentIt == current.end()) || (precedes(subpolyline, currentIt->subpolyline))) {
                current.insert(currentIt, CoverageData{date, 0_days, date, subpolyline});
                continue;
            }

            while ((currentIt != current.end()) && (precedes(currentIt->subpolyline, subpolyline))) {
                ++currentIt;
            }

            while ((currentIt != current.end()) && (intersects(subpolyline, currentIt->subpolyline))) {
                auto intSubpolyline = intersection(subpolyline, currentIt->subpolyline);
                result.push_back({
                    currentIt->from, date, currentIt->actualizationDate, {intSubpolyline}}
                );

                if (intSubpolyline == currentIt->subpolyline) {
                    currentIt = current.erase(currentIt);
                    continue;
                } else if (intSubpolyline.begin() == currentIt->subpolyline.begin()) {
                    currentIt->subpolyline.setBegin(intSubpolyline.end());
                    continue;
                } else if (intSubpolyline.end() == currentIt->subpolyline.end()) {
                    currentIt->subpolyline.setEnd(intSubpolyline.begin());
                } else {
                    current.insert(std::next(currentIt),
                        CoverageData{currentIt->from, 0_days, currentIt->actualizationDate,
                            {intSubpolyline.end(), currentIt->subpolyline.end()}});
                    currentIt->subpolyline.setEnd(intSubpolyline.begin());
                }

                ++currentIt;
            }

            ASSERT(currentIt == current.end() || precedes(subpolyline, currentIt->subpolyline));
            ASSERT(currentIt == current.begin() || precedes(std::prev(currentIt)->subpolyline, subpolyline));

            current.insert(currentIt, CoverageData{date, 0_days, date, subpolyline});
        }
    }

    const chrono::Days MAX_DATE = maxDate();

    for (auto& coverageData : current) {
        result.push_back(
            {coverageData.from, MAX_DATE, coverageData.actualizationDate,
                coverageData.subpolyline}
        );
    }

    return result;
}


void convertToCoverage(NYT::IClientBase& ytClient,
                       const TString& fromTable,
                       const TString& toTable)
{
    INFO() << "Converting to coverages " << fromTable
        << " -> " << toTable;

    const auto spec = NYT::TNode::CreateMap()
        ("title", "Calculate edges coverage");

    ytClient.MapReduce(
        NYT::TMapReduceOperationSpec()
            .ReduceBy({col::DATASET, col::CAMERA_DEVIATION,
                col::FROM_DATE, col::TO_DATE, col::ACTUALIZATION_DATE,
                col::GEO_ID, col::FC, col::IS_TOLL, col::PRIVATE_AREA,
                col::GRAPH_TYPE})
            .AddInput<NYT::TNode>(fromTable)
            .AddOutput<NYT::TNode>(toTable),
        new EdgeDataToEdgeCoverageMapper(),
        new AggregateByAttributesReducer(),
        NYT::TOperationOptions()
            .Spec(spec)
    );
    INFO() << "Finished convertion to coverages";
}


} // namespace maps::mrc::graph_coverage_export
