#include <maps/infra/yacare/include/parse.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/wiki_config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/metadata_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/graph_coverage_export/lib/coverage_export.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/graph_coverage_export/lib/strings.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>

#include <pqxx/transaction_base>

#include <optional>
#include <string>

using namespace maps::mrc::graph_coverage_export;

namespace {

void updateMetadata(maps::pgpool3::Pool& pool)
{
    INFO() << "Updating metadata";
    auto txn = pool.masterWriteableTransaction();

    maps::mrc::db::MetadataGateway{*txn}.upsertByKey(
        LAST_RUN_TIME,
        maps::chrono::formatSqlDateTime(maps::chrono::TimePoint::clock::now()));

    txn->commit();
}

} // namespace

int main(int argc, const char** argv) try {
    NYT::Initialize(argc, argv);
    using namespace maps;
    log8::setLevel(log8::Level::INFO);
    maps::cmdline::Parser parser;

    auto configPath = parser.string("config")
            .help("path to mrc config");

    auto wikiConfigDirPath = parser.string("wiki-config-dir")
            .defaultValue(maps::mrc::common::WIKI_TEMPLATE_CONFIG_DIR)
            .help("path to services config template for wikimap");

    auto wikiConfigPath =
        parser.string("wiki-config")
            .help("path to services config for wikimap");

    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru");

    auto mrcDatasetPath = parser.string("mrc-dataset-path")
            .help("Path to dataset yandex-maps-mrc")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc/");

    auto mrcFeaturesSecretDatasetPath = parser.string("mrc-features-secret-dataset-path")
            .help("Path to dataset yandex-maps-mrc")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-features-secret/");

    auto mrcRoadGraphPath = parser.string("mrc-road-graph")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-graph-pro/")
            .help("path to mrc road graph directory");

    auto mrcPedestrianGraphPath = parser.string("mrc-pedestrian-graph")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-pedestrian-graph-pro/")
            .help("path to mrc pedestrian graph directory");

    auto geoDataPath = parser.string("geodata-path")
            .defaultValue("/var/cache/geobase/geodata6.bin")
            .help("path to coverage for geoid");

    auto geodataPatchYtDirParam = parser.string("geodata-patch-yt-dir")
            .defaultValue("//home/maps/core/nmaps/analytics/geo-data")
            .help("YT dir with additional geobase regions");

    auto syslog = parser.string("syslog-tag")
            .help("redirect log output to syslog with given tag");

    parser.parse(argc, const_cast<char**>(argv));

    if (syslog.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslog));
    }

    INFO() << "Starting";
    const auto config =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, configPath);
    auto wikiConfig = wikiConfigPath.defined()
        ? maps::wiki::common::ExtendedXmlDoc(wikiConfigPath)
        : maps::mrc::common::makeWikiConfig(wikiConfigDirPath);

    maps::wiki::common::PoolHolder mrcPoolHolder =
        config.makePoolHolder(maps::mrc::common::LONG_READ_DB_ID,
                              maps::mrc::common::LONG_READ_POOL_ID);
    maps::pgp3utils::PgAdvisoryXactMutex mutex(mrcPoolHolder.pool(),
        static_cast<int64_t>(maps::mrc::common::LockId::GraphCoverageExport));
    if (!mutex.try_lock()) {
        INFO() << "Another process is ongoing";
        return EXIT_SUCCESS;
    }

    TString ytDir = TString(config.externals().yt().path()) + "/graph_coverage_export";
    coverageExport(
        config,
        wikiConfig,
        mrcDatasetPath,
        mrcFeaturesSecretDatasetPath,
        geoDataPath,
        geodataPatchYtDirParam,
        mrcRoadGraphPath,
        mrcPedestrianGraphPath,
        ytDir,
        /*bbox=*/std::nullopt);

    updateMetadata(mrcPoolHolder.pool());
    INFO() << "Done";

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    FATAL() << e;
    return EXIT_FAILURE;
} catch (const yexception& e) {
    FATAL() << e.what();

    if (e.BackTrace()) {
        FATAL() << e.BackTrace()->PrintToString();
    }

    return EXIT_FAILURE;
} catch (const std::exception& e) {
    FATAL() << e.what();
    return EXIT_FAILURE;
} catch (...) {
    FATAL() << "Caught unknown exception";
    return EXIT_FAILURE;
}
