#include <maps/wikimap/mapspro/services/mrc/long_tasks/graph_coverage_export/lib/graph_coverage.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>

#include <maps/libs/chrono/include/days.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/heading.h>
#include <maps/libs/geolib/include/segment.h>

#include <library/cpp/testing/gmock_in_unittest/gmock.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <optional>
#include <ostream>
#include <vector>

using namespace ::testing;
using namespace maps::chrono::literals;

namespace std {

template<typename C>
std::ostream& operator<<(std::ostream& out, const std::optional<C>& optObj)
{
    if (optObj.has_value()) {
        out << optObj.value();
    } else {
        out << "std::nullopt";
    }
    return out;
}

std::ostream& operator<<(std::ostream& out, maps::chrono::Days date)
{
    out << date.count() << "_days";
    return out;
}

} // namespace std

namespace maps::mrc::graph_coverage_export {

using namespace maps::mrc::common::geometry;

constexpr double EPS = 0.001;
constexpr double VISIBILITY_METERS = 30.;
constexpr chrono::Days MAX_DATE = chrono::Days(std::numeric_limits<int16_t>::max());

namespace {

bool Near(const std::optional<SubSegment>& one,
          const std::optional<SubSegment>& other,
          double eps)
{
    if (one.has_value() != other.has_value()) {
        return false;
    }

    if (!one.has_value()) {
        return true;
    }

    return (std::abs(one->begin() - other->begin()) < eps) &&
        (std::abs(one->end() - other->end()) < eps);
}


} // namespace

Y_UNIT_TEST_SUITE(eval_coverage_tests)
{
    Y_UNIT_TEST(eval_segment_coverage_test)
    {
        geolib3::Point2 pos{0, 0};
        db::Feature feature("sourceId", pos, geolib3::Heading(0.),
            "2018-01-01 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);
        geolib3::Direction2 direction(feature.heading());
        auto view = db::fieldOfView(db::getRay(feature));

        EXPECT_EQ(
            evalSegmentCoverage(
                geolib3::Segment2(pos, geolib3::fastGeoShift(pos, {VISIBILITY_METERS, 0})),
                direction,
                view
            ),
            std::nullopt
        );

        EXPECT_EQ(
            evalSegmentCoverage(
                geolib3::Segment2(pos, geolib3::fastGeoShift(pos, {0, VISIBILITY_METERS})),
                direction,
                view
            ),
            SubSegment(0, 1)
        );

        EXPECT_EQ(
            evalSegmentCoverage(
                geolib3::Segment2(pos, geolib3::fastGeoShift(pos, {0, 0.5 * VISIBILITY_METERS})),
                direction,
                view
            ),
            SubSegment(0, 1)
        );

        EXPECT_PRED3(
            Near,
            evalSegmentCoverage(
                geolib3::Segment2(pos, geolib3::fastGeoShift(pos, {0, 2 * VISIBILITY_METERS})),
                direction,
                view
            ),
            SubSegment(0, 0.5),
            EPS
        );

        EXPECT_EQ(
            evalSegmentCoverage(
                geolib3::reverse(
                    geolib3::Segment2(pos, geolib3::fastGeoShift(pos, {0, VISIBILITY_METERS}))
                ),
                direction,
                view
            ),
            std::nullopt
        );

        EXPECT_EQ(
            evalSegmentCoverage(
                geolib3::Segment2(
                    geolib3::fastGeoShift(pos, {0, 1.}),
                    geolib3::fastGeoShift(pos, {0, VISIBILITY_METERS})
                ),
                direction,
                view
            ),
            SubSegment(0, 1)
        );

        EXPECT_PRED3(
            Near,
            evalSegmentCoverage(
                geolib3::Segment2(
                    geolib3::fastGeoShift(pos, {0, -1.}),
                    geolib3::fastGeoShift(pos, {0, VISIBILITY_METERS -1.})
                ),
                direction,
                view
            ),
            SubSegment(1. / VISIBILITY_METERS, 1),
            EPS
        );

        EXPECT_PRED3(
            Near,
            evalSegmentCoverage(
                geolib3::Segment2(
                    geolib3::fastGeoShift(pos, {1, 0}),
                    geolib3::fastGeoShift(pos, {1, VISIBILITY_METERS})
                ),
                direction,
                view
            ),
            SubSegment(0.033333, 1),
            EPS
        );

        EXPECT_PRED3(
            Near,
            evalSegmentCoverage(
                geolib3::Segment2(
                    geolib3::fastGeoShift(pos, {1, 0}),
                    geolib3::fastGeoShift(pos, {1, 2 * VISIBILITY_METERS})
                ),
                direction,
                view
            ),
            SubSegment(0.01666, 0.5),
            EPS
        );
    } // Y_UNIT_TEST(eval_segment_coverage_test)

    Y_UNIT_TEST(eval_polyline_coverage_test_full_overlap)
    {
        geolib3::Point2 pos{0, 0};
        db::Feature feature1("sourceId", pos, geolib3::Heading(0.),
            "2018-01-01 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto feature1Date =
            std::chrono::duration_cast<chrono::Days>(
                feature1.timestamp().time_since_epoch());

        db::Feature feature2("sourceId", pos, geolib3::Heading(0.),
            "2018-01-02 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto feature2Date =
            std::chrono::duration_cast<chrono::Days>(
                feature2.timestamp().time_since_epoch());

        db::Features features = {feature1, feature2};
        EXPECT_THAT(
            evalPolylineCoverage(
                geolib3::Polyline2{
                    {
                        pos,
                        geolib3::fastGeoShift(pos, {0, 0.5}),
                        geolib3::fastGeoShift(pos, {0, 1})
                    }
                },
                features.begin(), features.end()
            ),
            ElementsAre(
                CoverageData{feature1Date, feature2Date, feature1Date, {{0, 0}, {1, 1}}},
                CoverageData{feature2Date, MAX_DATE, feature2Date, {{0, 0}, {1, 1}}}
            )
        );
    }

    Y_UNIT_TEST(eval_polyline_coverage_test_partial_overlap)
    {
        geolib3::Point2 pos1{0, 0};
        db::Feature feature1("sourceId", pos1, geolib3::Heading(0.),
            "2018-01-01 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto feature1Date =
            std::chrono::duration_cast<chrono::Days>(
                feature1.timestamp().time_since_epoch());

        auto pos2 = geolib3::fastGeoShift(pos1, {0, 1});

        db::Feature feature2("sourceId", pos2, geolib3::Heading(0.),
            "2018-01-02 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto feature2Date =
            std::chrono::duration_cast<chrono::Days>(
                feature2.timestamp().time_since_epoch());

        db::Features features = {feature1, feature2};
        EXPECT_THAT(
            evalPolylineCoverage(
                geolib3::Polyline2{
                    {
                        pos1,
                        pos2,
                        geolib3::fastGeoShift(pos2, {0, 1})
                    }
                },
                features.begin(), features.end()
            ),
            ElementsAre(
                CoverageData{feature1Date, feature2Date, feature1Date, {{1, 0}, {1, 1}}},
                CoverageData{feature1Date, MAX_DATE, feature1Date, {{0, 0}, {0, 1}}},
                CoverageData{feature2Date, MAX_DATE, feature2Date, {{1, 0}, {1, 1}}}
            )
        );
    }

    Y_UNIT_TEST(eval_polyline_coverage_test_partial_overlap2)
    {
        //              0----0.25-----0.5----0.75-------1
        // 2018-01-01    --------        --------    --------
        // 2018-01-02               --------

        geolib3::Point2 pos1{0, 0};
        db::Feature feature1("sourceId", pos1, geolib3::Heading(0.),
            "2018-01-01 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto feature1Date =
            std::chrono::duration_cast<chrono::Days>(
                feature1.timestamp().time_since_epoch());

        auto pos2 = geolib3::fastGeoShift(pos1, {0, VISIBILITY_METERS * 2});

        db::Feature feature2("sourceId", pos2, geolib3::Heading(0.),
            "2018-01-01 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto pos3 = geolib3::fastGeoShift(pos1, {0, VISIBILITY_METERS * 3.5});

        db::Feature feature3("sourceId", pos3, geolib3::Heading(0.),
            "2018-01-01 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto pos4 = geolib3::fastGeoShift(pos1, {0, VISIBILITY_METERS * 1.5});

        db::Feature feature4("sourceId", pos4, geolib3::Heading(0.),
            "2018-01-02 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto feature4Date =
            std::chrono::duration_cast<chrono::Days>(
                feature4.timestamp().time_since_epoch());

        db::Features features = {feature1, feature2, feature3, feature4};
        EXPECT_THAT(
            evalPolylineCoverage(
                geolib3::Polyline2{
                    geolib3::PointsVector{
                        pos1,
                        geolib3::fastGeoShift(pos1, {0, VISIBILITY_METERS * 4})
                    }
                },
                features.begin(), features.end()
            ),
            ElementsAre(
                    CoverageData{feature1Date, feature4Date, feature1Date, {{0, 0.5}, {0, 0.625}}},
                    CoverageData{feature1Date, MAX_DATE, feature1Date, {{0, 0}, {0, 0.25}}},
                    CoverageData{feature4Date, MAX_DATE, feature4Date, {{0, 0.375}, {0, 0.625}}},
                    CoverageData{feature1Date, MAX_DATE, feature1Date, {{0, 0.625}, {0, 0.75}}},
                    CoverageData{feature1Date, MAX_DATE, feature1Date, {{0, 0.875}, {0, 1}}}
            )
        );
    }

    Y_UNIT_TEST(eval_polyline_coverage_test_covered)
    {
        //              0----0.25-----0.5----0.75-------1
        // 2018-01-01    --------------------------------
        // 2018-01-02            ----------------
        geolib3::Point2 pos1{0, 0};
        db::Feature feature1("sourceId", pos1, geolib3::Heading(0.),
            "2018-01-01 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto feature1Date =
            std::chrono::duration_cast<chrono::Days>(
                feature1.timestamp().time_since_epoch());

        auto pos2 = geolib3::fastGeoShift(pos1, {0, VISIBILITY_METERS});

        db::Feature feature2("sourceId", pos2, geolib3::Heading(0.),
            "2018-01-01 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto pos3 = geolib3::fastGeoShift(pos1, {0, VISIBILITY_METERS * 0.5});

        db::Feature feature3("sourceId", pos3, geolib3::Heading(0.),
            "2018-01-02 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto feature3Date =
            std::chrono::duration_cast<chrono::Days>(
                feature3.timestamp().time_since_epoch());

        db::Features features = {feature1, feature2, feature3};
        EXPECT_THAT(
            evalPolylineCoverage(
                geolib3::Polyline2{
                    geolib3::PointsVector{
                        pos1,
                        geolib3::fastGeoShift(pos1, {0, VISIBILITY_METERS * 2})
                    }
                },
                features.begin(), features.end()
            ),
            ElementsAre(
                    CoverageData{feature1Date, feature3Date, feature1Date, {{0, 0.25}, {0, 0.75}}},
                    CoverageData{feature1Date, MAX_DATE, feature1Date, {{0, 0}, {0, 0.25}}},
                    CoverageData{feature3Date, MAX_DATE, feature3Date, {{0, 0.25}, {0, 0.75}}},
                    CoverageData{feature1Date, MAX_DATE, feature1Date, {{0, 0.75}, {0, 1}}}
            )
        );
    }

    Y_UNIT_TEST(eval_polyline_coverage_test_covered2)
    {
        //              0----0.25-----0.5----0.75-------1
        // 2018-01-01            ----------------
        // 2018-01-02    --------------------------------

        geolib3::Point2 pos{0, 0};

        db::Feature feature1("sourceId", geolib3::fastGeoShift(pos, {0, VISIBILITY_METERS * 0.5}), geolib3::Heading(0.),
            "2018-01-01 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto feature1Date =
            std::chrono::duration_cast<chrono::Days>(
                feature1.timestamp().time_since_epoch());

        db::Feature feature2("sourceId", pos, geolib3::Heading(0.),
            "2018-01-02 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        auto feature2Date =
            std::chrono::duration_cast<chrono::Days>(
                feature2.timestamp().time_since_epoch());

        db::Feature feature3("sourceId", geolib3::fastGeoShift(pos, {0, VISIBILITY_METERS}), geolib3::Heading(0.),
            "2018-01-02 00:00:00", {"mdskey", "path"}, db::Dataset::Agents);

        db::Features features{feature1, feature2, feature3};
        EXPECT_THAT(
            evalPolylineCoverage(
                geolib3::Polyline2{
                    geolib3::PointsVector{
                        pos,
                        geolib3::fastGeoShift(pos, {0, VISIBILITY_METERS * 2})
                    }
                },
                features.begin(), features.end()
            ),
            ElementsAre(
                    CoverageData{feature1Date, feature2Date, feature1Date, {{0, 0.25}, {0, 0.75}}},
                    CoverageData{feature2Date, MAX_DATE, feature2Date, {{0, 0}, {0, 1}}}
            )
        );
    }

} // Y_UNIT_TEST_SUITE(eval_coverage_tests)

} // namespace maps::mrc::graph_coverage_export
