#include <maps/wikimap/mapspro/services/mrc/long_tasks/graph_coverage_export/lib/context.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/graph_coverage_export/lib/coverage_export.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/graph_coverage_export/lib/features_index.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/graph_coverage_export/lib/graph_coverage.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/export_gen/lib/exporter.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/write.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/coverage_rtree_writer.h>
#include <maps/wikimap/mapspro/services/mrc/libs/geobase/include/geobase.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/version.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/coverage_rtree_reader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/serialization.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/mock_loader.h>

#include <maps/libs/chrono/include/days.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/introspection/include/comparison.h>
#include <maps/libs/introspection/include/stream_output.h>
#include <maps/libs/road_graph/include/graph.h>
#include <maps/libs/succinct_rtree/include/rtree.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/heading.h>
#include <maps/libs/geolib/include/segment.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/mrc/unittest/utils.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>

#include <mapreduce/yt/tests/yt_unittest_lib/yt_unittest_lib.h>

#include <library/cpp/geobase/lookup.hpp>
#include <library/cpp/testing/gmock_in_unittest/gmock.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <list>
#include <optional>
#include <ostream>
#include <vector>

using namespace ::testing;

namespace maps::mrc::graph_coverage_export {

namespace {

using maps::introspection::operator==;
using maps::introspection::operator<<;

constexpr auto GRAPH_TYPE = db::GraphType::Road;
constexpr int64_t MAX_DATE = std::numeric_limits<int16_t>::max();
const std::string GRAPH_DATA_ROOT = "maps/data/test/graph4/";
const geolib3::BoundingBox BBOX({37.466925, 55.351466}, {37.569892, 55.390929});

using Playground = unittest::WithUnittestConfig<unittest::DatabaseFixture>;

Playground& playground()
{
    static Playground playground;
    return playground;
}


struct MainFeatureData {
    geolib3::Point2 geodeticPos;
    geolib3::Heading heading;
    std::string timestamp;
    db::Dataset dataset;
    db::CameraDeviation cameraDeviation = db::CameraDeviation::Front;
};


class Fixture: public NUnitTest::TBaseFixture,
               public ContextBase
{
public:
    Fixture()
        : graph_(BinaryPath(GRAPH_DATA_ROOT + "road_graph.fb"))
        , graphRTree_(BinaryPath(GRAPH_DATA_ROOT + "rtree.fb"), graph_)
        , graphPersistentIndex_(BinaryPath(GRAPH_DATA_ROOT + "edges_persistent_index.fb"))
        , geobase_("geodata6.bin")
        , ytClientPtr_(NYT::NTesting::CreateTestClient())
        , privateAreaIndex_(objectLoader_)
    {
        playground().postgres().truncateTables();
    }

    void insertData(const std::vector<MainFeatureData>& featureData)
    {
        db::Features features;
        features.reserve(featureData.size());
        for (const auto& data : featureData) {
            features
                .emplace_back("SOURCE_ID",
                              data.geodeticPos,
                              data.heading,
                              data.timestamp,
                              mds::Key{"123", "MDS_PATH"},
                              data.dataset)
                .setSize(6, 9)
                .setAutomaticShouldBePublished(true)
                .setIsPublished(true)
                .setCameraDeviation(data.cameraDeviation);
        }
        auto txn = pool().masterWriteableTransaction();
        db::FeatureGateway(*txn).insert(features);
        txn->commit();
    }

    NYT::IClientBase& ytClient() { return *ytClientPtr_; }
    pgpool3::Pool& pool() { return playground().pool(); }

    const road_graph::Graph& graph() const override { return graph_; }
    const succinct_rtree::Rtree& graphRTree() const override { return graphRTree_; }
    const road_graph::PersistentIndex& graphPersistentIndex() const override { return graphPersistentIndex_; }
    db::GraphType graphType() const override { return GRAPH_TYPE; }
    const geobase::GeobaseInterface& geobase() const override { return geobase_; }
    const PrivateAreaIndex& privateAreaIndex() const override { return privateAreaIndex_; }

private:
    road_graph::Graph graph_;
    succinct_rtree::Rtree graphRTree_;
    road_graph::PersistentIndex graphPersistentIndex_;
    geobase::Geobase6Adapter geobase_;
    NYT::IClientPtr ytClientPtr_;
    object::MockLoader objectLoader_;
    PrivateAreaIndex privateAreaIndex_;
};

struct EdgeDataSummary
{
    int64_t fc;
    bool isToll;
    size_t geoIdsNum;
    size_t featuresNum;
    db::GraphType graphType;

    template<typename T>
    static auto introspect(T& o) {
        return std::tie(o.fc, o.isToll, o.geoIdsNum, o.featuresNum, o.graphType);
    }
};

std::list<EdgeDataSummary>
readEdgeDataSummary(NYT::TTableReader<NYT::TNode>& reader)
{
    std::list<EdgeDataSummary> result;
    for (; reader.IsValid(); reader.Next()) {
        auto& row = reader.GetRow();
        result.push_back(
            EdgeDataSummary{
                row["fc"].IntCast<int64_t>(),
                row["is_toll"].AsBool(),
                row["geo_ids"].AsList().size(),
                row["features"].AsList().size(),
                db::GraphType(row["graph_type"].IntCast<int64_t>())
            }
        );
    }
    return result;
}

struct CoverageDataSummary
{
    int64_t from;
    int64_t to;
    int64_t actualizationDate;
    int64_t fc;
    int64_t isToll;
    int64_t geoId;
    std::optional<db::Dataset> dataset;
    double length;
    int64_t cameraDeviation;
    db::GraphType graphType;

    template<typename T>
    static auto introspect(T& o) {
        return std::tie(o.from, o.to, o.actualizationDate,
            o.fc, o.isToll, o.geoId, o.dataset, o.length, o.cameraDeviation, o.graphType);
    }
};


bool operator==(const CoverageDataSummary& one, const CoverageDataSummary& other)
{
    return one.from == other.from &&
        one.to == other.to &&
        one.actualizationDate == other.actualizationDate &&
        one.fc == other.fc &&
        one.isToll == other.isToll &&
        one.geoId == other.geoId &&
        one.dataset == other.dataset &&
        std::abs(one.length - other.length) < 0.5 &&
        one.cameraDeviation == other.cameraDeviation &&
        one.graphType == other.graphType;
}


MATCHER(DatedCoverageEdgeDataApproxEq, "")
{
    const DatedCoverageEdgeData& one = std::get<0>(arg);
    const DatedCoverageEdgeData& other = std::get<1>(arg);
    const double EPS = 0.01;

    DatedCoverageEdgeData copyOne = one;
    DatedCoverageEdgeData copyOther = other;
    copyOne.coverageEdgeData.length = 0;
    copyOther.coverageEdgeData.length = 0;

    return
        testing::ExplainMatchResult(
            testing::DoubleNear(one.coverageEdgeData.length, EPS),
            other.coverageEdgeData.length,
            result_listener) &&
        testing::ExplainMatchResult(
            testing::Eq(copyOne), copyOther, result_listener);
};


std::list<CoverageDataSummary>
readCoverageDataSummary(NYT::TTableReader<NYT::TNode>& reader)
{
    std::list<CoverageDataSummary> result;
    for (; reader.IsValid(); reader.Next()) {
        auto& row = reader.GetRow();

        std::optional<db::Dataset> dataset;
        if (!row["dataset"].IsNull()) {
            dataset = static_cast<db::Dataset>(row["dataset"].IntCast<int64_t>());
        }

        result.push_back(
            CoverageDataSummary{
                row["from_date"].IntCast<int64_t>(),
                row["to_date"].IntCast<int64_t>(),
                row["actualization_date"].IntCast<int64_t>(),
                row["fc"].IntCast<int64_t>(),
                row["is_toll"].IntCast<int64_t>(),
                row["geo_id"].IntCast<int64_t>(),
                dataset,
                row["length"].AsDouble(),
                row["camera_deviation"].IntCast<int64_t>(),
                db::GraphType(row["graph_type"].IntCast<int64_t>())
            }
        );
    }
    return result;
}

struct FbExportInfo {
    std::string mrcDatasetPath;
    std::string mrcFeaturesSecretDatasetPath;
    std::string mrcGraphCoveragePath;
};

void generateGraphCoverage(
    pgpool3::Pool& pgpool,
    const road_graph::Graph& graph,
    const std::string& outputPath)
{
    auto ctx = export_gen::Context(BinaryPath(GRAPH_DATA_ROOT), db::GraphType::Road);
    auto features = db::FeatureGateway(*pgpool.slaveTransaction()).load();
    auto photos = export_gen::Photos{};
    for (const auto& feature: features) {
        photos.push_back(export_gen::toPhoto(feature));
    }
    auto [coverageGraph, _] = export_gen::makeGraphSummary(
        ctx, photos, export_gen::makeTrackPointProvider(pgpool));
    coverageGraph.mrcVersion = fb::makeVersion(std::chrono::system_clock::now());
    fb::writeToFile(coverageGraph, outputPath + "/graph_coverage.fb");
    fb::writeCoverageRtreeToDir(graph, coverageGraph, outputPath);
}

FbExportInfo exportToFb(pgpool3::Pool& pgpool, const road_graph::Graph& graph)
{
    std::string mrcPath = "mrc_export";
    std::string featuresSecretPath = "mrc_features_secret";
    std::string graphCoveragePath = "mrc_graph_coverage";

    std::filesystem::create_directories(mrcPath);
    std::filesystem::create_directories(featuresSecretPath);
    std::filesystem::create_directories(graphCoveragePath);

    export_gen::generateExport(
        pgpool,
        mrcPath,
        featuresSecretPath,
        fb::makeVersion(std::chrono::system_clock::now()));

    generateGraphCoverage(pgpool, graph, graphCoveragePath);

    return {mrcPath, featuresSecretPath, graphCoveragePath};
}

std::vector<DatedCoverageEdgeData> readCoverageTimeline(
    NYT::TTableReader<NYT::TNode>& reader, const CoverageEdgeData& testEdgeData)
{
    auto matches = [&](auto edgeData) {
        edgeData.coverageEdgeData.length = testEdgeData.length;
        return edgeData.coverageEdgeData == testEdgeData;
    };

    std::vector<DatedCoverageEdgeData> result;

    for (; reader.IsValid(); reader.Next()) {
        auto row = reader.GetRow();
        auto datedCoverageEdgeData =
            yt::deserialize<DatedCoverageEdgeData>(row);

        if (datedCoverageEdgeData.coverageEdgeData.dataset == std::nullopt) {
            INFO() << "readCoverageTimeline" << datedCoverageEdgeData;
        }

        if (matches(datedCoverageEdgeData)) {
            result.push_back(std::move(datedCoverageEdgeData));
        }
    }
    return result;
}

std::vector<DatedCoverageEdgeData>
generateDatedCoverageEdgeData(int32_t startDate, CoverageAgeCategory ageCategory,
    CoverageEdgeData baseEdgeData,
    const std::vector<double>& lengths)
{
    std::vector<DatedCoverageEdgeData> result;
    result.reserve(lengths.size());

    DatedCoverageEdgeData current{
        .coverageEdgeData = baseEdgeData,
        .ageCategory = static_cast<uint8_t>(ageCategory),
        .date = startDate,
    };

    for (double length : lengths) {
        current.coverageEdgeData.length = length;
        result.push_back(current);
        ++current.date;
    }
    return result;
}

} // namespace


Y_UNIT_TEST_SUITE_F(yt_integration_tests, Fixture)
{

Y_UNIT_TEST(full_process_simple_test)
{
    const std::optional<db::Dataset> ALL_DATASETS = std::nullopt;
    const std::optional<db::Dataset> AGENTS_DATASET = db::Dataset::Agents;
    const std::optional<db::Dataset> RIDES_DATASET = db::Dataset::Rides;

    const std::string DATE_STR = "2018-07-01 00:00:00";
    const std::string TIMELINE_END_DATE_STR = "2018-07-03 00:00:00";

    auto timelineEndDate =
        std::chrono::time_point_cast<std::chrono::days>(chrono::parseSqlDateTime(TIMELINE_END_DATE_STR))
            .time_since_epoch().count();

    insertData(
        {
            {{37.511455, 55.359990}, geolib3::Heading(98), DATE_STR, db::Dataset::Agents},
            {{37.511597, 55.359978}, geolib3::Heading(98), DATE_STR, db::Dataset::Agents},
            {{37.511767, 55.359963}, geolib3::Heading(98), DATE_STR, db::Dataset::Rides},
            {{37.511767, 55.359963}, geolib3::Heading(98), DATE_STR, db::Dataset::Agents, db::CameraDeviation::Right}
        }
    );

    auto exportInfo = exportToFb(pool(), graph());
    FeaturesIndex featuresIndex(exportInfo.mrcDatasetPath,
                                exportInfo.mrcFeaturesSecretDatasetPath);
    fb::CoverageRtreeReader coverageRtreeReader(graph(), exportInfo.mrcGraphCoveragePath);


    const TString edgesTableName = "//tmp/edge_data";
    const TString coverageTableName = "//tmp/coverage_data";
    const TString coverageTimelineTableName = "//tmp/coverage_timeline";
    const auto writer = ytClient().CreateTableWriter<NYT::TNode>(edgesTableName);

    auto coveredEdgeIdsRange = coverageRtreeReader.getCoveredEdgeIdsByBbox(BBOX);
    std::unordered_set<int32_t> referencedGeoIds;

    for (auto edgeId : coveredEdgeIdsRange) {
        road_graph::EdgeData edgeData = graph().edgeData(edgeId);
        geolib3::Polyline2 edgeGeometry = edgeData.geometry();
        db::Features features = searchFeaturesForEdge(edgeGeometry, featuresIndex);

        processEdge(edgeData, edgeGeometry, features,
                        referencedGeoIds, *writer);
    }

    writer->Finish();

    auto edgesReader = ytClient().CreateTableReader<NYT::TNode>(edgesTableName);

    EXPECT_THAT(
        readEdgeDataSummary(*edgesReader),
        UnorderedElementsAre(
            EdgeDataSummary{5, false, 4, 1, GRAPH_TYPE},
            EdgeDataSummary{5, false, 4, 4, GRAPH_TYPE}
        )
    );

    convertToCoverage(ytClient(), edgesTableName, coverageTableName);

    calculateGraphCoverageTimeline(ytClient(), timelineEndDate, coverageTableName, coverageTimelineTableName);

    auto timelineReader = ytClient().CreateTableReader<NYT::TNode>(coverageTimelineTableName);

    CoverageEdgeData testEdgeData{
                    .dataset = std::nullopt,
                    .fc = 5,
                    .isToll = 0,
                    .geoId = 225,
                    .privateArea = 0,
                    .graphType = 0,
                    .cameraDeviation = 0};

    EXPECT_THAT(
        readCoverageTimeline(*timelineReader, testEdgeData),
        Pointwise(
            DatedCoverageEdgeDataApproxEq(),
            generateDatedCoverageEdgeData(
                17713,
                CoverageAgeCategory::Age_0_7,
                testEdgeData,
                {49.8175, 49.8175})
        )
    );

    auto coverageReader = ytClient().CreateTableReader<NYT::TNode>(TString(coverageTableName));

    int64_t fromDate = std::chrono::duration_cast<chrono::Days>(
            chrono::parseSqlDateTime(DATE_STR).time_since_epoch()).count();

    // UnorderedElementsAre works for max 10 arguments,
    // so split read summaries before checking
    auto readSummary = readCoverageDataSummary(*coverageReader);
    std::vector<CoverageDataSummary> cameraRightSummary;
    std::vector<CoverageDataSummary> allDatasetsSummary;
    std::vector<CoverageDataSummary> agentsSummary;
    std::vector<CoverageDataSummary> ridesSummary;
    for (auto& item : readSummary) {
        if (item.cameraDeviation == 90) {
            cameraRightSummary.push_back(std::move(item));
        } else {
            if (item.dataset == ALL_DATASETS) {
                allDatasetsSummary.push_back(std::move(item));
            } else if (item.dataset == AGENTS_DATASET) {
                agentsSummary.push_back(std::move(item));
            } else if (item.dataset == RIDES_DATASET) {
                ridesSummary.push_back(std::move(item));
            }
        }
    }

    EXPECT_THAT(
        allDatasetsSummary,
        UnorderedElementsAre(
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 1, ALL_DATASETS, 49.8568, 0, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 3, ALL_DATASETS, 49.8568, 0, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 225, ALL_DATASETS, 49.8568, 0, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 10747, ALL_DATASETS, 49.8568, 0, GRAPH_TYPE}
        )
    );

    EXPECT_THAT(
        agentsSummary,
        UnorderedElementsAre(
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 1, AGENTS_DATASET, 38.9719, 0, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 3, AGENTS_DATASET, 38.9719, 0, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 225, AGENTS_DATASET, 38.9719, 0, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 10747, AGENTS_DATASET, 38.9719, 0, GRAPH_TYPE}
        )
    );

    EXPECT_THAT(
        ridesSummary,
        UnorderedElementsAre(
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 1, RIDES_DATASET, 29.9152, 0, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 3, RIDES_DATASET, 29.9152, 0, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 225, RIDES_DATASET, 29.9152, 0, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 10747, RIDES_DATASET, 29.9152, 0, GRAPH_TYPE}
        )
    );
    EXPECT_THAT(
        cameraRightSummary,
        UnorderedElementsAre(
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 1, ALL_DATASETS, 19.8301, 90, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 3, ALL_DATASETS, 19.8301, 90, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 225, ALL_DATASETS, 19.8301, 90, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 10747, ALL_DATASETS, 19.8301, 90, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 1, AGENTS_DATASET, 19.8301, 90, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 3, AGENTS_DATASET, 19.8301, 90, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 225, AGENTS_DATASET, 19.8301, 90, GRAPH_TYPE},
            CoverageDataSummary{fromDate, MAX_DATE, fromDate, 5, 0, 10747, AGENTS_DATASET, 19.8301, 90, GRAPH_TYPE}
        )
    );
} // Y_UNIT_TEST

} // Y_UNIT_TEST_SUITE

} // namespace maps::mrc::graph_coverage_export
