#include <maps/wikimap/mapspro/services/mrc/long_tasks/graph_coverage_export/lib/coverage_export.h>

#include <maps/infra/yacare/include/parse.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/introspection/include/comparison.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/services/mrc/libs/common/include/wiki_config.h>

#include <optional>
#include <string>

using namespace maps::mrc::graph_coverage_export;

namespace {

} // namespace

int main(int argc, const char** argv) try {
    NYT::Initialize(argc, argv);
    using namespace maps;
    log8::setLevel(log8::Level::INFO);
    maps::cmdline::Parser parser;

    auto configPath = parser.string("config")
            .help("path to mrc config");

    auto wikiConfigDirPath = parser.string("wiki-config-dir")
            .defaultValue(maps::mrc::common::WIKI_TEMPLATE_CONFIG_DIR)
            .help("path to services config template for wikimap");

    auto wikiConfigPath =
        parser.string("wiki-config")
            .help("path to services config for wikimap");

    auto secretVersion = parser.string("secret-version")
            .help("version for secrets from yav.yandex-team.ru");

    auto outputYtDirParam = parser.string("output-yt-dir")
            .help("base path for output YT tables")
            .required();

    auto bboxStr = parser.string("bbox")
            .help("region bbox in format lt_x,lt_y~rb_x,rb_y in geodetic coordinates");

    auto mrcDatasetPath = parser.string("mrc-dataset-path")
            .help("Path to dataset yandex-maps-mrc")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc/");

    auto mrcFeaturesSecretDatasetPath = parser.string("mrc-features-secret-dataset-path")
            .help("Path to dataset yandex-maps-mrc")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-features-secret/");

    auto geobasePath = parser.string("geobase")
            .defaultValue("/var/cache/geobase/geodata6.bin")
            .help("path to geodata binary file");

    auto mrcRoadGraphPath = parser.string("mrc-road-graph")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-graph-pro")
            .help("path to mrc road graph directory");

    auto mrcPedestrianGraphPath = parser.string("mrc-pedestrian-graph")
            .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-pedestrian-graph-pro")
            .help("path to mrc pedestrian graph directory");

    auto geodataPatchYtDirParam = parser.string("geodata-patch-yt-dir")
            .defaultValue("//home/maps/core/nmaps/analytics/geo-data")
            .help("YT dir with additional geobase regions");

    parser.parse(argc, const_cast<char**>(argv));

    std::optional<geolib3::BoundingBox> geoBbox;
    if (bboxStr.defined()) {
        geoBbox = yacare::Parser<geolib3::BoundingBox>()(bboxStr);
    }

    INFO() << "Loading config";
    const auto config =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, configPath);
    auto wikiConfig = wikiConfigPath.defined()
        ? maps::wiki::common::ExtendedXmlDoc(wikiConfigPath)
        : maps::mrc::common::makeWikiConfig(wikiConfigDirPath);

    coverageExport(config,
                   wikiConfig,
                   mrcDatasetPath,
                   mrcFeaturesSecretDatasetPath,
                   geobasePath,
                   geodataPatchYtDirParam,
                   mrcRoadGraphPath,
                   mrcPedestrianGraphPath,
                   TString{outputYtDirParam},
                   geoBbox);

    INFO() << "Done";

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    FATAL() << e;
    return EXIT_FAILURE;
} catch (const yexception& e) {
    FATAL() << "Worker failed: " << e.what();
    if (e.BackTrace()) {
        FATAL() << e.BackTrace()->PrintToString();
    }
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    FATAL() << e.what();
    return EXIT_FAILURE;
} catch (...) {
    FATAL() << "Caught unknown exception";
    return EXIT_FAILURE;
}
