#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_nexar/lib/metadata.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_nexar/lib/nexar_client.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_nexar/lib/import.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/pg_locks.h>
#include <maps/wikimap/mapspro/services/mrc/libs/config/include/config.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/read.h>
#include <maps/wikimap/mapspro/services/mrc/libs/privacy/include/geo_id_provider.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool3utils/include/yandex/maps/pgpool3utils/pg_advisory_mutex.h>
#include <maps/libs/road_graph/include/graph.h>
#include <maps/libs/succinct_rtree/include/rtree.h>
#include <maps/libs/introspection/include/stream_output.h>

#include <pqxx/transaction_base>

#include <chrono>
#include <filesystem>

namespace fs = std::filesystem;

namespace maps::mrc::import_nexar {

using maps::introspection::operator<<;

constexpr int TILING_ZOOM = 12;
const std::chrono::seconds NEXAR_TOKEN_REFRESH_PERIOD {23 * 60 * 60};

void importPhotosFromNexar(
    maps::pgpool3::Pool& pool,
    maps::mds::Mds& mdsClient,
    const fb::GraphReader& mrcGraphReader,
    const maps::road_graph::Graph& graph,
    const maps::succinct_rtree::Rtree& rtree,
    const privacy::GeoIdProvider& geoIdProvider,
    const INexarClient& nexarClient)
{
    INFO() << "Starting import photos from Nexar";

    auto metadata = loadMetadata(*pool.slaveTransaction());
    auto importConfigs =
        loadImportConfigs(*pool.slaveTransaction());
    bool updatedLastSuccessTime = false;


    INFO() << "There are " << importConfigs.size() << " import configs";

    for (const auto& config : importConfigs) {
        const auto checkTime = chrono::TimePoint::clock::now();
        auto tileImportParams = evalTileImportParams(
            geoIdProvider, config, TILING_ZOOM);
        tileImportParams = evalTileImportParamsToCheck(
                metadata,
                config,
                std::move(tileImportParams),
                checkTime
            );

        INFO() << "There are " << tileImportParams.size() << " tiles to update";

        for (const auto& tile : tileImportParams)
        try {
            INFO() << "Processing tile " << tile;
            auto tileUpdateInfo = importImagesForTile(
                pool, mdsClient, mrcGraphReader, graph, rtree, nexarClient, tile);
            INFO() << "Successfully processed tile " << tile;

            auto txn = pool.masterWriteableTransaction();

            metadata.setTileUpdateInfo(tileUpdateInfo);
            metadata.setLastSuccessRunTime(checkTime);
            updatedLastSuccessTime = true;
            saveMetadata(*txn, metadata);

            txn->commit();
        } catch (const Exception& ex) {
            ERROR() << "Failed to process tile " << tile << ": " << ex;
        } catch (const std::exception& ex) {
            ERROR() << "Failed to process tile " << tile << ": " << ex.what();
        }
    }

    if (!updatedLastSuccessTime) {
        INFO() << "There are no tiles to import";
        auto txn = pool.masterWriteableTransaction();
        metadata.setLastSuccessRunTime(chrono::TimePoint::clock::now());
        saveMetadata(*txn, metadata);
        txn->commit();
        return;
    }
}

} // namespace maps::mrc::import_nexar

int main(int argc, char* argv[])
try {
    maps::cmdline::Parser parser;
    auto configPath = parser
         .string("config")
         .help("path to configuration");

    auto secretVersion = parser
        .string("secret")
        .help("version for secrets from yav.yandex-team.ru");

    auto syslog = parser
        .string("syslog-tag")
        .help("redirect log output to syslog with given tag");

    auto graphDirOpt = parser.string("graph-dir")
        .defaultValue("/var/lib/yandex/maps/ecstatic/data/yandex-maps-mrc-graph-pro")
        .help("directory graph data is loaded from");

    auto geoIdCoveragePathOpt = parser.string("geoid-coverage-path")
        .help("path to coverage5-geoid mms");

    parser.parse(argc, argv);

    if (syslog.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslog));
    }

    INFO() << "Starting";

    auto config =
        maps::mrc::common::templateConfigFromCmdPath(secretVersion, configPath);

    auto poolHolder = config.makePoolHolder(
        maps::mrc::common::LONG_READ_DB_ID,
        maps::mrc::common::LONG_READ_POOL_ID);

    maps::pgp3utils::PgAdvisoryXactMutex lock(
                poolHolder.pool(),
                static_cast<int64_t>(maps::mrc::common::LockId::ImportNexar));
    if (!lock.try_lock()) {
        INFO() << "Could not acquire lock";
        INFO() << "Exiting";
        return EXIT_SUCCESS;
    }

    config.enableTvmClient();

    const std::string geoIdCoveragePathStr =
        geoIdCoveragePathOpt.defined()
            ? std::string(geoIdCoveragePathOpt)
            : config.externals().geoIdCoveragePath();

    auto mdsClient = config.makeMdsClient();

    const fs::path graphDir(std::string{graphDirOpt});
    maps::road_graph::Graph roadGraph(fs::canonical(graphDir / "road_graph.fb").string());
    maps::succinct_rtree::Rtree roadGraphRtree(fs::canonical(graphDir / "rtree.fb").string(), roadGraph);
    const auto& nexarConfig = config.externals().nexar();
    maps::mrc::import_nexar::NexarCachingTokenProvider nexarCachingTokenProvider(
        nexarConfig.host(),
        nexarConfig.refreshToken(),
        maps::mrc::import_nexar::NEXAR_TOKEN_REFRESH_PERIOD);
    auto geoIdProvider = maps::mrc::privacy::makeGeoIdProvider(geoIdCoveragePathStr);

    importPhotosFromNexar(
        poolHolder.pool(),
        mdsClient,
        maps::mrc::fb::GraphReader(fs::canonical(graphDir / "graph_coverage.fb").string()),
        roadGraph,
        roadGraphRtree,
        *geoIdProvider,
        maps::mrc::import_nexar::NexarHttpClient(nexarConfig.host(), nexarCachingTokenProvider)
    );

    INFO() << "Exiting";

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
}
