#include "nexar_client.h"
#include "maps/libs/http/include/header_map.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/visibility.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/enum_io/include/enum_io.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/heading.h>
#include <maps/libs/geolib/include/segment.h>
#include <maps/libs/introspection/include/comparison.h>
#include <maps/libs/log8/include/log8.h>

#include <contrib/libs/h3/h3lib/include/h3api.h>

#include <algorithm>
#include <mutex>
#include <set>
#include <shared_mutex>
#include <vector>

namespace maps::mrc::import_nexar {

namespace {

GeoCoord toGeoCoord(const geolib3::Point2 point)
{
    return GeoCoord{
        .lat=geolib3::degreesToRadians(point.y()),
        .lon=geolib3::degreesToRadians(point.x())
    };
}

/// Searches for h3 hexagons which centers is inside the @param geodeticBbox
std::vector<H3Index>
evalH3indices(const geolib3::BoundingBox& geodeticBbox, size_t h3Resolution)
{
    std::vector<GeoCoord> coords;
    coords.reserve(5);
    auto corners = geodeticBbox.corners();
    // put points in ccw order
    coords.push_back(toGeoCoord(corners.at(0)));
    coords.push_back(toGeoCoord(corners.at(1)));
    coords.push_back(toGeoCoord(corners.at(2)));
    coords.push_back(toGeoCoord(corners.at(3)));
    coords.push_back(toGeoCoord(corners.at(0)));
    const auto geoPoly = GeoPolygon{
        Geofence{static_cast<int>(coords.size()), &coords[0]}, 0, nullptr};

    const int numHexagons = maxPolyfillSize(&geoPoly, h3Resolution);

    constexpr H3Index DEFAULT_INDEX = 0;
    std::vector<H3Index> hexes(numHexagons, DEFAULT_INDEX);
    polyfill(&geoPoly, h3Resolution, hexes.data());
    hexes.erase(std::remove(hexes.begin(), hexes.end(), DEFAULT_INDEX), hexes.end());

    return hexes;
}

} // namespace

using introspection::operator==;
using introspection::operator<;

constexpr enum_io::Representations<RoundedHeading> ROUNDED_HEADING_ENUM_REPRESENTATION {
    {RoundedHeading::North, "NORTH"},
    {RoundedHeading::NorthEast, "NORTH_EAST"},
    {RoundedHeading::East, "EAST"},
    {RoundedHeading::SouthEast, "SOUTH_EAST"},
    {RoundedHeading::South, "SOUTH"},
    {RoundedHeading::SouthWest, "SOUTH_WEST"},
    {RoundedHeading::West, "WEST"},
    {RoundedHeading::NorthWest, "NORTH_WEST"}
};
DEFINE_ENUM_IO(RoundedHeading, ROUNDED_HEADING_ENUM_REPRESENTATION);

bool operator==(NexarImageMetaCRef one, NexarImageMetaCRef other)
{
    return one.get().frameId == other.get().frameId;
}

void fillDayPart(std::vector<NexarImageMeta>& images) {
    for (auto& image : images) {
        image.dayPart = common::getDayPart(
            image.capturedAt,
            image.position.y() /* lat */,
            image.position.x() /* lon */);
    }
}

RoundedHeading roundHeading(const geolib3::Heading heading)
{
    constexpr double step = 360. / 16;
    const geolib3::Heading shiftedHeading =
        geolib3::normalize(heading + geolib3::Heading(step));
    int value = static_cast<int>(shiftedHeading.value() / (2 * step));
    ASSERT(0 <= value && value <= 7);
    return RoundedHeading(value);
}

std::string toString(H3Index index)
{
    constexpr size_t MAX_H3INDEX_STR_SIZE = 17;
    std::string result;
    result.resize(MAX_H3INDEX_STR_SIZE);
    h3ToString(index, result.data(), result.size());
    return {result.c_str()};
}


H3Index h3indexFromString(const std::string& str)
{
    return stringToH3(str.data());
}

std::vector<NexarSpatialSearchParams>
evalSearchParams(const geolib3::Polyline2& geodeticPolyline, int h3Resolution)
{
    std::set<NexarSpatialSearchParams> result;
    const double h3hexagonEdgeLengthMeters = edgeLengthM(h3Resolution);

    for (const auto& segment : geodeticPolyline.segments())
    {
        const auto mercatorSegment = geolib3::convertGeodeticToMercator(segment);
        const auto heading = geolib3::Direction2(mercatorSegment).heading();
        const auto roundedHeading = roundHeading(heading);
        auto bbox = segment.boundingBox();
        bbox = db::expandBbox(bbox, h3hexagonEdgeLengthMeters);
        for (const auto h3index : evalH3indices(bbox, h3Resolution)) {
            result.insert(NexarSpatialSearchParams{
                .h3index = h3index, .heading = roundedHeading});
        }
    }
    return {result.begin(), result.end()};
}

NexarCachingTokenProvider::NexarCachingTokenProvider(
    std::string host,
    std::string refreshToken,
    std::chrono::seconds refreshPeriod)
    : host_(std::move(host))
    , refreshToken_(std::move(refreshToken))
    , refreshPeriod_(refreshPeriod)
{
    retryPolicy_.setInitialCooldown(std::chrono::milliseconds(500))
        .setTryNumber(3);
}


std::string NexarCachingTokenProvider::getToken()
{
    {
        std::shared_lock<std::shared_mutex> lock(mutex_);
        if (isTokenValid()) {
            return accessToken_.value();
        }
    }

    std::unique_lock<std::shared_mutex> lock(mutex_);
    /// Recheck token in case it was updated in parallel thread
    if (!isTokenValid()) {
        updateToken();
    }
    return accessToken_.value();
}

void NexarCachingTokenProvider::updateToken()
{
    accessToken_ = acquireToken();
    accessTokenAcquiredAt_ = chrono::TimePoint::clock::now();
}

bool NexarCachingTokenProvider::isTokenValid() const
{
    if (accessTokenAcquiredAt_.has_value() && accessToken_.has_value()) {
        auto now = chrono::TimePoint::clock::now();
        if (now - accessTokenAcquiredAt_.value() < refreshPeriod_) {
            return true;
        }
    }
    return false;
}

std::string NexarCachingTokenProvider::acquireToken()
{
    INFO() << "Acquiring Nexar access token";
    auto url = http::URL()
        .setScheme("https")
        .setHost(host_)
        .setPath("/dev-portal/refresh-token");

    const std::string body = "refresh_token=" + refreshToken_;

    auto [responseBody, responseStatus] = httpClient_.post(
        url,
        http::HeaderMap(
            {{"Accept", "application/json"},
             {"Content-Type", "application/x-www-form-urlencoded"}}),
        body,
        retryPolicy_);

    REQUIRE(responseStatus == 200, "Got unexpected response status " << responseStatus
        << " for request " << url << " body: " << responseBody);

    auto responseJson = json::Value::fromString(responseBody);
    return responseJson["access_token"].as<std::string>();
}

void NexarCachingTokenProvider::invalidateToken()
{
    std::unique_lock<std::shared_mutex> lock(mutex_);
    accessToken_.reset();
    accessTokenAcquiredAt_.reset();
}

NexarHttpClient::NexarHttpClient(std::string host, ITokenProvider& tokenProvider)
    : host_(std::move(host))
    , tokenProvider_(tokenProvider)
{
    retryPolicy_.setInitialCooldown(std::chrono::milliseconds(500))
        .setTryNumber(3);
}

std::vector<NexarImageMeta> NexarHttpClient::search(
    const NexarSpatialSearchParams& spatialSearchParams,
    const std::optional<chrono::TimePoint>& capturedAfter,
    size_t limit) const
{
    auto url = http::URL()
        .setScheme("https")
        .setHost(host_)
        .setPath("/api/roadItem/findRawFrames/v3");

    auto [responseBody, responseStatus] = httpClient_.post(
        url,
        makeSearchRequestHeaderMap(),
        makeSearchRequestBody(spatialSearchParams, capturedAfter, limit),
        retryPolicy_);

    onResponseStatusReceived(responseStatus);
    REQUIRE(responseStatus == 200, "Got unexpected response status " << responseStatus
        << " for request " << url);
    return parseRawFramesResponse(responseBody);
}

std::string NexarHttpClient::loadImage(const std::string& url) const
{
    http::HeaderMap headers;
    setAuthorizationHeaders(headers);
    auto [responseBody, responseStatus] = httpClient_.get(url, headers, retryPolicy_);
    onResponseStatusReceived(responseStatus);
    REQUIRE(responseStatus == 200, "Got unexpected response status " << responseStatus
        << " for request " << url);
    return responseBody;
}

void NexarHttpClient::onResponseStatusReceived(int httpStatus) const
{
    if (httpStatus == 403) {
        INFO() << "Invalidating token";
        tokenProvider_.invalidateToken();
    }
}

std::string NexarHttpClient::makeSearchRequestBody(
    const NexarSpatialSearchParams& spatialSearchParams,
    std::optional<chrono::TimePoint> capturedAfter,
    size_t limit) const
{
    json::Builder builder;
    builder << [&](json::ObjectBuilder obj)
        {
            obj["limit"] = limit;
            obj["filters"] =
                [&](json::ObjectBuilder obj)
                {
                    obj["h3_indices"] = [&](json::ObjectBuilder obj) {
                        obj["h3_indices"] = [&](json::ArrayBuilder arrayBuilder) {
                            arrayBuilder << spatialSearchParams.h3index;
                        };
                    };
                    if (capturedAfter.has_value()) {
                        obj["start_timestamp"] =
                            std::chrono::time_point_cast<std::chrono::milliseconds>(capturedAfter.value())
                                .time_since_epoch().count();
                    }

                    obj["heading"] = toString(spatialSearchParams.heading);
                };
        };
    return builder.str();
}

http::HeaderMap NexarHttpClient::makeSearchRequestHeaderMap() const
{
    http::HeaderMap headers;
    setAuthorizationHeaders(headers);
    headers.emplace("Accept", "application/json");
    headers.emplace("Content-Type", "application/json");
    return headers;
}

void NexarHttpClient::setAuthorizationHeaders(http::HeaderMap& headers) const
{
    headers.emplace("Authority", "live-api.nexar.mobi");
    headers.emplace("Authorization", "Bearer " + tokenProvider_.getToken());
}

std::vector<NexarImageMeta> NexarHttpClient::parseRawFramesResponse(
    const std::string& responseBody) const
{
    auto responseJson = json::Value::fromString(responseBody);
    auto framesJson = responseJson["raw_frames"];
    if (!framesJson.exists()) {
        return {};
    }

    std::vector<NexarImageMeta> result;
    result.reserve(framesJson.size());

    for (auto frameJson : framesJson) {
        result.push_back(parseImageMeta(frameJson));
    }

    return result;
}

NexarImageMeta NexarHttpClient::parseImageMeta(const json::Value& frameJson) const
{
    auto parseGpsPoint =
        [](const json::Value& pointJson) -> geolib3::Point2
        {
            return geolib3::Point2(
                pointJson["longtitude"].as<double>(),
                pointJson["latitude"].as<double>()
            );
        };

    return NexarImageMeta{
        .frameId = frameJson["frame_id"].as<FrameId>(),
        .capturedAt =
            chrono::TimePoint(
                std::chrono::duration_cast<chrono::TimePoint::duration>(
                    std::chrono::milliseconds(
                        frameJson["captured_on_ms"].as<int64_t>()
                    )
                )
            ),
        .position = parseGpsPoint(frameJson["gps_point"]),
        .heading = geolib3::Heading(frameJson["course_of_camera"].as<double>()),
        .imageUrl = frameJson["image_url"].as<std::string>()
    };
}

} // namespace maps::mrc::import_nexar
