#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/suncalc.h>

#include <maps/libs/common/include/retry.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/enum_io/include/enum_io_fwd.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/heading.h>
#include <maps/libs/geolib/include/test_tools/io_operations.h>
#include <maps/libs/http/include/http.h>
#include <maps/libs/json/include/value.h>

#include <contrib/libs/h3/h3lib/include/h3api.h>

#include <chrono>
#include <functional>
#include <optional>
#include <shared_mutex>

namespace maps::mrc::import_nexar {

using FrameId = std::string;

struct NexarImageMeta {
    FrameId frameId;
    chrono::TimePoint capturedAt;
    std::optional<common::DayPart> dayPart;
    geolib3::Point2 position;
    geolib3::Heading heading;
    std::string imageUrl;

    template<typename T>
    static auto introspect(T& t)
    {
        return std::tie(t.frameId, t.capturedAt, t.position, t.heading, t.imageUrl);
    }
};

using NexarImageMetaCRef = std::reference_wrapper<const NexarImageMeta>;
using NexarImageMetaCRefs = std::vector<NexarImageMetaCRef>;

bool operator==(NexarImageMetaCRef one, NexarImageMetaCRef other);

void fillDayPart(std::vector<NexarImageMeta>& images);

enum class RoundedHeading {
    North,
    NorthEast,
    East,
    SouthEast,
    South,
    SouthWest,
    West,
    NorthWest
};

DECLARE_ENUM_IO(RoundedHeading);

RoundedHeading roundHeading(const geolib3::Heading heading);

struct NexarSpatialSearchParams {
    H3Index h3index;
    RoundedHeading heading;

    template<typename T>
    static auto introspect(T& t)
    {
        return std::tie(t.h3index, t.heading);
    }
};

std::string toString(H3Index index);
H3Index h3indexFromString(const std::string& str);

std::vector<NexarSpatialSearchParams> evalSearchParams(
    const geolib3::Polyline2& geodeticPolyline, int h3Resolution);

class INexarClient {
public:
    virtual ~INexarClient() = default;

    virtual std::vector<NexarImageMeta> search(
        const NexarSpatialSearchParams& spatialSearchParams,
        const std::optional<chrono::TimePoint>& capturedAfter,
        size_t limit) const = 0;

    virtual std::string loadImage(const std::string& url) const = 0;
};


class ITokenProvider
{
public:
    virtual ~ITokenProvider() = default;

    virtual std::string getToken() = 0;

    /// Resets cached token to force its reacquisition in next
    /// getToken() call
    virtual void invalidateToken() = 0;
};

/// Provides a token for accessing Nexar API by exchanging it
/// for refreshToken throught special API handle.
/// Caches acquired tokens for @param refreshPeriod period.
class NexarCachingTokenProvider : public ITokenProvider{
public:
    NexarCachingTokenProvider(
        std::string host,
        std::string refreshToken,
        std::chrono::seconds refreshPeriod);

    std::string getToken() override;

    void invalidateToken() override;

private:
    std::string acquireToken();
    bool isTokenValid() const;
    void updateToken();

    std::string host_;
    std::string refreshToken_;
    std::chrono::seconds refreshPeriod_;
    maps::common::RetryPolicy retryPolicy_;
    http::Client httpClient_;

    std::shared_mutex mutex_;
    std::condition_variable cv_;

    std::optional<std::string> accessToken_;
    std::optional<chrono::TimePoint> accessTokenAcquiredAt_;
};

class NexarHttpClient : public INexarClient {
public:
    NexarHttpClient(std::string host, ITokenProvider& tokenProvider);

    std::vector<NexarImageMeta> search(
        const NexarSpatialSearchParams& spatialSearchParams,
        const std::optional<chrono::TimePoint>& capturedAfter,
        size_t limit) const override;

    std::string loadImage(const std::string& url) const override;

private:
    std::string makeSearchRequestBody(
            const NexarSpatialSearchParams& spatialSearchParams,
            std::optional<chrono::TimePoint> capturedAfter,
            size_t limit) const;

    http::HeaderMap makeSearchRequestHeaderMap() const;
    void setAuthorizationHeaders(http::HeaderMap&) const;
    void onResponseStatusReceived(int httpStatus) const;

    std::vector<NexarImageMeta> parseRawFramesResponse(const std::string& responseBody) const;
    NexarImageMeta parseImageMeta(const json::Value& frameJson) const;

    const std::string host_;
    ITokenProvider& tokenProvider_;
    mutable http::Client httpClient_;
    maps::common::RetryPolicy retryPolicy_;
};

} // namespace maps::mrc::import_nexar
