#include "common.h"
#include "fixture.h"
#include "maps/wikimap/mapspro/services/mrc/libs/fb/include/common.h"
#include "maps/wikimap/mapspro/services/mrc/long_tasks/import_nexar/lib/common.h"
#include "mocks.h"

#include "gmock/gmock-actions.h"
#include "gmock/gmock-cardinalities.h"
#include "nexar_frame_to_feature.h"
#include "nexar_frame_to_feature_gateway.h"
#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_nexar/lib/nexar_client.h>


#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_nexar/lib/import.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/nexar_import_tile_update_info.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/write.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/segment.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>
#include <maps/libs/geolib/include/test_tools/io_operations.h>
#include <maps/libs/introspection/include/comparison.h>
#include <maps/libs/introspection/include/stream_output.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/road_graph/include/graph.h>
#include <maps/libs/road_graph/include/types.h>
#include <maps/libs/succinct_rtree/include/rtree.h>
#include <maps/libs/tile/include/geometry.h>
#include <maps/libs/tile/include/tile.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <chrono>
#include <mutex>

namespace maps::geolib3 {
using maps::geolib3::io::operator<<;
} // namespace maps::geolib3

namespace maps::mrc::db {
using introspection::operator==;
using maps::introspection::operator<<;
} // namespace maps::mrc::db

namespace maps::mrc::import_nexar {

using introspection::operator==;
using introspection::operator<;
using maps::introspection::operator<<;

} // namespace maps::mrc::import_nexar

namespace maps::mrc::import_nexar::tests {

using namespace std::chrono_literals;

TEST(shouldUpdateTile, basic_test) {
    const int32_t geoId = 213;
    const int16_t fc = 1;
    const int32_t thresholdDays = 7;
    const int32_t x = 512;
    const int32_t y = 234;
    const int32_t z = 13;
    const db::ImportConfig importConfig(db::Dataset::NexarDashcams, geoId, fc, thresholdDays);
    const chrono::TimePoint checkTime =
        chrono::parseSqlDateTime("2015-06-16 16:57:27.797396+03");

    EXPECT_TRUE(shouldUpdateTile(
        importConfig,
        db::NexarImportTileUpdateInfo(geoId, fc, x, y, z),
        checkTime));

    // more than threshold time passed but we didn't find anything last time
    EXPECT_FALSE(shouldUpdateTile(
        importConfig,
        db::NexarImportTileUpdateInfo(geoId, fc, x, y, z, checkTime - std::chrono::days(thresholdDays + 1)),
        checkTime));

    // more than max threshold time passed, recheck
    EXPECT_TRUE(shouldUpdateTile(
        importConfig,
        db::NexarImportTileUpdateInfo(geoId, fc, x, y, z, checkTime - std::chrono::days(31)),
        checkTime));
}

TEST(evalGeodeticPolylinesToImport, basic_case)
{
    const std::string TEST_GRAPH_PATH = BinaryPath("maps/data/test/graph4");
    const road_graph::Graph graph(TEST_GRAPH_PATH + "/road_graph.fb");
    const succinct_rtree::Rtree rtree(TEST_GRAPH_PATH + "/rtree.fb", graph);
    const int32_t x = 158445;
    const int32_t y = 82203;
    const int32_t z = 18;
    const auto mercatorBbox = tile::mercatorBBox(tile::Tile(x, y, z)).polygon();

    auto polylines = evalGeodeticPolylinesToImport(
        graph,
        rtree,
        TileImportParams{
            .geoId = 225, .fc = 4, .x = x, .y = y, .z = z});

    EXPECT_EQ(polylines.size(), 3u);
    for (const auto& polyline: polylines) {
        auto mercatorPolyline = geolib3::convertGeodeticToMercator(polyline.geometry);
        EXPECT_TRUE(geolib3::spatialRelation(
            mercatorPolyline, mercatorBbox, geolib3::SpatialRelation::Within));
    }
}

TEST(evalUniqueActualCoverage, unnecessary_images_are_excluded)
{
    const geolib3::Point2 startPoint{0, 0};
    const geolib3::Polyline2 polyline{{startPoint,
            geolib3::fastGeoShift(startPoint, {100, 0})}};

    chrono::TimePoint startTime = chrono::parseIsoDateTime("2015-06-16T09:00:00Z");
    auto segmentHeading = geolib3::Direction2(polyline.segmentAt(0).vector()).heading();

    std::vector<NexarImageMeta> images{
        NexarImageMeta{
            .frameId = "1",
            .capturedAt = startTime,
            .position = polyline.pointAt(0),
            .heading = segmentHeading
        },
        NexarImageMeta{
            .frameId = "2",
            .capturedAt = startTime + 2s,
            .position = polyline.segmentAt(0).pointByPosition(0.1),
            .heading = segmentHeading
        },
        NexarImageMeta{
            .frameId = "3",
            .capturedAt = startTime,
            .position = polyline.segmentAt(0).pointByPosition(0.20),
            .heading = segmentHeading
        },
        /// These images must be excluded
        NexarImageMeta{
            .frameId = "4",
            .capturedAt = startTime - 2s, // too old
            .position = polyline.segmentAt(0).pointByPosition(0.15),
            .heading = segmentHeading
        },
        NexarImageMeta{
            .frameId = "5",
            .capturedAt = startTime + 2s,
            .position = startPoint,
            .heading = segmentHeading + geolib3::Heading(90) // looks in wrong direction
        },
        NexarImageMeta{
            .frameId = "6",
            .capturedAt = startTime + 2s,
            .position = geolib3::Point2{0, 10}, // too far
            .heading = segmentHeading
        },
    };
    fillDayPart(images);

    NexarImageMetaCRefs imageRefs{images.begin(), images.end()};
    auto result = evalUniqueActualCoverage(polyline, imageRefs);
    std::vector<FrameId> frameIds;
    for (const SubpolylineWithImageRef item : result) {
        frameIds.push_back(item.value.get().frameId);
    }

    EXPECT_THAT(frameIds, ::testing::UnorderedElementsAre("1", "2", "3"));
}

TEST(evalUniqueActualCoverage, prefer_daytime_images)
{
    const geolib3::Point2 startPoint{37, 55};
    const geolib3::Polyline2 polyline{{startPoint,
            geolib3::fastGeoShift(startPoint, {100, 0})}};

    chrono::TimePoint oldDayTime = chrono::parseIsoDateTime("2021-11-01T09:00:00Z");
    chrono::TimePoint medDayTime = chrono::parseIsoDateTime("2021-11-22T09:00:00Z");
    chrono::TimePoint newNightTime = chrono::parseIsoDateTime("2021-12-02T04:00:00Z");
    auto segmentHeading = geolib3::Direction2(polyline.segmentAt(0).vector()).heading();

    std::vector<NexarImageMeta> images{
        /// 2 should be choosen in favor of 1 because it daytime
        NexarImageMeta{
            .frameId = "1",
            .capturedAt = newNightTime,
            .position = polyline.pointAt(0),
            .heading = segmentHeading
        },
        NexarImageMeta{
            .frameId = "2",
            .capturedAt = medDayTime,
            .position = polyline.pointAt(0),
            .heading = segmentHeading
        },
        /// 3 should be choosen in favor of 4 because it is far more newer
        NexarImageMeta{
            .frameId = "3",
            .capturedAt = newNightTime,
            .position = polyline.segmentAt(0).pointByPosition(0.1),
            .heading = segmentHeading
        },
        NexarImageMeta{
            .frameId = "4",
            .capturedAt = oldDayTime,
            .position = polyline.segmentAt(0).pointByPosition(0.1),
            .heading = segmentHeading
        },
    };
    fillDayPart(images);

    NexarImageMetaCRefs imageRefs{images.begin(), images.end()};
    auto result = evalUniqueActualCoverage(polyline, imageRefs);
    std::vector<FrameId> frameIds;
    for (const SubpolylineWithImageRef item : result) {
        frameIds.push_back(item.value.get().frameId);
    }

    EXPECT_THAT(frameIds, ::testing::UnorderedElementsAre("2", "3"));
}


TEST(selectBestCoveringFrames, nexar_search_queries_are_unique)
{
    const std::string TEST_GRAPH_PATH = BinaryPath("maps/data/test/graph4");
    const road_graph::Graph graph(TEST_GRAPH_PATH + "/road_graph.fb");
    const succinct_rtree::Rtree rtree(TEST_GRAPH_PATH + "/rtree.fb", graph);
    const int32_t x = 158445;
    const int32_t y = 82203;
    const int32_t z = 18;
    const auto tileImportParams =
        TileImportParams{.geoId = 225, .fc = 4, .x = x, .y = y, .z = z};

    class DummyNexarClient : public INexarClient {
    public:
        std::vector<NexarImageMeta> search(
            const NexarSpatialSearchParams& spatialSearchParams,
            const std::optional<chrono::TimePoint>& /*capturedAfter*/,
            size_t /*limit*/) const override
        {
            std::unique_lock<std::mutex> lock(mutex);
            EXPECT_TRUE(performedCalls.insert(spatialSearchParams).second);
            return {};
        }

        std::string loadImage(const std::string&) const override { return {}; }

        mutable std::set<NexarSpatialSearchParams> performedCalls;
        mutable std::mutex mutex;
    };
    DummyNexarClient dummyClient;

    selectBestCoveringFrames(graph, rtree, tileImportParams, dummyClient);
    EXPECT_THAT(dummyClient.performedCalls.size(), ::testing::Gt(0u));
}

TEST(improvesEdgeCoverage_should, tests)
{
    chrono::TimePoint startTime =
        chrono::parseIsoDateTime("2015-06-16T09:00:00Z");
    NexarImageMeta image{.capturedAt = startTime};

    // Do not import empty coverage
    EXPECT_FALSE(improvesEdgeCoverage(
        ImageWithCoverage{
            .imageMeta = image,
            .subpolyline{{0, 0.}, {0, 0.}}},
        fb::TEdge{}));

    // Import non-empty coverage
    EXPECT_TRUE(improvesEdgeCoverage(
        ImageWithCoverage{
            .imageMeta = image,
            .subpolyline{{0, 0.}, {0, 0.5}}},
        fb::TEdge{}));

    // Do not import photo if it's not fresh enought
    EXPECT_FALSE(improvesEdgeCoverage(
        ImageWithCoverage{
            .imageMeta = image,
            .subpolyline{{0, 0.}, {0, 1.}}},
        fb::TEdge{
            .coverages = {fb::TEdgeCoverage{
                .actualizationDate = startTime,
                .coveredSubpolylines = {{1, {0, 0.}, {0, 1.}}},
                .cameraDeviation = db::CameraDeviation::Front}},
        }));

    // Import fresh photo
    EXPECT_TRUE(improvesEdgeCoverage(
        ImageWithCoverage{
            .imageMeta = image,
            .subpolyline{{0, 0.}, {0, 1.}}},
        fb::TEdge{
            .coverages = {fb::TEdgeCoverage{
                .actualizationDate = startTime - std::chrono::days(20),
                .coveredSubpolylines = {{1, {0, 0.}, {0, 1.}}},
                .cameraDeviation = db::CameraDeviation::Front}},
        }));

    // Ignore coverages with non-front cameraDeviation
    EXPECT_TRUE(improvesEdgeCoverage(
        ImageWithCoverage{
            .imageMeta = image,
            .subpolyline{{0, 0.}, {0, 1.}}},
        fb::TEdge{
            .coverages = {fb::TEdgeCoverage{
                .actualizationDate = startTime,
                .coveredSubpolylines = {{1, {0, 0.}, {0, 1.}}},
                .cameraDeviation = db::CameraDeviation::Right,
            }},
        }));

    // Import photo if it improves current coverage
    EXPECT_TRUE(improvesEdgeCoverage(
        ImageWithCoverage{
            .imageMeta = image,
            .subpolyline{{0, 0.}, {0, 1.}}},
        fb::TEdge{
            .coverages = {fb::TEdgeCoverage{
                .actualizationDate = startTime,
                .coveredSubpolylines = {{1, {0, 0.}, {0, 0.5}}, {2, {0, 0.6}, {0, 1.}}},
                .cameraDeviation = db::CameraDeviation::Front,
            }},
        }));
}

TEST_F(MdsDatabaseFixture, importImagesForTile_should)
{
    NexarImageMeta imageMeta{
        .frameId = "1",
        .capturedAt = chrono::parseIsoDateTime("2015-06-16T09:00:00Z"),
        .position = geolib3::Point2(37, 55),
        .heading = geolib3::Heading(0),
        .imageUrl = "http:://test_url"};

    const std::string imageData = "ENCODED_IMAGE";

    auto mdsClient = config().makeMdsClient();

    NexarClientMock nexarClient;

    EXPECT_CALL(nexarClient, loadImage(imageMeta.imageUrl))
        .Times(1)
        .WillOnce(::testing::Return(imageData));

    auto feature = importImage(pool(), nexarClient, mdsClient, imageMeta);

    EXPECT_NE(feature.mdsPath(), "");
    EXPECT_NE(feature.mdsGroupId(), "");
    EXPECT_EQ(feature.timestamp(), imageMeta.capturedAt);
    EXPECT_EQ(feature.geodeticPos(), imageMeta.position);
    EXPECT_EQ(feature.heading(), imageMeta.heading);

    EXPECT_EQ(mdsClient.get(feature.mdsKey()), imageData);

    auto txn = pool().masterWriteableTransaction();
    auto loadedFeature = db::FeatureGateway(*txn).loadById(feature.id());

    EXPECT_EQ(loadedFeature, feature);
}


class EdgeFilledWithPhotosFixture : public MdsDatabaseFixture
{
public:
    EdgeFilledWithPhotosFixture()
        : TEST_GRAPH_PATH(BinaryPath("maps/data/test/graph4"))
        , graph(TEST_GRAPH_PATH + "/road_graph.fb")
        , rtree(TEST_GRAPH_PATH + "/rtree.fb", graph)
    {
        selectEdgePolylineInsideTile();
        startTime = chrono::parseIsoDateTime("2015-06-16T09:00:00Z");
        nexarImages = generatePhotosAlongThePolyline(edgeTileGeoPolyline, 10.);

        const std::string testImagePath = GetWorkPath() + "/120073503.jpg";
        testImageData = maps::common::readFileToString(testImagePath);

        EXPECT_CALL(nexarClient, search(::testing::_, ::testing::_, ::testing::_))
            .Times(::testing::AtLeast(1))
            .WillRepeatedly(::testing::Return(nexarImages));
    }

    tile::Tile tile() const
    {
        return tile::Tile(
            tileImportParams.x, tileImportParams.y, tileImportParams.z);
    }

    geolib3::BoundingBox tileMercatorBbox() const
    {
        return tile::mercatorBBox(tile());
    }

    geolib3::BoundingBox tileGeodeticBbox() const
    {
        return geolib3::convertMercatorToGeodetic(tileMercatorBbox());
    }

    std::vector<road_graph::EdgeId> edgeIdsInTile()
    {
        std::vector<road_graph::EdgeId> result;
        for (auto id: rtree.baseEdgesInWindow(tileGeodeticBbox())) {
            auto edgeData = graph.edgeData(id);
            if (edgeData.category() != static_cast<uint16_t>(tileImportParams.fc)) {
                continue;
            }
            result.push_back(id);
        }
        return result;
    }

    void selectEdgePolylineInsideTile()
    {
        for (auto id: rtree.baseEdgesInWindow(tileGeodeticBbox())) {
            auto edgeData = graph.edgeData(id);
            if (edgeData.category() != static_cast<uint16_t>(tileImportParams.fc)) {
                continue;
            }

            const geolib3::Polyline2 geodeticPolyline = edgeData.geometry();
            const auto mercatorPolyline = geolib3::convertGeodeticToMercator(geodeticPolyline);
            const auto polylinesInTile = geolib3::intersection(mercatorPolyline, tileMercatorBbox());

            if (!polylinesInTile.empty()) {
                edgeId = id;
                edgeTileGeoPolyline = geolib3::convertMercatorToGeodetic(polylinesInTile.front());
                return;
            }
        }
        REQUIRE(false, "Could not find an edge");
    };

    std::vector<NexarImageMeta> generatePhotosAlongThePolyline(
        const geolib3::Polyline2 geoPolyline, double intervalMeters)
    {
        std::vector<NexarImageMeta> result;
        for (const auto& segment : geoPolyline.segments()) {
            auto length = geolib3::fastGeoDistance(segment.start(), segment.end());
            const geolib3::Heading heading = geolib3::Direction2(segment.vector()).heading();
            const double partRatio = intervalMeters / length;
            const int partsNumber = 1 / partRatio;
            for (int i = 0; i < partsNumber; ++i) {
                result.push_back(
                    NexarImageMeta {
                        .frameId = std::to_string(result.size()),
                        .capturedAt = startTime + std::chrono::seconds(15),
                        .position = segment.pointByPosition(i * partRatio),
                        .heading = heading,
                        .imageUrl = IMAGE_URL
                    }
                );
            }
        }
        return result;
    }

    fb::GraphReader makeGraphCoverage(std::vector<fb::TEdge> edges = {})
    {
        auto testFile = "graph_coverage.fb";
        fb::TGraph fbGraph{.version = static_cast<std::string>(graph.version())};
        fbGraph.edges = std::move(edges);
        fb::writeToFile(fbGraph, testFile);
        return fb::GraphReader{testFile};
    };

    std::string TEST_GRAPH_PATH;
    road_graph::Graph graph;
    succinct_rtree::Rtree rtree;
    TileImportParams tileImportParams{.geoId = 225, .fc = 4, .x = 158445, .y = 82203, .z = 18};
    chrono::TimePoint startTime;
    std::string IMAGE_URL = "http://test";
    geolib3::Polyline2 edgeTileGeoPolyline;
    road_graph::EdgeId edgeId;
    std::vector<NexarImageMeta> nexarImages;
    NexarClientMock nexarClient;
    std::string testImageData;
}; // EdgeFilledWithPhotosFixture

TEST_F(EdgeFilledWithPhotosFixture, importImage_should_import_if_no_coverage)
{
    fb::GraphReader graphCoverage = makeGraphCoverage();
    auto mdsClient = config().makeMdsClient();

    EXPECT_CALL(nexarClient, loadImage(IMAGE_URL))
        .Times(::testing::AtMost(nexarImages.size()))
        .WillRepeatedly(::testing::Return(testImageData));

    auto tileUpdateInfo = importImagesForTile(
        pool(),
        mdsClient,
        graphCoverage,
        graph,
        rtree,
        nexarClient,
        tileImportParams
    );

    EXPECT_TRUE(tileUpdateInfo.medianPhotoAgeDays().has_value());

    auto txn = pool().masterWriteableTransaction();
    EXPECT_EQ(db::FeatureGateway(*txn).count(), nexarImages.size());
    EXPECT_EQ(db::NexarFrameToFeatureGateway(*txn).count(), nexarImages.size());
}

TEST_F(EdgeFilledWithPhotosFixture, importImage_must_not_import_already_imported_images)
{
    fb::GraphReader graphCoverage = makeGraphCoverage();
    auto mdsClient = config().makeMdsClient();

    {
        auto txn = pool().masterWriteableTransaction();
        std::vector<db::Feature> features;
        for (const auto& image : nexarImages) {
            features.push_back(convertToFeature(image, testImageData));

        }
        db::FeatureGateway(*txn).insert(features);

        std::vector<db::NexarFrameToFeature> frameToFeatureVec;
        for(size_t i = 0; i < nexarImages.size(); ++i) {
            frameToFeatureVec.emplace_back(nexarImages[i].frameId, features[i].id());
        }
        db::NexarFrameToFeatureGateway(*txn).insert(frameToFeatureVec);

        txn->commit();
    }

    auto tileUpdateInfo = importImagesForTile(
        pool(),
        mdsClient,
        graphCoverage,
        graph,
        rtree,
        nexarClient,
        tileImportParams
    );

    EXPECT_TRUE(tileUpdateInfo.medianPhotoAgeDays().has_value());

    auto txn = pool().masterWriteableTransaction();
    EXPECT_EQ(db::FeatureGateway(*txn).count(), nexarImages.size());
}

TEST_F(EdgeFilledWithPhotosFixture, importImage_must_not_import_if_our_coverage_is_good)
{
    std::vector<fb::TEdge> fbEdges;
    for (auto edgeId: edgeIdsInTile()) {
        auto edgeData = graph.edgeData(edgeId);
        const geolib3::Polyline2 geodeticPolyline = edgeData.geometry();
        const uint16_t segmentsNumber = static_cast<uint16_t>(geodeticPolyline.segmentsNumber());
        fbEdges.push_back(fb::TEdge{
            .id = edgeId.value(),
            .coverages = {fb::TEdgeCoverage{
                .coverageFraction = 1.0,
                .actualizationDate = startTime,
                .coveredSubpolylines = {
                    {1, {0, 0.}, {segmentsNumber, 1.}}},
                .cameraDeviation = db::CameraDeviation::Front}}});
    }

    fb::GraphReader graphCoverage = makeGraphCoverage(fbEdges);
    auto mdsClient = config().makeMdsClient();

    auto tileUpdateInfo = importImagesForTile(
        pool(),
        mdsClient,
        graphCoverage,
        graph,
        rtree,
        nexarClient,
        tileImportParams
    );

    EXPECT_TRUE(tileUpdateInfo.medianPhotoAgeDays().has_value());

    auto txn = pool().masterWriteableTransaction();
    EXPECT_EQ(db::FeatureGateway(*txn).count(), 0u);
}

TEST_F(EdgeFilledWithPhotosFixture, importImage_sould_import_if_our_coverage_is_old)
{
    chrono::TimePoint oldActualizationDate = startTime - std::chrono::days(30);
    std::vector<fb::TEdge> fbEdges;
    for (auto edgeId: edgeIdsInTile()) {
        auto edgeData = graph.edgeData(edgeId);
        const geolib3::Polyline2 geodeticPolyline = edgeData.geometry();
        const uint16_t segmentsNumber = static_cast<uint16_t>(geodeticPolyline.segmentsNumber());
        fbEdges.push_back(fb::TEdge{
            .id = edgeId.value(),
            .coverages = {fb::TEdgeCoverage{
                .coverageFraction = 1.0,
                .actualizationDate = oldActualizationDate,
                .coveredSubpolylines = {
                    {1, {0, 0.}, {segmentsNumber, 1.}}},
                .cameraDeviation = db::CameraDeviation::Front}}});
    }

    fb::GraphReader graphCoverage = makeGraphCoverage(fbEdges);
    auto mdsClient = config().makeMdsClient();

    EXPECT_CALL(nexarClient, loadImage(IMAGE_URL))
        .Times(::testing::AtMost(nexarImages.size()))
        .WillRepeatedly(::testing::Return(testImageData));

    auto tileUpdateInfo = importImagesForTile(
        pool(),
        mdsClient,
        graphCoverage,
        graph,
        rtree,
        nexarClient,
        tileImportParams
    );

    EXPECT_TRUE(tileUpdateInfo.medianPhotoAgeDays().has_value());

    auto txn = pool().masterWriteableTransaction();
    EXPECT_EQ(db::FeatureGateway(*txn).count(), nexarImages.size());
}


} // maps::mrc::import_nexar::tests
