#include "fixture.h"
#include "gmock/gmock-actions.h"
#include "gmock/gmock-cardinalities.h"
#include "gtest/gtest.h"
#include "mocks.h"
#include "gmock/gmock-matchers.h"
#include "maps/libs/geolib/include/point.h"

#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_nexar/lib/nexar_client.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/geolib/include/heading.h>
#include <maps/libs/http/include/test_utils.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/introspection/include/comparison.h>
#include <maps/libs/introspection/include/stream_output.h>

#include <contrib/libs/h3/h3lib/include/h3api.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <chrono>
#include <thread>


namespace maps::geolib3 {
using maps::geolib3::io::operator<<;
}

namespace maps::mrc::import_nexar {

using introspection::operator==;
using maps::introspection::operator<<;

} // namespace maps::mrc::import_nexar

namespace maps::mrc::import_nexar::tests {

namespace {

const std::string TEST_NEXAR_HOST = "nexar.test";
const std::string TEST_ACCESS_TOKEN = "ACCESS_TOKEN";
const std::string NEXAR_API_FIND_ENDPOINT = "https://" + TEST_NEXAR_HOST + "/api/roadItem/findRawFrames/v3";
const std::string NEXAR_API_GET_TOKEN_ENDPOINT = "https://" + TEST_NEXAR_HOST + "/dev-portal/refresh-token";

} // namespace

TEST(roundHeading_should, basic_test) {
    EXPECT_EQ(roundHeading(geolib3::Heading(0)), RoundedHeading::North);
    EXPECT_EQ(roundHeading(geolib3::Heading(22.4)), RoundedHeading::North);
    EXPECT_EQ(roundHeading(geolib3::Heading(22.6)), RoundedHeading::NorthEast);
    EXPECT_EQ(roundHeading(geolib3::Heading(90)), RoundedHeading::East);
    EXPECT_EQ(roundHeading(geolib3::Heading(113)), RoundedHeading::SouthEast);
    EXPECT_EQ(roundHeading(geolib3::Heading(180)), RoundedHeading::South);
    EXPECT_EQ(roundHeading(geolib3::Heading(230)), RoundedHeading::SouthWest);
    EXPECT_EQ(roundHeading(geolib3::Heading(280)), RoundedHeading::West);
    EXPECT_EQ(roundHeading(geolib3::Heading(320)), RoundedHeading::NorthWest);
    EXPECT_EQ(roundHeading(geolib3::Heading(359)), RoundedHeading::North);
}

TEST(NexarCachingTokenProvider_should, provide_token)
{
    const std::string REFRESH_TOKEN = "REFRESH_TOKEN";

    int getTokenRequests = 0;

    auto mock = http::addMock(NEXAR_API_GET_TOKEN_ENDPOINT,
        [&](const http::MockRequest& request) {
            ++getTokenRequests;
            EXPECT_EQ(request.body,
                "refresh_token=" + REFRESH_TOKEN);

            const std::string body = "{\"access_token\":\"" +
                TEST_ACCESS_TOKEN + "\"}";
            return http::MockResponse{body};
        });

    NexarCachingTokenProvider provider(TEST_NEXAR_HOST, REFRESH_TOKEN, std::chrono::seconds(10));
    EXPECT_EQ(provider.getToken(), TEST_ACCESS_TOKEN);
    EXPECT_EQ(provider.getToken(), TEST_ACCESS_TOKEN);
    provider.invalidateToken();
    EXPECT_EQ(provider.getToken(), TEST_ACCESS_TOKEN);

    EXPECT_EQ(getTokenRequests, 2);
}

TEST(NexarCachingTokenProvider_should, survive_temporal_fault)
{
    const std::string REFRESH_TOKEN = "REFRESH_TOKEN";

    int getTokenRequests = 0;

    auto mock = http::addMock(NEXAR_API_GET_TOKEN_ENDPOINT,
        [&](const http::MockRequest& request) {
            EXPECT_EQ(request.body,
                "refresh_token=" + REFRESH_TOKEN);

            if (++getTokenRequests < 3) {
                return http::MockResponse("").withStatus(500);
            }

            const std::string body = "{\"access_token\":\"" +
                TEST_ACCESS_TOKEN + "\"}";
            return http::MockResponse{body};
        });

    NexarCachingTokenProvider provider(TEST_NEXAR_HOST, REFRESH_TOKEN, std::chrono::seconds(10));
    EXPECT_EQ(provider.getToken(), TEST_ACCESS_TOKEN);
    EXPECT_EQ(getTokenRequests, 3);
}


TEST(NexarCachingTokenProvider_should, invalidate_token_by_time)
{
    const std::string REFRESH_TOKEN = "REFRESH_TOKEN";

    int getTokenRequests = 0;

    auto mock = http::addMock(NEXAR_API_GET_TOKEN_ENDPOINT,
        [&](const http::MockRequest& request) {
            EXPECT_EQ(request.body,
                "refresh_token=" + REFRESH_TOKEN);

            ++getTokenRequests;

            const std::string body = "{\"access_token\":\"" +
                TEST_ACCESS_TOKEN + "\"}";
            return http::MockResponse{body};
        });

    std::chrono::seconds refreshPeriod(1);
    NexarCachingTokenProvider provider(TEST_NEXAR_HOST, REFRESH_TOKEN, refreshPeriod);
    EXPECT_EQ(provider.getToken(), TEST_ACCESS_TOKEN);
    std::this_thread::sleep_for(refreshPeriod * 2);
    EXPECT_EQ(provider.getToken(), TEST_ACCESS_TOKEN);
    EXPECT_EQ(getTokenRequests, 2);
}


TEST(NexarHttpClient_should, search)
{
    const H3Index h3index = 631307745614571519;
    const chrono::TimePoint capturedAfter =
        chrono::parseIsoDateTime("2015-06-16T09:00:00Z");
    const int limit = 10;

    const RoundedHeading heading = RoundedHeading::South;

    TokenProviderMock tokenProviderMock;
    EXPECT_CALL(tokenProviderMock, getToken())
        .Times(::testing::AtMost(1))
        .WillOnce(::testing::Return(TEST_ACCESS_TOKEN));

    NexarHttpClient client(TEST_NEXAR_HOST, tokenProviderMock);

    auto mock = http::addMock(NEXAR_API_FIND_ENDPOINT,
        [&](const http::MockRequest& request) {
            auto jsonRequest = json::Value::fromString(request.body);
            EXPECT_EQ(jsonRequest["limit"].as<int>(), limit);
            EXPECT_EQ(
                jsonRequest["filters"]["h3_indices"]["h3_indices"][0].as<size_t>(),
                h3index);
            EXPECT_EQ(jsonRequest["filters"]["start_timestamp"].as<size_t>(),
                chrono::sinceEpoch<std::chrono::milliseconds>(capturedAfter));
            EXPECT_EQ(request.header("Authority"), "live-api.nexar.mobi");
            EXPECT_EQ(request.header("Authorization"), "Bearer " + TEST_ACCESS_TOKEN);

            const std::string body = R"(
{
    "raw_frames": [
        {
            "captured_on_ms": 1434445200000,
            "captured_timezone_offset": 0,
            "course_of_camera": 270.5,
            "frame_id": "060324cbf16b5f89690f81d6ca5404b4",
            "gps_point": {
                "altitude": 0.0,
                "latitude": 32.079556,
                "longtitude": 34.779919
            },
            "h3_indices": {
                "h3_index_res12": "631307745614571519",
                "h3_index_res13": "635811345241941695"
            },
            "heading": "WEST",
            "image_quality": {
                "internal_car_average": 0.010086407
            },
            "image_url": "https://test_domain/1.jpg"
        }
    ]
}
                )";
            return http::MockResponse{body};
        });

    auto images = client.search({h3index, heading}, capturedAfter, limit);
    EXPECT_THAT(images,
        ::testing::ElementsAre(
            NexarImageMeta{
                .frameId = "060324cbf16b5f89690f81d6ca5404b4",
                .capturedAt = chrono::parseIsoDateTime("2015-06-16T09:00:00Z"),
                .position = geolib3::Point2(34.779919, 32.079556),
                .heading = geolib3::Heading(270.5),
                .imageUrl = "https://test_domain/1.jpg",
            }
        ));
}

TEST(NexarHttpClient_should, load_image_success_case)
{
    const std::string url = "http://test/url";
    const std::string responseBody = "response";

    TokenProviderMock tokenProviderMock;
    EXPECT_CALL(tokenProviderMock, getToken())
        .Times(::testing::AtMost(1))
        .WillOnce(::testing::Return(TEST_ACCESS_TOKEN));

    NexarHttpClient client(TEST_NEXAR_HOST, tokenProviderMock);

    auto mock = http::addMock(url,
        [&](const http::MockRequest& request) {
            EXPECT_EQ(request.header("Authority"), "live-api.nexar.mobi");
            EXPECT_EQ(request.header("Authorization"), "Bearer " + TEST_ACCESS_TOKEN);

            return http::MockResponse{responseBody};
        });

    EXPECT_EQ(client.loadImage(url), responseBody);
}

TEST(NexarHttpClient_should, load_image_temporal_error)
{
    const std::string url = "http://test/url";
    const std::string responseBody = "response";

    TokenProviderMock tokenProviderMock;
    EXPECT_CALL(tokenProviderMock, getToken())
        .Times(::testing::AtMost(1))
        .WillOnce(::testing::Return(TEST_ACCESS_TOKEN));

    NexarHttpClient client(TEST_NEXAR_HOST, tokenProviderMock);

    int requestCnt = 0;

    auto mock = http::addMock(url,
        [&](const http::MockRequest& request) {
            EXPECT_EQ(request.header("Authority"), "live-api.nexar.mobi");
            EXPECT_EQ(request.header("Authorization"), "Bearer " + TEST_ACCESS_TOKEN);

            if (++requestCnt < 3) {
                return http::MockResponse("").withStatus(500);
            }
            return http::MockResponse{responseBody};
        });

    EXPECT_EQ(client.loadImage(url), responseBody);
}

TEST(NexarHttpClient_should, invalidate_token)
{
    const std::string url = "http://test/url";
    const std::string responseBody = "response";

    TokenProviderMock tokenProviderMock;
    EXPECT_CALL(tokenProviderMock, getToken())
        .Times(::testing::AtMost(1))
        .WillOnce(::testing::Return(TEST_ACCESS_TOKEN));
    EXPECT_CALL(tokenProviderMock, invalidateToken())
        .Times(1);

    NexarHttpClient client(TEST_NEXAR_HOST, tokenProviderMock);

    auto mock = http::addMock(url,
        [&](const http::MockRequest& /*request*/) {
            return http::MockResponse("").withStatus(403);
        });

    EXPECT_THROW(client.loadImage(url), Exception);
}

TEST(h3index_funcs_should, string_serialization)
{
    H3Index index(630814729605382655);
    auto str = toString(index);
    EXPECT_EQ(str, "8c11aa7853681ff");
    EXPECT_EQ(h3indexFromString(str), index);
}

TEST(evalSearchParams_should, basic_case)
{
    EXPECT_THAT(
        evalSearchParams(
            geolib3::Polyline2(
                {{37.597776, 55.734229},
                 {37.600024, 55.733512},
                 {37.600142, 55.733745},
                 {37.598012, 55.734453}}),
            10),
        ::testing::UnorderedElementsAre(
            NexarSpatialSearchParams{
                .h3index = 621807530350510079,
                .heading = RoundedHeading::SouthEast},
            NexarSpatialSearchParams{
                .h3index = 621807530350542847,
                .heading = RoundedHeading::North},
            NexarSpatialSearchParams{
                .h3index = 621807530350542847,
                .heading = RoundedHeading::SouthEast},
            NexarSpatialSearchParams{
                .h3index = 621807530350542847,
                .heading = RoundedHeading::NorthWest},
            NexarSpatialSearchParams{
                .h3index = 621807530350673919,
                .heading = RoundedHeading::SouthEast},
            NexarSpatialSearchParams{
                .h3index = 621807530350673919,
                .heading = RoundedHeading::NorthWest},
            NexarSpatialSearchParams{
                .h3index = 621807530981163007,
                .heading = RoundedHeading::NorthWest}));
}

} // namespace maps::mrc::import_nexar::tests
