#include "common.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/metadata_gateway.h>
#include <maps/libs/log8/include/log8.h>

#include <vector>

namespace maps::mrc::import_taxi {

namespace {

const std::string APP_NAME = "import_taxi";
const std::string LAST_RUN_TIME_KEY = APP_NAME + ".last_run_timestamp";
const std::string MAX_PROCESSED_EVENT_ID_KEY = APP_NAME + ".max_processed_event_id";

} // namespace

void updateMetadata(pgpool3::Pool& pool, db::TId maxProcessedEventId)
{
    INFO() << "Update metadata";
    auto now = chrono::TimePoint::clock::now();
    auto txn = pool.masterWriteableTransaction();
    auto gtw = db::MetadataGateway{*txn};

    gtw.upsert(std::vector<db::Metadata>{
        {LAST_RUN_TIME_KEY, chrono::formatSqlDateTime(now)},
        {MAX_PROCESSED_EVENT_ID_KEY, std::to_string(maxProcessedEventId)}
    });
    txn->commit();
}

Metadata getMetadata(pgpool3::Pool& pool)
{
    Metadata metadata{};

    auto txn = pool.masterReadOnlyTransaction();
    auto gtw = db::MetadataGateway{*txn};
    auto records = gtw.loadByIds({LAST_RUN_TIME_KEY, MAX_PROCESSED_EVENT_ID_KEY});

    for (const auto& r : records) {
        if (r.key() == LAST_RUN_TIME_KEY) {
            metadata.lastRunTimestamp = chrono::parseSqlDateTime(r.value());
        } else if (r.key() == MAX_PROCESSED_EVENT_ID_KEY) {
            metadata.maxProcessedEventId = std::stoull(r.value());
        }
    }
    return metadata;
}

chrono::TimePoint toTimePoint(double seconds)
{
    return chrono::sinceEpochToTimePoint<std::chrono::milliseconds>(
        static_cast<uint64_t>(seconds * 1000));
}

} // namespace maps::mrc::import_taxi
