#include "track_provider.h"
#include "yt_track_mapper.h"

#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/io.h>
#include <maps/libs/log8/include/log8.h>

#include <iomanip>

namespace maps::mrc::import_taxi {

namespace {

constexpr auto DURATION_BEFORE = std::chrono::seconds{5};
constexpr auto DURATION_AFTER = std::chrono::seconds{65};

chrono::TimePoint dayStart(chrono::TimePoint timePoint)
{
    return std::chrono::floor<std::chrono::days>(timePoint);
}

std::string dailyTableName(chrono::TimePoint timePoint)
{
    std::chrono::year_month_day ymd{std::chrono::floor<std::chrono::days>(timePoint)};

    std::ostringstream oss;
    oss << static_cast<int>(ymd.year())
        << "-" << std::setfill('0') << std::setw(2) << static_cast<unsigned>(ymd.month())
        << "-" << std::setfill('0') << std::setw(2) << static_cast<unsigned>(ymd.day());

    return oss.str();
}

DeviceIdToTimeIntervals collectTimeIntervals(const VideoEvents& videoEvents)
{
    DeviceIdToTimeIntervals result;
    for (const auto& event : videoEvents) {
        auto begin = event.eventTime - DURATION_BEFORE;
        auto end = event.eventTime + DURATION_AFTER;
        result[event.sourceId].push_back({begin, end});
    }
    for (auto& [deviceId, intervals] : result) {
        intervals = mergeTimeIntervals(std::move(intervals));
    }
    return result;
}

std::vector<std::string> evalTableNames(
    const DeviceIdToTimeIntervals& deviceIdToTimeIntervals)
{
    std::set<std::string> tableNames;
    for (const auto& [deviceId, timeIntervals] : deviceIdToTimeIntervals) {
        for (const auto& timeInterval : timeIntervals) {
            auto from = timeInterval.begin;
            while (from < timeInterval.end) {
                tableNames.insert(dailyTableName(from));
                from = dayStart(from + std::chrono::days{1});
            }

        }
    }
    return {tableNames.begin(), tableNames.end()};
}

} // namespace

TrackProvider::TrackProvider(
    NYT::IClientBasePtr ytClient,
    const std::string& trackTablesDir,
    const std::string& tmpTablePath
)
    : ytClient_(ytClient)
    , trackTablesDir_(trackTablesDir)
    , tmpTablePath_(tmpTablePath)
{}

void TrackProvider::preloadTracks(const VideoEvents& videoEvents)
{
    if (videoEvents.empty()) {
        return;
    }

    INFO() << "Preload tracks for " << videoEvents.size() << " events";
    auto deviceIdToTimeIntervals = collectTimeIntervals(videoEvents);
    std::vector<TString> inputTablePaths;

    auto ytTxn = ytClient_->StartTransaction();

    for (const auto& tableName : evalTableNames(deviceIdToTimeIntervals)) {
        auto tablePath = TString(trackTablesDir_ + "1d/" + tableName);
        if (ytTxn->Exists(tablePath)) {
            INFO() << "Use table " << tablePath;
            inputTablePaths.push_back(tablePath);
        } else {
            WARN() << "Table does not exist: " << tablePath;
        }
    }

    if (inputTablePaths.empty()) {
        WARN() << "No input tables found to preload tracks";
        return;
    }

    loadSelectedTracks(*ytTxn, inputTablePaths, tmpTablePath_, deviceIdToTimeIntervals);
    INFO() << "Tracks preloaded to " << tmpTablePath_;

    auto reader = ytTxn->CreateTableReader<NYT::TNode>(tmpTablePath_);

    for (; reader->IsValid(); reader->Next()) {
        auto record = yt::deserialize<DeviceTrackRecord>(reader->GetRow());

        db::TrackPoint trackPoint{};
        trackPoint.setSourceId(record.deviceId)
            .setTimestamp(record.timePoint())
            .setGeodeticPos(geolib3::Point2(record.lon, record.lat));
        if (record.speed) {
            trackPoint.setSpeedMetersPerSec(*record.speed);
        }
        if (record.direction) {
            trackPoint.setHeading(geolib3::Heading(*record.direction));
        }
        deviceTracks_[record.deviceId].push_back(std::move(trackPoint));
    }

    for (auto& [deviceId, track] : deviceTracks_) {
        std::sort(track.begin(), track.end(),
            [](const auto& lhs, const auto& rhs) {
                return lhs.timestamp() < rhs.timestamp();
            });
    }

    INFO() << "Track preloaded for " << deviceTracks_.size() << " deviceIds";
    ytTxn->Remove(tmpTablePath_);
    ytTxn->Commit();
}

void TrackProvider::clearTracks()
{
    deviceTracks_.clear();
}

db::TrackPoints TrackProvider::getTrack(const VideoEvent& videoEvent) const
{
    auto itr = deviceTracks_.find(videoEvent.sourceId);
    if (itr == deviceTracks_.end()) {
        return {};
    }

    const auto& fullTrack = itr->second;
    auto from = videoEvent.eventTime - DURATION_BEFORE;
    auto to = videoEvent.eventTime + DURATION_AFTER;

    auto cmp = [](const db::TrackPoint& trackPoint, chrono::TimePoint timePoint) {
        return trackPoint.timestamp() < timePoint;
    };

    auto fromIt = std::lower_bound(fullTrack.begin(), fullTrack.end(), from, cmp);
    auto toIt = std::lower_bound(fromIt, fullTrack.end(), to, cmp);

    return {fromIt, toIt};
}


} // namespace maps::mrc::import_taxi
