#include "track_utils.h"

#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/segment.h>

#include <chrono>

namespace maps::mrc::import_taxi {

namespace {

template <typename Duration>
std::chrono::milliseconds toMillis(Duration interval)
{
    using namespace std::chrono;
    return duration_cast<milliseconds>(interval);
}

template <typename Duration>
double toSeconds(Duration interval)
{
    return std::chrono::duration<double>(interval).count();
}

db::TrackPoint pointByFactor(const adapters::TrackSegment& segment, double ratio)
{
    ASSERT(0.0 <= ratio && ratio <= 1.0);

    auto position = segment.segment.pointByPosition(ratio);
    auto timestamp = segment.startTime
        + toMillis((segment.endTime - segment.startTime) * ratio);

    return db::TrackPoint{}
        .setGeodeticPos(position)
        .setTimestamp(timestamp)
        .setHeading(geolib3::Direction2(segment.segment).heading());
}

} // namespace

db::TrackPoints pickTrackPoints(
    const adapters::TrackSegments& trackSegments,
    std::chrono::milliseconds timeStep,
    double spaceStepMeters)
{
    db::TrackPoints result;
    if (trackSegments.empty()) {
        return result;
    }

    for (const auto& segment : trackSegments) {
        auto segmentDuration = segment.endTime - segment.startTime;
        auto segmentLength = geolib3::geoLength(segment.segment);

        auto timeDistance = std::chrono::milliseconds{0};
        auto spaceDistance = 0.0;
        auto ratio = 0.0;

        while (ratio < 1.0) {
            result.push_back(pointByFactor(segment, ratio));

            ratio = std::min(
                toSeconds(timeDistance + timeStep) / toSeconds(segmentDuration),
                (spaceDistance + spaceStepMeters) / segmentLength);

            timeDistance = toMillis(segmentDuration * ratio);
            spaceDistance = segmentLength * ratio;
        }
    }
    return result;
}

} // namespace maps::mrc::import_taxi
