#include "s3_client.h"
#include "static_events_feed.h"
#include "test_track_provider.h"

#include <library/cpp/testing/gtest/gtest.h>

#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/http/include/client.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool/include/pgpool3.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/import_config_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/import_taxi_event_config_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/video_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/write.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/export_gen/lib/exporter.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_taxi/lib/importer.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_taxi/lib/video_events_feed.h>


#include <yandex/maps/mrc/unittest/database_fixture.h>
#include <yandex/maps/mrc/unittest/local_server.h>
#include <yandex/maps/mrc/unittest/unittest_config.h>

namespace maps::mrc::import_taxi::tests {

const std::string TEST_DATA_DIR = BinaryPath("maps/wikimap/mapspro/services/mrc/long_tasks/import_taxi/tests/base_tests/data");
const std::string TEST_GRAPH_PATH = BinaryPath("maps/data/test/graph4");
const std::string TEST_GEOID_PATH = BinaryPath("maps/data/test/geoid");

class Playground : public unittest::WithUnittestConfig<
    unittest::DatabaseFixture,
    unittest::MdsStubFixture>
{
public:
    Playground()
    {
        s3Client_.createBucket();
    }

    S3Client& s3Client() { return s3Client_; }

    static auto& instance() {
        static Playground playground;
        return playground;
    }

private:
    S3Client s3Client_;
};


struct ImportFixture : testing::Test
{
public:
    void clearData()
    {
        auto mds = Playground::instance().config().makeMdsClient();
        {
            auto txn = pool().slaveTransaction();
            for (const auto& feature : db::FeatureGateway{*txn}.load()) {
                mds.del(feature.mdsKey());
            }
            for (const auto& video : db::VideoGateway{*txn}.load()) {
                mds.del(video.mdsKey());
            }
        }
        Playground::instance().postgres().truncateTables();
        Playground::instance().s3Client().emptyBucket();
    }

    void insertFeatures(db::Features& features)
    {
        auto txn = pool().masterWriteableTransaction();
        db::FeatureGateway{*txn}.insert(features);
        txn->commit();
    }

    void insertImportConfigs(db::ImportConfigs importConfigs)
    {
        auto txn = pool().masterWriteableTransaction();
        db::ImportConfigGateway{*txn}.insert(std::move(importConfigs));
        txn->commit();
    }

    void insertTaxiEventsConfigs(db::ImportTaxiEventConfig eventConfigs)
    {
        auto txn = pool().masterWriteableTransaction();
        db::ImportTaxiEventConfigGateway{*txn}.insert(std::move(eventConfigs));
        txn->commit();
    }

    void makeCoverage()
    {
        auto ctx =
            export_gen::Context(TEST_GRAPH_PATH, db::GraphType::Road);
        auto features = db::FeatureGateway(*pool().slaveTransaction()).load();
        auto photos = export_gen::Photos{};
        for (const auto& feature : features) {
            photos.push_back(export_gen::toPhoto(feature));
        }
        auto [graph, _] = export_gen::makeGraphSummary(
            ctx, photos, export_gen::makeTrackPointProvider(pool()));
        graph.mrcVersion = fb::makeVersion(std::chrono::system_clock::now());
        fb::writeToFile(graph, TEST_GRAPH_PATH + "/graph_coverage.fb");
    }

    Importer makeImporter(std::vector<VideoEvent> videoEvents)
    {
        return Importer(
            pool(),
            Playground::instance().config().makeMdsClient(),
            std::make_shared<StaticEventsFeed>(std::move(videoEvents)),
            std::make_unique<TestTrackProvider>(),
            TEST_GRAPH_PATH,
            TEST_GEOID_PATH + "/geoid.mms.1");
    }

    void addS3Video(const std::string& key, const std::string& fileName)
    {
        Playground::instance().s3Client().uploadFile(key, fileName);
    }

    std::string getS3VideoUrl(const std::string& key)
    {
        return Playground::instance().s3Client().makeReadingUrl(key);
    }

    pgpool3::Pool& pool() { return Playground::instance().pool(); }

    const common::Config& config() { return Playground::instance().config(); }
};

}  // namespace maps::mrc::import_taxi::tests
