#include "fixture.h"

#include <library/cpp/testing/gtest/gtest.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/http/include/client.h>
#include <maps/libs/log8/include/log8.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/track_point.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_taxi/lib/importer.h>

#include <yandex/maps/mrc/unittest/database_fixture.h>
#include <yandex/maps/mrc/unittest/local_server.h>
#include <yandex/maps/mrc/unittest/unittest_config.h>

#include <chrono>
#include <thread>

using namespace std::literals::chrono_literals;
using std::chrono::days;

namespace maps::mrc::import_taxi::tests {

namespace {

const std::string TEST_VIDEO_S3_KEY = "test-video-key";
const std::string SOURCE_ID = "signalq2-source";
const std::string EVENT_TYPE = "mrc_capture";
const std::string VIDEO_PATH = static_cast<std::string>(TEST_DATA_DIR + "/video.mp4");
constexpr uint64_t MOSCOW_GEO_ID = 213;
constexpr uint64_t RUSSIA_GEO_ID = 225;

constexpr int32_t FC(int32_t fc) { return fc; }

template <typename Duration>
int64_t toMillis(Duration interval)
{
    using namespace std::chrono;
    return duration_cast<milliseconds>(interval).count();
}

db::Feature makeFeature(
    chrono::TimePoint timestamp,
    geolib3::Point2 pos,
    double heading)
{
    return sql_chemistry::GatewayAccess<db::Feature>::construct()
        .setTimestamp(timestamp)
        .setGeodeticPos(pos)
        .setHeading(geolib3::Heading{heading})
        .setSourceId("source-id")
        .setDataset(db::Dataset::Rides)
        .setSize(800, 600)
        .setMdsKey(mds::Key("100500", "import-test"))
        .setGraph(db::GraphType::Road)
        .setUserId("test-uid")
        .setModeratorsShouldBePublished(true)
        .setAutomaticShouldBePublished(true)
        .setIsPublished(true);
}

db::Features makeTestFeaturesWithFullCoverage(chrono::TimePoint startTimestamp)
{
    return db::Features{
        makeFeature(startTimestamp + 0s, geolib3::Point2(37.627773, 55.755602), 70.),
        makeFeature(startTimestamp + 1s, geolib3::Point2(37.628022, 55.755675), 70.),
        makeFeature(startTimestamp + 2s, geolib3::Point2(37.628280, 55.755759), 70.),
        makeFeature(startTimestamp + 3s, geolib3::Point2(37.628577, 55.755857), 70.),
        makeFeature(startTimestamp + 4s, geolib3::Point2(37.628855, 55.755960), 65.)
    };
}

db::Features makeTestFeaturesWithPartialCoverage(chrono::TimePoint startTimestamp)
{
    return db::Features{
        makeFeature(startTimestamp + 0s, geolib3::Point2(37.628022, 55.755675), 70.),
        makeFeature(startTimestamp + 2s, geolib3::Point2(37.628280, 55.755759), 70.),
        makeFeature(startTimestamp + 3s, geolib3::Point2(37.628577, 55.755857), 70.),
        makeFeature(startTimestamp + 4s, geolib3::Point2(37.628855, 55.755960), 65.)
    };
}

db::TrackPoint makeTrackPoint(chrono::TimePoint timestamp, geolib3::Point2 pos)
{
    return db::TrackPoint{}
        .setTimestamp(timestamp)
        .setGeodeticPos(pos)
        .setSourceId(SOURCE_ID);
}

chrono::TimePoint makeTimestamp()
{
    auto ts = chrono::sinceEpochToTimePoint<std::chrono::seconds>(
        chrono::sinceEpoch<std::chrono::seconds>());
    // Make events older than 1 day, otherwise importer will skip them
    return ts - std::chrono::hours(25);
}

VideoEvents makeVideoEvents(ImportFixture& fixture, chrono::TimePoint eventTs)
{
    fixture.addS3Video(TEST_VIDEO_S3_KEY, VIDEO_PATH);

    return VideoEvents{
        VideoEvent{
            1, EVENT_TYPE, fixture.getS3VideoUrl(TEST_VIDEO_S3_KEY), SOURCE_ID, eventTs,
            db::TrackPoints{
                makeTrackPoint(eventTs - 2s, geolib3::Point2(37.628086, 55.755697)),
                makeTrackPoint(eventTs - 1s, geolib3::Point2(37.628349, 55.755778)),
                makeTrackPoint(eventTs, geolib3::Point2(37.628602, 55.755865))
            }
        }
    };
}

} // namespace

TEST_F(ImportFixture, test_s3_mock)
{
    addS3Video(TEST_VIDEO_S3_KEY, VIDEO_PATH);
    auto readingUrl = getS3VideoUrl(TEST_VIDEO_S3_KEY);

    http::Client httpClient;
    auto [body, status] = httpClient.get(readingUrl, maps::common::RetryPolicy());
    EXPECT_EQ(status, 200);
    EXPECT_EQ(body, maps::common::readFileToString(VIDEO_PATH));
}

TEST_F(ImportFixture, test_import_with_empty_events)
{
    auto features = makeTestFeaturesWithFullCoverage(chrono::parseSqlDateTime("2021-01-01 12:00:00+03"));
    clearData();
    insertFeatures(features);
    makeCoverage();

    auto importer = makeImporter(VideoEvents{});

    importer.import();

    // Check that nothing was imported
    auto txn = pool().slaveTransaction();
    auto loadedFeatures = db::FeatureGateway{*txn}.load();
    EXPECT_EQ(loadedFeatures.size(), features.size());
    EXPECT_EQ(db::VideoGateway{*txn}.count(), 0u);
    EXPECT_EQ(db::FrameToVideoGateway{*txn}.count(), 0u);
}


TEST_F(ImportFixture, test_import_with_all_events_younger_than_1day)
{
    const auto EVENT_TS = maps::chrono::TimePoint::clock::now()
        - std::chrono::hours(23);

    auto features = makeTestFeaturesWithFullCoverage(EVENT_TS - days{30});
    clearData();
    insertFeatures(features);
    makeCoverage();

    auto importConfigs = db::ImportConfigs{
        db::ImportConfig{db::Dataset::TaxiSignalQ2, MOSCOW_GEO_ID, FC(7), days{20}.count()},
    };
    insertImportConfigs(importConfigs);
    insertTaxiEventsConfigs({ db::ImportTaxiEventConfig{EVENT_TYPE, 2} });

    auto videoEvents = VideoEvents{
        VideoEvent{1, EVENT_TYPE, "mock://url-1.mp4", SOURCE_ID, EVENT_TS, db::TrackPoints{}},
        VideoEvent{2, EVENT_TYPE, "mock://url-2.mp4", SOURCE_ID, EVENT_TS + 10s, db::TrackPoints{}},
        VideoEvent{3, EVENT_TYPE, "mock://url-3.mp4", SOURCE_ID, EVENT_TS + 20s, db::TrackPoints{}}
    };

    auto importer = makeImporter(videoEvents);

    importer.import();

    // Check that nothing was imported
    auto txn = pool().slaveTransaction();
    auto loadedFeatures = db::FeatureGateway{*txn}.load();
    EXPECT_EQ(loadedFeatures.size(), features.size());
    EXPECT_EQ(db::VideoGateway{*txn}.count(), 0u);
    EXPECT_EQ(db::FrameToVideoGateway{*txn}.count(), 0u);

    // Make sure the events are still unprocessed
    EXPECT_EQ(getMetadata(pool()).maxProcessedEventId, videoEvents.front().id);
}

TEST_F(ImportFixture, test_import_with_empty_config)
{
    const auto EVENT_TS = makeTimestamp();

    auto features = makeTestFeaturesWithFullCoverage(EVENT_TS - days{30});
    clearData();
    insertFeatures(features);
    makeCoverage();

    auto videoEvents = makeVideoEvents(*this, EVENT_TS);

    auto importer = makeImporter(videoEvents);

    importer.import();

    // Check that nothing was imported
    auto txn = pool().slaveTransaction();
    auto loadedFeatures = db::FeatureGateway{*txn}.load();
    EXPECT_EQ(loadedFeatures.size(), features.size());
    EXPECT_EQ(db::VideoGateway{*txn}.count(), 0u);
    EXPECT_EQ(db::FrameToVideoGateway{*txn}.count(), 0u);

    EXPECT_EQ(getMetadata(pool()).maxProcessedEventId, videoEvents.back().id);
}

TEST_F(ImportFixture, test_dont_import_unknown_event_type)
{
    const auto EVENT_TS = makeTimestamp();

    clearData();
    auto importConfigs = db::ImportConfigs{
        db::ImportConfig{db::Dataset::TaxiSignalQ2, MOSCOW_GEO_ID, FC(7), days{20}.count()},
    };
    insertImportConfigs(importConfigs);

    // Don't add taxi events config for EVENT_TYPE

    auto videoEvents = makeVideoEvents(*this, EVENT_TS);

    auto importer = makeImporter(videoEvents);
    importer.import();

    auto txn = pool().slaveTransaction();
    auto videos = db::VideoGateway{*txn}.load();
    EXPECT_TRUE(videos.empty());
}

TEST_F(ImportFixture, test_import_basic_case)
{
    const auto EVENT_TS = makeTimestamp();

    auto features = makeTestFeaturesWithFullCoverage(EVENT_TS - days{30});
    clearData();
    insertFeatures(features);
    makeCoverage();

    auto importConfigs = db::ImportConfigs{
        db::ImportConfig{db::Dataset::TaxiSignalQ2, MOSCOW_GEO_ID, FC(7), days{20}.count()},
    };
    insertImportConfigs(importConfigs);
    insertTaxiEventsConfigs({ db::ImportTaxiEventConfig{EVENT_TYPE, 2} });

    auto videoEvents = makeVideoEvents(*this, EVENT_TS);

    auto importer = makeImporter(videoEvents);

    importer.import();

    const size_t EXPECTED_NEW_FEATURES_SIZE = 8;

    auto txn = pool().slaveTransaction();
    auto videos = db::VideoGateway{*txn}.load();
    auto loadedFeatures = db::FeatureGateway{*txn}.load();
    auto framesToVideo = db::FrameToVideoGateway{*txn}.load();
    EXPECT_EQ(videos.size(), 1u);
    ASSERT_EQ(loadedFeatures.size(), features.size() + EXPECTED_NEW_FEATURES_SIZE);
    EXPECT_EQ(framesToVideo.size(), EXPECTED_NEW_FEATURES_SIZE);

    EXPECT_TRUE(videos[0].eventId());
    EXPECT_EQ(*videos[0].eventId(), 1u);

    auto mds = config().makeMdsClient();
    for (size_t i = 0; i < framesToVideo.size(); ++i) {
        auto featureIdx = features.size() + i;
        EXPECT_EQ(loadedFeatures[featureIdx].sourceId(), SOURCE_ID);
        EXPECT_EQ(loadedFeatures[featureIdx].dataset(), db::Dataset::TaxiSignalQ2);
        EXPECT_EQ(framesToVideo[i].videoId(), videos[0].id());
        EXPECT_EQ(framesToVideo[i].frameId(), loadedFeatures[featureIdx].id());

        auto imageData = maps::common::readFileToString(TEST_DATA_DIR + "/frame_" + std::to_string(i) + ".jpg");
        EXPECT_EQ(imageData, mds.get(loadedFeatures[featureIdx].mdsKey()));
    }

    auto offset = features.size();
    EXPECT_EQ(toMillis(EVENT_TS - loadedFeatures[offset].timestamp()), 2000);
    EXPECT_EQ(loadedFeatures[offset].geodeticPos(), geolib3::Point2(37.6281066773, 55.7556762225));
    EXPECT_EQ(toMillis(EVENT_TS - loadedFeatures[offset + 1].timestamp()), 1734);
    EXPECT_EQ(loadedFeatures[offset + 1].geodeticPos(), geolib3::Point2(37.628176195, 55.7556981297));
    EXPECT_EQ(toMillis(EVENT_TS - loadedFeatures[offset + 2].timestamp()), 1467);
    EXPECT_EQ(loadedFeatures[offset + 2].geodeticPos(), geolib3::Point2(37.6282457127, 55.7557200369));
    EXPECT_EQ(toMillis(EVENT_TS - loadedFeatures[offset + 3].timestamp()), 1200);
    EXPECT_EQ(loadedFeatures[offset + 3].geodeticPos(), geolib3::Point2(37.6283152303, 55.755741944));
    EXPECT_EQ(toMillis(EVENT_TS - loadedFeatures[offset + 4].timestamp()), 1000);
    EXPECT_EQ(loadedFeatures[offset + 4].geodeticPos(), geolib3::Point2(37.6283682536, 55.7557586533));
    EXPECT_EQ(toMillis(EVENT_TS - loadedFeatures[offset + 5].timestamp()), 700);
    EXPECT_EQ(loadedFeatures[offset + 5].geodeticPos(), geolib3::Point2(37.6284377713, 55.7557805605));
    EXPECT_EQ(toMillis(EVENT_TS - loadedFeatures[offset + 6].timestamp()), 434);
    EXPECT_EQ(loadedFeatures[offset + 6].geodeticPos(), geolib3::Point2(37.6285072891, 55.7558024677));
    EXPECT_EQ(toMillis(EVENT_TS - loadedFeatures[offset + 7].timestamp()), 167);
    EXPECT_EQ(loadedFeatures[offset + 7].geodeticPos(), geolib3::Point2(37.6285768069, 55.7558243749));

    EXPECT_EQ(getMetadata(pool()).maxProcessedEventId, videoEvents.back().id);
}

TEST_F(ImportFixture, test_import_event_config)
{
    const auto EVENT_TS = makeTimestamp();

    auto videoEvents = VideoEvents{
        VideoEvent{
            1, EVENT_TYPE, getS3VideoUrl(TEST_VIDEO_S3_KEY), SOURCE_ID, EVENT_TS,
            db::TrackPoints{
                makeTrackPoint(EVENT_TS - 3s, geolib3::Point2(37.627875, 55.755628)),
                makeTrackPoint(EVENT_TS - 2s, geolib3::Point2(37.628086, 55.755697)),
                makeTrackPoint(EVENT_TS - 1s, geolib3::Point2(37.628349, 55.755778)),
                makeTrackPoint(EVENT_TS,      geolib3::Point2(37.628602, 55.755865)),
                makeTrackPoint(EVENT_TS + 1s, geolib3::Point2(37.628778, 55.755929)),
                makeTrackPoint(EVENT_TS + 2s, geolib3::Point2(37.628968, 55.756004)),
                makeTrackPoint(EVENT_TS + 3s, geolib3::Point2(37.629152, 55.756101))
            }
        }
    };

    // Import event with video starting 2 seconds before the event time
    {
        clearData();
        addS3Video(TEST_VIDEO_S3_KEY, VIDEO_PATH);
        insertImportConfigs({ db::ImportConfig{db::Dataset::TaxiSignalQ2, MOSCOW_GEO_ID, FC(7), 20} });
        insertTaxiEventsConfigs({ db::ImportTaxiEventConfig{EVENT_TYPE, 2} });

        auto importer = makeImporter(videoEvents);
        importer.import();
        auto txn = pool().slaveTransaction();
        auto features = db::FeatureGateway{*txn}.load();
        ASSERT_EQ(features.size(), 10u);
        // Check that first photo is 2 seconds before event time
        EXPECT_EQ(EVENT_TS, features[0].timestamp() + 2000ms);
        EXPECT_EQ(features[0].geodeticPos(), geolib3::Point2(37.6281066773, 55.7556762225));
    }

    // Import save event with video starting exactly at event time
    {
        clearData();
        addS3Video(TEST_VIDEO_S3_KEY, VIDEO_PATH);
        insertImportConfigs({ db::ImportConfig{db::Dataset::TaxiSignalQ2, MOSCOW_GEO_ID, FC(7), 20} });
        insertTaxiEventsConfigs({ db::ImportTaxiEventConfig{EVENT_TYPE, 0} });

        auto importer = makeImporter(videoEvents);
        importer.import();
        auto txn = pool().slaveTransaction();
        auto features = db::FeatureGateway{*txn}.load();
        ASSERT_EQ(features.size(), 10u);
        // Check that first photo has exactly event time
        EXPECT_EQ(EVENT_TS, features[0].timestamp());
        EXPECT_EQ(features[0].geodeticPos(), geolib3::Point2(37.6286267628, 55.7558401175));
    }
}

TEST_F(ImportFixture, test_import_existing_coverage_is_partial)
{
    const auto EVENT_TS = makeTimestamp();

    auto features = makeTestFeaturesWithPartialCoverage(EVENT_TS - days{20});
    clearData();
    insertFeatures(features);
    makeCoverage();

    auto importConfigs = db::ImportConfigs{
        db::ImportConfig{db::Dataset::TaxiSignalQ2, MOSCOW_GEO_ID, FC(7), days{30}.count()},
    };
    insertImportConfigs(importConfigs);
    insertTaxiEventsConfigs({ db::ImportTaxiEventConfig{EVENT_TYPE, 2} });

    auto videoEvents = makeVideoEvents(*this, EVENT_TS);

    auto importer = makeImporter(videoEvents);
    importer.import();

    const size_t EXPECTED_NEW_FEATURES_SIZE = 8;

    // Old features made only partial coverage, so new features have been imported
    auto txn = pool().slaveTransaction();
    auto videos = db::VideoGateway{*txn}.load();
    auto loadedFeatures = db::FeatureGateway{*txn}.load();
    auto framesToVideo = db::FrameToVideoGateway{*txn}.load();
    EXPECT_EQ(videos.size(), 1u);
    EXPECT_EQ(loadedFeatures.size(), features.size() + EXPECTED_NEW_FEATURES_SIZE);
    EXPECT_EQ(framesToVideo.size(), EXPECTED_NEW_FEATURES_SIZE);

    EXPECT_EQ(getMetadata(pool()).maxProcessedEventId, videoEvents.back().id);
}

TEST_F(ImportFixture, test_import_existing_coverage_is_actual)
{
    const auto EVENT_TS = makeTimestamp();

    auto features = makeTestFeaturesWithFullCoverage(EVENT_TS - days{20});
    clearData();
    insertFeatures(features);
    makeCoverage();

    auto importConfigs = db::ImportConfigs{
        db::ImportConfig{db::Dataset::TaxiSignalQ2, MOSCOW_GEO_ID, FC(7), days{30}.count()},
    };
    insertImportConfigs(importConfigs);
    insertTaxiEventsConfigs({ db::ImportTaxiEventConfig{EVENT_TYPE, 2} });

    auto videoEvents = makeVideoEvents(*this, EVENT_TS);

    auto importer = makeImporter(videoEvents);
    importer.import();

    // Check that nothing was imported
    auto txn = pool().slaveTransaction();
    auto loadedFeatures = db::FeatureGateway{*txn}.load();
    EXPECT_EQ(loadedFeatures.size(), features.size());
    EXPECT_EQ(db::VideoGateway{*txn}.count(), 0u);
    EXPECT_EQ(db::FrameToVideoGateway{*txn}.count(), 0u);

    EXPECT_EQ(getMetadata(pool()).maxProcessedEventId, videoEvents.back().id);
}

TEST_F(ImportFixture, test_import_geo_id_order_and_fc)
{
    const auto EVENT_TS = makeTimestamp();

    auto features = makeTestFeaturesWithFullCoverage(EVENT_TS - days{30});
    clearData();
    insertFeatures(features);
    makeCoverage();

    // Record with Moscow geoId will be used, so video will not be imported
    auto importConfigs = db::ImportConfigs{
        db::ImportConfig{db::Dataset::TaxiSignalQ2, MOSCOW_GEO_ID, FC(7), days{40}.count()},
        db::ImportConfig{db::Dataset::TaxiSignalQ2, MOSCOW_GEO_ID, FC(8), days{20}.count()},
        db::ImportConfig{db::Dataset::TaxiSignalQ2, RUSSIA_GEO_ID, FC(7), days{20}.count()},
    };
    insertImportConfigs(importConfigs);
    insertTaxiEventsConfigs({ db::ImportTaxiEventConfig{EVENT_TYPE, 2} });

    auto videoEvents = makeVideoEvents(*this, EVENT_TS);

    auto importer = makeImporter(videoEvents);
    importer.import();

    // Check that nothing was imported
    auto txn = pool().slaveTransaction();
    auto loadedFeatures = db::FeatureGateway{*txn}.load();
    EXPECT_EQ(loadedFeatures.size(), features.size());
    EXPECT_EQ(db::VideoGateway{*txn}.count(), 0u);
    EXPECT_EQ(db::FrameToVideoGateway{*txn}.count(), 0u);

    EXPECT_EQ(getMetadata(pool()).maxProcessedEventId, videoEvents.back().id);
}

TEST_F(ImportFixture, test_dont_import_same_event_twice)
{
    const auto EVENT_TS = makeTimestamp();

    auto features = makeTestFeaturesWithPartialCoverage(EVENT_TS - days{20});
    clearData();
    insertFeatures(features);
    makeCoverage();

    auto importConfigs = db::ImportConfigs{
        db::ImportConfig{db::Dataset::TaxiSignalQ2, MOSCOW_GEO_ID, FC(7), days{30}.count()},
    };
    insertImportConfigs(importConfigs);
    insertTaxiEventsConfigs({ db::ImportTaxiEventConfig{EVENT_TYPE, 2} });

    auto videoEvents = makeVideoEvents(*this, EVENT_TS);

    auto importer = makeImporter(videoEvents);
    importer.import();

    const size_t EXPECTED_NEW_FEATURES_SIZE = 8;

    auto txn = pool().slaveTransaction();
    auto videos = db::VideoGateway{*txn}.load();
    auto loadedFeatures = db::FeatureGateway{*txn}.load();
    auto framesToVideo = db::FrameToVideoGateway{*txn}.load();
    EXPECT_EQ(videos.size(), 1u);
    EXPECT_EQ(loadedFeatures.size(), features.size() + EXPECTED_NEW_FEATURES_SIZE);
    EXPECT_EQ(framesToVideo.size(), EXPECTED_NEW_FEATURES_SIZE);

    EXPECT_EQ(getMetadata(pool()).maxProcessedEventId, videoEvents.back().id);

    importer.setEventsFeed(std::make_shared<StaticEventsFeed>(std::move(videoEvents)));
    importer.import();

    // Check that no new photos/videos were imported
    txn = pool().slaveTransaction();
    videos = db::VideoGateway{*txn}.load();
    loadedFeatures = db::FeatureGateway{*txn}.load();
    EXPECT_EQ(videos.size(), 1u);
    EXPECT_EQ(loadedFeatures.size(), features.size() + EXPECTED_NEW_FEATURES_SIZE);
}

TEST_F(ImportFixture, test_dont_import_event_with_same_hash_twice)
{
    const auto EVENT_TS = makeTimestamp();

    auto features = makeTestFeaturesWithPartialCoverage(EVENT_TS - days{20});
    clearData();
    insertFeatures(features);
    makeCoverage();

    auto importConfigs = db::ImportConfigs{
        db::ImportConfig{db::Dataset::TaxiSignalQ2, MOSCOW_GEO_ID, FC(7), days{30}.count()},
    };
    insertImportConfigs(importConfigs);
    insertTaxiEventsConfigs({ db::ImportTaxiEventConfig{EVENT_TYPE, 2} });

    auto videoEvents = makeVideoEvents(*this, EVENT_TS);

    auto importer = makeImporter(videoEvents);
    importer.import();

    auto videos = db::VideoGateway{*pool().slaveTransaction()}.load();
    EXPECT_EQ(videos.size(), 1u);
    EXPECT_EQ(getMetadata(pool()).maxProcessedEventId, videoEvents.back().id);

    // Update eventId but keep video url the same
    videoEvents.front().id = 2;
    importer.setEventsFeed(std::make_shared<StaticEventsFeed>(videoEvents));
    importer.import();

    // Check that no new photos/videos were imported
    videos = db::VideoGateway{*pool().slaveTransaction()}.load();
    EXPECT_EQ(videos.size(), 1u);
    EXPECT_EQ(getMetadata(pool()).maxProcessedEventId, videoEvents.back().id);
}

}  // namespace maps::mrc::import_taxi::tests
