#include "s3_client.h"

#include <maps/libs/common/include/exception.h>

#include <contrib/libs/aws-sdk-cpp/aws-cpp-sdk-core/include/aws/core/Aws.h>
#include <contrib/libs/aws-sdk-cpp/aws-cpp-sdk-core/include/aws/core/auth/AWSCredentialsProviderChain.h>
#include <contrib/libs/aws-sdk-cpp/aws-cpp-sdk-core/include/aws/core/client/ClientConfiguration.h>
#include <contrib/libs/aws-sdk-cpp/aws-cpp-sdk-s3/include/aws/s3/model/CreateBucketRequest.h>
#include <contrib/libs/aws-sdk-cpp/aws-cpp-sdk-s3/include/aws/s3/model/ListObjectsRequest.h>
#include <contrib/libs/aws-sdk-cpp/aws-cpp-sdk-s3/include/aws/s3/model/DeleteObjectRequest.h>
#include <contrib/libs/aws-sdk-cpp/aws-cpp-sdk-s3/include/aws/s3/model/PutObjectRequest.h>

#include <atomic>
#include <fstream>

namespace maps::mrc::import_taxi::tests {

namespace {

const std::string TEST_BUCKET = "test-bucket";

std::atomic<int> g_initCount = 0;

struct Api {
    static void init()
    {
        if (g_initCount++ > 0) {
            return;
        }
        Aws::InitAPI(Aws::SDKOptions{});
    }

    static void shutdown()
    {
        if (--g_initCount > 0) {
            return;
        }
        Aws::ShutdownAPI({});
    }

    Api() { init(); }
    ~Api() { shutdown(); }
};

} // namespace


S3Client::S3Client()
    : port_(std::getenv("S3MDS_PORT"))
    , bucket_(TEST_BUCKET)
{
    Api::init();

    Aws::Client::ClientConfiguration config{};
    config.scheme = Aws::Http::Scheme::HTTP;
    config.endpointOverride = "127.0.0.1:" + port_;
    config.connectTimeoutMs = 5000;
    config.requestTimeoutMs = 5000;

    auto auth = std::make_shared<Aws::Auth::SimpleAWSCredentialsProvider>(
        "TestAccessKeyId", "TestAccessKeySecret");

    client_ = std::make_shared<Aws::S3::S3Client>(auth, config,
        Aws::Client::AWSAuthV4Signer::PayloadSigningPolicy::Never,
        /*useVirtualAddressing = */false);
}

void S3Client::createBucket()
{
    Aws::S3::Model::CreateBucketRequest request;
    request.WithBucket(bucket_)
           .WithACL(Aws::S3::Model::BucketCannedACL::public_read);

    auto outcome = client_->CreateBucket(request);
    REQUIRE(outcome.IsSuccess(),
            "Failed to create bucket " << bucket_ << ": " << outcome.GetError());
}

void S3Client::emptyBucket()
{
    Aws::S3::Model::ListObjectsRequest request;
    request.WithBucket(bucket_);

    auto outcome = client_->ListObjects(request);
    REQUIRE(outcome.IsSuccess(), "Failed to list object in bucket " << bucket_);
    for (const auto& object : outcome.GetResult().GetContents()) {
        const auto& key = object.GetKey();
        Aws::S3::Model::DeleteObjectRequest deleteRequest;
        deleteRequest.WithKey(key).WithBucket(bucket_);
        auto deleteOutcome = client_->DeleteObject(deleteRequest);
        REQUIRE(outcome.IsSuccess(), "Failed to delete object with key " << key);
    }
}

void S3Client::upload(const std::string& key, const std::string& data)
{
    Aws::S3::Model::PutObjectRequest request;
    request.SetBucket(bucket_);
    request.SetKey(key);

    auto stream = Aws::MakeShared<Aws::StringStream>("");
    *stream << data.c_str();

    request.SetBody(stream);

    Aws::S3::Model::PutObjectOutcome outcome = client_->PutObject(request);

    REQUIRE(outcome.IsSuccess(),
            "Failed to upload object with key " << key << ": " << outcome.GetError());
}

void S3Client::uploadFile(const std::string& key, const std::string& fileName)
{
 Aws::S3::Model::PutObjectRequest request;
    request.SetBucket(bucket_);
    request.SetKey(key);

    std::shared_ptr<Aws::IOStream> stream = 
        Aws::MakeShared<Aws::FStream>("Tag", 
            fileName.c_str(), 
            std::ios_base::in | std::ios_base::binary);

    request.SetBody(stream);

    Aws::S3::Model::PutObjectOutcome outcome = client_->PutObject(request);

    REQUIRE(outcome.IsSuccess(),
            "Failed to upload object with key " << key << ": " << outcome.GetError());  

}

std::string S3Client::makeReadingUrl(const std::string& key) const
{
    // Use bucket in path, since `bucket.host` style does not work with localhost
    return "http://127.0.0.1:" + port_ + "/" + bucket_ + "/" + key;
}

}  // namespace maps::mrc::import_taxi::tests
