#include "maps/libs/chrono/include/time_point.h"
#include "maps/wikimap/mapspro/services/mrc/libs/graph_matcher_adapter/include/matcher.h"
#include <library/cpp/testing/gtest/gtest.h>

#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/io.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/serialization.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_taxi/lib/track_utils.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_taxi/lib/yt_track_mapper.h>

#include <mapreduce/yt/tests/yt_unittest_lib/yt_unittest_lib.h>
#include <chrono>

using namespace ::testing;
using namespace std::literals::chrono_literals;

namespace maps::mrc::import_taxi::tests {

const auto START_TS = chrono::sinceEpochToTimePoint<std::chrono::seconds>(1600000000);

const auto SEGMENT_1 = adapters::TrackSegment{
    geolib3::Segment2({44.007286, 56.322372}, {44.008058, 56.322412}),
    START_TS,
    START_TS + 2s
};

const auto SEGMENT_2 = adapters::TrackSegment{
    geolib3::Segment2({44.008058, 56.322412}, {44.007776, 56.322917}),
    START_TS + 2s,
    START_TS + 3s
};

TEST(TrackPointsPicker, test_empty_track)
{
    auto trackPoints = pickTrackPoints({}, 1s, 5.0);
    EXPECT_TRUE(trackPoints.empty());
}

TEST(TrackPointsPicker, test_single_point)
{
    auto segments = adapters::TrackSegments{ SEGMENT_1 };
    auto trackPoints = pickTrackPoints(segments, 2100ms, 50.0);

    EXPECT_EQ(trackPoints.size(), 1u);
    EXPECT_EQ(trackPoints[0].geodeticPos(), segments[0].segment.start());
    EXPECT_EQ(trackPoints[0].timestamp(), START_TS);
}

TEST(TrackPointsPicker, test_one_segment_space_step)
{
    auto segments = adapters::TrackSegments{ SEGMENT_1 };
    auto trackPoints = pickTrackPoints(segments, 2100ms, 15.0);

    EXPECT_EQ(trackPoints.size(), 4u);

    EXPECT_EQ(trackPoints[0].geodeticPos(), segments[0].segment.start());
    EXPECT_EQ(trackPoints[0].timestamp(), START_TS);
    EXPECT_EQ(trackPoints[1].geodeticPos(), geolib3::Point2(44.00752739, 56.32238451));
    EXPECT_EQ(trackPoints[1].timestamp(), START_TS + 625ms);
    EXPECT_EQ(trackPoints[2].geodeticPos(), geolib3::Point2(44.00776877, 56.32239701));
    EXPECT_EQ(trackPoints[2].timestamp(), START_TS + 1250ms);
    EXPECT_EQ(trackPoints[3].geodeticPos(), geolib3::Point2(44.00801016, 56.32240952));
    EXPECT_EQ(trackPoints[3].timestamp(), START_TS + 1876ms);
}

TEST(TrackPointsPicker, test_one_segment_time_step)
{
    auto segments = adapters::TrackSegments{ SEGMENT_1 };
    auto trackPoints = pickTrackPoints(segments, 800ms, 50.0);

    EXPECT_EQ(trackPoints.size(), 3u);

    EXPECT_EQ(trackPoints[0].geodeticPos(), segments[0].segment.start());
    EXPECT_EQ(trackPoints[0].timestamp(), START_TS);
    EXPECT_EQ(trackPoints[1].geodeticPos(), geolib3::Point2(44.0075948, 56.322388));
    EXPECT_EQ(trackPoints[1].timestamp(), START_TS + 800ms);
    EXPECT_EQ(trackPoints[2].geodeticPos(), geolib3::Point2(44.0079036, 56.322404));
    EXPECT_EQ(trackPoints[2].timestamp(), START_TS + 1600ms);
}

TEST(TrackPointsPicker, test_two_segments_general_case)
{
    auto segments = adapters::TrackSegments{ SEGMENT_1, SEGMENT_2 };
    auto trackPoints = pickTrackPoints(segments, 400ms, 20.0);

    EXPECT_EQ(trackPoints.size(), 8u);

    EXPECT_EQ(trackPoints[0].geodeticPos(), segments[0].segment.start());
    EXPECT_EQ(trackPoints[0].timestamp(), START_TS);
    EXPECT_EQ(trackPoints[1].geodeticPos(), geolib3::Point2(44.0074404, 56.32238));
    EXPECT_EQ(trackPoints[1].timestamp(), START_TS + 400ms);
    EXPECT_EQ(trackPoints[2].geodeticPos(), geolib3::Point2(44.0075948, 56.322388));
    EXPECT_EQ(trackPoints[2].timestamp(), START_TS + 800ms);
    EXPECT_EQ(trackPoints[3].geodeticPos(), geolib3::Point2(44.0077492, 56.322396));
    EXPECT_EQ(trackPoints[3].timestamp(), START_TS + 1200ms);
    EXPECT_EQ(trackPoints[4].geodeticPos(), geolib3::Point2(44.0079036, 56.322404));
    EXPECT_EQ(trackPoints[4].timestamp(), START_TS + 1600ms);
    EXPECT_EQ(trackPoints[5].geodeticPos(), geolib3::Point2(44.008058, 56.322412));
    EXPECT_EQ(trackPoints[5].timestamp(), START_TS + 2000ms);
    EXPECT_EQ(trackPoints[6].geodeticPos(), geolib3::Point2(44.0079622, 56.32258355));
    EXPECT_EQ(trackPoints[6].timestamp(), START_TS + 2339ms);
    EXPECT_EQ(trackPoints[7].geodeticPos(), geolib3::Point2(44.00786641, 56.3227551));
    EXPECT_EQ(trackPoints[7].timestamp(), START_TS + 2679ms);
}

TEST(MergeIntervals, test_merge_intervals)
{
    auto merged1 = mergeTimeIntervals(TimeIntervals{
        {chrono::parseIsoDateTime("2021-12-12T12:00:00"), chrono::parseIsoDateTime("2021-12-12T12:01:00")}
    });
    EXPECT_EQ(merged1.size(), 1u);
    EXPECT_EQ(merged1[0].begin, chrono::parseIsoDateTime("2021-12-12T12:00:00"));
    EXPECT_EQ(merged1[0].end,   chrono::parseIsoDateTime("2021-12-12T12:01:00"));

    auto merged2 = mergeTimeIntervals(TimeIntervals{
        {chrono::parseIsoDateTime("2021-12-12T12:00:00"), chrono::parseIsoDateTime("2021-12-12T12:01:00")},
        {chrono::parseIsoDateTime("2021-12-12T12:00:30"), chrono::parseIsoDateTime("2021-12-12T12:01:30")}
    });
    EXPECT_EQ(merged2.size(), 1u);
    EXPECT_EQ(merged2[0].begin, chrono::parseIsoDateTime("2021-12-12T12:00:00"));
    EXPECT_EQ(merged2[0].end,   chrono::parseIsoDateTime("2021-12-12T12:01:30"));

    auto merged3 = mergeTimeIntervals(TimeIntervals{
        {chrono::parseIsoDateTime("2021-12-12T12:00:00"), chrono::parseIsoDateTime("2021-12-12T12:01:00")},
        {chrono::parseIsoDateTime("2021-12-12T12:00:30"), chrono::parseIsoDateTime("2021-12-12T12:00:45")},
        {chrono::parseIsoDateTime("2021-12-12T12:01:30"), chrono::parseIsoDateTime("2021-12-12T12:02:00")}
    });
    EXPECT_EQ(merged3.size(), 2u);
    EXPECT_EQ(merged3[0].begin, chrono::parseIsoDateTime("2021-12-12T12:00:00"));
    EXPECT_EQ(merged3[0].end,   chrono::parseIsoDateTime("2021-12-12T12:01:00"));
    EXPECT_EQ(merged3[1].begin, chrono::parseIsoDateTime("2021-12-12T12:01:30"));
    EXPECT_EQ(merged3[1].end,   chrono::parseIsoDateTime("2021-12-12T12:02:00"));
}

}  // namespace maps::mrc::import_taxi::tests
