#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/io.h>
#include <maps/wikimap/mapspro/services/mrc/libs/yt/include/serialization.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_taxi/lib/track_provider.h>
#include <maps/wikimap/mapspro/services/mrc/long_tasks/import_taxi/lib/yt_types.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/env.h>
#include <mapreduce/yt/tests/yt_unittest_lib/yt_unittest_lib.h>
#include <chrono>

namespace maps::mrc::import_taxi::tests {

namespace {

const TString TRACK_TABLES_PATH = "//tmp/taxi-tracks/";
const TString TRACK_TABLE_DAY_1 = TRACK_TABLES_PATH + "1d/2021-12-12";
const TString TRACK_TABLE_DAY_2 = TRACK_TABLES_PATH + "1d/2021-12-13";
const TString OUTPUT_TABLE = "//tmp/output";
const std::string DEVICE_ID_1 = "device-1";
const std::string DEVICE_ID_2 = "device-2";
const std::string EVENT_TYPE = "mrc_capture";

int64_t usec(const std::string& isoDateTime)
{
    auto timePoint = chrono::parseIsoDateTime(isoDateTime);
    return chrono::sinceEpoch<std::chrono::microseconds>(timePoint);
}

class TrackProviderFixture : public NUnitTest::TBaseFixture {
public:
    TrackProviderFixture()
        : ytClient_(NYT::NTesting::CreateTestClient())
    {
        prepareTracks();
    }

    NYT::IClientBasePtr ytClient() { return ytClient_; }

private:
    NYT::IClientBasePtr ytClient_;

    void prepareTracks()
    {
        for (const auto& table : {TRACK_TABLE_DAY_1, TRACK_TABLE_DAY_2}) {
            if (!ytClient()->Exists(table)) {
                ytClient()->Create(table, NYT::NT_TABLE, NYT::TCreateOptions().Recursive(true));
            }
        }

        std::vector<DeviceTrackRecord> tracksDay1 {
            {DEVICE_ID_1, usec("2021-12-12T12:00:00"), 1.0, 1.0, 10.0, 0.0},
            {DEVICE_ID_1, usec("2021-12-12T12:00:01"), 1.1, 1.1, 10.0, 0.0},
            {DEVICE_ID_1, usec("2021-12-12T12:00:02"), 1.2, 1.2, 10.0, 0.0},
            {DEVICE_ID_1, usec("2021-12-12T23:59:58"), 1.3, 1.3, 10.0, 0.0},
            {DEVICE_ID_1, usec("2021-12-12T23:59:59"), 1.4, 1.4, 10.0, 0.0},
            {DEVICE_ID_2, usec("2021-12-12T12:00:00"), 2.0, 2.0, std::nullopt, 0.0},
            {DEVICE_ID_2, usec("2021-12-12T12:00:01"), 2.1, 2.1, 10.0, std::nullopt}
        };
        yt::saveToTable(*ytClient(), TRACK_TABLE_DAY_1, tracksDay1);

        std::vector<DeviceTrackRecord> tracksDay2 {
            {DEVICE_ID_1, usec("2021-12-13T00:00:00"), 1.0, 1.0, 10.0, 0.0},
            {DEVICE_ID_1, usec("2021-12-13T00:00:01"), 1.1, 1.1, 10.0, 0.0},
            {DEVICE_ID_1, usec("2021-12-13T00:00:02"), 1.2, 1.2, 10.0, 0.0}
        };
        yt::saveToTable(*ytClient(), TRACK_TABLE_DAY_2, tracksDay2);
    }
};

void allPointsHaveSourceId(const db::TrackPoints& track, const std::string& sourceId)
{
    for (const auto& point : track) {
        UNIT_ASSERT_EQUAL(point.sourceId(), sourceId);
    }
}

}  // namespace

Y_UNIT_TEST_SUITE_F(track_provider, TrackProviderFixture)
{

Y_UNIT_TEST(test_track_provider_with_empty_events)
{
    TrackProvider trackProvider(ytClient(), TRACK_TABLES_PATH, OUTPUT_TABLE);
    VideoEvents videoEvents{};
    UNIT_ASSERT_NO_EXCEPTION(trackProvider.preloadTracks(videoEvents));
}

Y_UNIT_TEST(test_track_provider)
{
    TrackProvider trackProvider(ytClient(), TRACK_TABLES_PATH, OUTPUT_TABLE);
    VideoEvents videoEvents {
        {0, EVENT_TYPE, "url-0.ru", DEVICE_ID_1, chrono::parseIsoDateTime("2021-12-12T12:00:01"), {}},
        {1, EVENT_TYPE, "url-1.ru", DEVICE_ID_1, chrono::parseIsoDateTime("2021-12-13T00:00:00"), {}},
        {2, EVENT_TYPE, "url-2.ru", DEVICE_ID_2, chrono::parseIsoDateTime("2021-12-12T11:59:30"), {}},
        // No track exists for this date
        {3, EVENT_TYPE, "url-3.ru", DEVICE_ID_2, chrono::parseIsoDateTime("2021-12-14T12:00:00"), {}},
        // No track exists for this deviceId
        {4, EVENT_TYPE, "url-4.ru", "unknown",   chrono::parseIsoDateTime("2021-12-12T12:00:00"), {}}
    };

    trackProvider.preloadTracks(videoEvents);
    {
        auto track = trackProvider.getTrack(videoEvents[0]);
        UNIT_ASSERT_EQUAL(track.size(), 3u);
        allPointsHaveSourceId(track, DEVICE_ID_1);
        UNIT_ASSERT_EQUAL(track[0].timestamp(), chrono::parseIsoDateTime("2021-12-12T12:00:00"));
        UNIT_ASSERT_EQUAL(track[1].timestamp(), chrono::parseIsoDateTime("2021-12-12T12:00:01"));
        UNIT_ASSERT_EQUAL(track[2].timestamp(), chrono::parseIsoDateTime("2021-12-12T12:00:02"));
    }

    {
        auto track = trackProvider.getTrack(videoEvents[1]);
        UNIT_ASSERT_EQUAL(track.size(), 5u);
        allPointsHaveSourceId(track, DEVICE_ID_1);
        UNIT_ASSERT_EQUAL(track[0].timestamp(), chrono::parseIsoDateTime("2021-12-12T23:59:58"));
        UNIT_ASSERT_EQUAL(track[1].timestamp(), chrono::parseIsoDateTime("2021-12-12T23:59:59"));
        UNIT_ASSERT_EQUAL(track[2].timestamp(), chrono::parseIsoDateTime("2021-12-13T00:00:00"));
        UNIT_ASSERT_EQUAL(track[3].timestamp(), chrono::parseIsoDateTime("2021-12-13T00:00:01"));
        UNIT_ASSERT_EQUAL(track[4].timestamp(), chrono::parseIsoDateTime("2021-12-13T00:00:02"));
    }

    {
        auto track = trackProvider.getTrack(videoEvents[2]);
        UNIT_ASSERT_EQUAL(track.size(), 2u);
        allPointsHaveSourceId(track, DEVICE_ID_2);
        UNIT_ASSERT_EQUAL(track[0].timestamp(), chrono::parseIsoDateTime("2021-12-12T12:00:00"));
        UNIT_ASSERT_EQUAL(track[1].timestamp(), chrono::parseIsoDateTime("2021-12-12T12:00:01"));
    }

    {
        auto track = trackProvider.getTrack(videoEvents[3]);
        UNIT_ASSERT(track.empty());
    }

    {
        auto track = trackProvider.getTrack(videoEvents[4]);
        UNIT_ASSERT(track.empty());
    }

    trackProvider.clearTracks();
    {
        auto track = trackProvider.getTrack(videoEvents[0]);
        UNIT_ASSERT(track.empty());
    }
}

} // Y_UNIT_TEST_SUITE_F

}  // namespace maps::mrc::import_taxi::tests
